"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ParsedSpec = void 0;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const path = require("path");
const log4js_1 = require("log4js");
const projen_1 = require("projen");
const util_1 = require("projen/lib/util");
const logger = log4js_1.getLogger();
/**
 * Component for parsing the yaml OpenAPI spec as a single json object, resolving references etc.
 */
class ParsedSpec extends projen_1.Component {
    constructor(project, options) {
        super(project);
        this.options = options;
    }
    static parse(specPath, outputPath) {
        // Parse the spec and write to the target output path
        util_1.exec(`./parse-openapi-spec --specPath=${specPath} --outputPath=${outputPath}`, {
            cwd: path.resolve(__dirname, "..", "..", "..", "..", "scripts", "parser"),
        });
    }
    synthesize() {
        super.synthesize();
        ParsedSpec.parse(this.options.specPath, this.options.outputPath);
        const singleSpecFile = util_1.tryReadFileSync(this.options.outputPath);
        if (!singleSpecFile) {
            throw new Error(`Failed to parse specification at ${this.options.specPath}`);
        }
        const parsedSpec = JSON.parse(singleSpecFile);
        // To avoid duplicating custom generated code (eg. OperationConfig or handler wrappers) and causing build errors, we
        // will apply the OpenAPI Normalizer to KEEP_ONLY_FIRST_TAG_IN_OPERATION when generating code. Tags are still
        // preserved in the specification to allow for better documentation.
        // See: https://github.com/OpenAPITools/openapi-generator/pull/14465
        const operationsWithMultipleTags = Object.entries(parsedSpec.paths).flatMap(([urlPath, methods]) => Object.entries(methods ?? {})
            .filter(([, operation]) => typeof operation === "object" &&
            "tags" in operation &&
            operation.tags &&
            operation.tags.length > 1)
            .map(([httpMethod]) => `${httpMethod} ${urlPath}`));
        if (operationsWithMultipleTags.length > 0) {
            logger.warn(`The following operations had multiple tags: ${operationsWithMultipleTags.join(", ")}. Code will only be generated for each operation's first tag.`);
        }
    }
}
exports.ParsedSpec = ParsedSpec;
//# sourceMappingURL=data:application/json;base64,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