# -*- coding: utf-8 -*-
from setuptools import setup

package_dir = \
{'': 'src'}

packages = \
['signal_processing_algorithms',
 'signal_processing_algorithms.e_divisive',
 'signal_processing_algorithms.e_divisive.calculators']

package_data = \
{'': ['*']}

install_requires = \
['misc-utils-py>=0.1.0,<0.2.0',
 'numpy>=1.18.1,<2.0.0',
 'scipy>=1.3,<2.0',
 'structlog>=19.1,<20.0',
 'typing-extensions>=3.7.4,<4.0.0']

setup_kwargs = {
    'name': 'signal-processing-algorithms',
    'version': '0.0.0rc1',
    'description': 'Signal Processing Algorithms from MongoDB',
    'long_description': '# Signal Processing Algorithms\n\nA suite of algorithms implementing [E-Divisive with Means](https://arxiv.org/pdf/1306.4933.pdf) and\n [Generalized ESD Test for Outliers](https://www.itl.nist.gov/div898/handbook/eda/section3/eda35h3.htm) in python.\n\n## Getting Started - Users\n```\npip install signal-processing-algorithms\n```\n\n## Getting Started - Developers\n\nGetting the code:\n\n```\n$ git clone git@github.com:mongodb/signal-processing-algorithms.git\n$ cd signal-processing-algorithms\n```\n\nMaking a virtual environment and installing the stuff you need into it:\n```\n$ virtualenv -p python3 venv\n$ source venv/bin/activate\n$ pip install -e .\n$ pip install -r requirements.txt\n```\nTesting stuff:\n```\n$ pytest\n```\n\nRunning the slow tests:\n```\n$ pytest --runslow\n```\n\n**Some of the larger tests can take a significant amount of time (more than 2 hours).**\n\nThe linters:\n```\n$ pytest src --flake8 --black --pydocstyle --mypy\n```\n\n## Intro to E-Divisive\n\nDetecting distributional changes in a series of numerical values can be surprisingly difficult. Simple systems based on thresholds or\n mean values can be yield false positives due to outliers in the data, and will fail to detect changes in the noise\n profile of the series you are analyzing.\n \nOne robust way of detecting many of the changes missed by other approaches is to use [E-Divisive with Means](https://arxiv.org/pdf/1306.4933.pdf), an energy\n statistic based approach that compares the expected distance (Euclidean norm) between samples of two portions of the\n series with the expected distance between samples within those portions.\n \nThat is to say, assuming that the two portions can each be modeled as i.i.d. samples drawn from distinct random variables\n (X for the first portion, Y for the second portion), you would expect the following to be non-zero if there is a\n sdifference between the two portions: \n \n <a href="https://www.codecogs.com/eqnedit.php?latex=\\varepsilon&space;(X,&space;Y;&space;\\alpha&space;)&space;=&space;2E|X-Y|^\\alpha&space;-&space;E|X-X\'|^\\alpha&space;-&space;E|Y-Y\'|^\\alpha" target="_blank"><img src="https://latex.codecogs.com/gif.latex?\\varepsilon&space;(X,&space;Y;&space;\\alpha&space;)&space;=&space;2E|X-Y|^\\alpha&space;-&space;E|X-X\'|^\\alpha&space;-&space;E|Y-Y\'|^\\alpha" title="\\varepsilon (X, Y; \\alpha ) = 2E|X-Y|^\\alpha - E|X-X\'|^\\alpha - E|Y-Y\'|^\\alpha" /></a>\n\nWhere alpha is some fixed constant in (0, 2).\nThis can be calculated empirically with samples from the portions corresponding to X, Y as follows:\n \n<a href="https://www.codecogs.com/eqnedit.php?latex=\\frac{2}{mn}&space;\\underset{i=1}{\\overset{n}{\\sum}}\\underset{j=1}{\\overset{m}{\\sum}}|X_{i}-Y_{j}|^\\alpha&space;-\\binom{n}{2}^{-1}\\underset{1\\leq&space;i<k\\leq&space;n}{\\sum}|X_{i}-X_{k}|^\\alpha&space;-&space;\\binom{m}{2}^{-1}\\underset{1&space;\\leq&space;j<k&space;\\leq&space;m}{\\sum}|Y_{j}-Y_{k}|^\\alpha" target="_blank"><img src="https://latex.codecogs.com/gif.latex?\\frac{2}{mn}&space;\\underset{i=1}{\\overset{n}{\\sum}}\\underset{j=1}{\\overset{m}{\\sum}}|X_{i}-Y_{j}|^\\alpha&space;-\\binom{n}{2}^{-1}\\underset{1\\leq&space;i<k\\leq&space;n}{\\sum}|X_{i}-X_{k}|^\\alpha&space;-&space;\\binom{m}{2}^{-1}\\underset{1&space;\\leq&space;j<k&space;\\leq&space;m}{\\sum}|Y_{j}-Y_{k}|^\\alpha" title="\\frac{2}{mn} \\underset{i=1}{\\overset{n}{\\sum}}\\underset{j=1}{\\overset{m}{\\sum}}|X_{i}-Y_{j}|^\\alpha -\\binom{n}{2}^{-1}\\underset{1\\leq i<k\\leq n}{\\sum}|X_{i}-X_{k}|^\\alpha - \\binom{m}{2}^{-1}\\underset{1 \\leq j<k \\leq m}{\\sum}|Y_{j}-Y_{k}|^\\alpha" /></a>\n \nThus for a series Z of length L, we find the most likely change point by solving the following for argmax(&tau;) (with a scaling factor of mn/(m+n) and &alpha;=1 for simplicity):\n\n<a href="https://www.codecogs.com/eqnedit.php?latex=Z&space;=&space;\\{Z_{1},&space;...,&space;Z_{\\tau}&space;,&space;...&space;,&space;Z_{L}\\},&space;X&space;=\\{Z_{1},...,Z_{\\tau}\\},&space;Y=\\{Z_{\\tau&plus;1}\\,...,Z_{L}\\}" target="_blank"><img src="https://latex.codecogs.com/gif.latex?Z&space;=&space;\\{Z_{1},&space;...,&space;Z_{\\tau}&space;,&space;...&space;,&space;Z_{L}\\},&space;X&space;=\\{Z_{1},...,Z_{\\tau}\\},&space;Y=\\{Z_{\\tau&plus;1}\\,...,Z_{L}\\}" title="Z = \\{Z_{1}, ..., Z_{\\tau} , ... , Z_{L}\\}, X =\\{Z_{1},...,Z_{\\tau}\\}, Y=\\{Z_{\\tau+1}\\,...,Z_{L}\\}" /></a>\n\n<a href="https://www.codecogs.com/eqnedit.php?latex=\\frac{2}{L}(&space;\\underset{i=1}{\\overset{\\tau}{\\sum}}\\underset{j=\\tau&plus;1}{\\overset{L}{\\sum}}|X_{i}-Y_{j}|&space;-\\frac{L-\\tau}{\\tau-1}\\underset{1\\leq&space;i<k\\leq&space;\\tau}{\\sum}|X_{i}-X_{k}|&space;-&space;\\frac{\\tau}{L-\\tau-1}\\underset{\\tau&space;<&space;j<k&space;\\leq&space;L}{\\sum}|Y_{j}-Y_{k}|)" target="_blank"><img src="https://latex.codecogs.com/gif.latex?\\frac{2}{L}(&space;\\underset{i=1}{\\overset{\\tau}{\\sum}}\\underset{j=\\tau&plus;1}{\\overset{L}{\\sum}}|X_{i}-Y_{j}|&space;-\\frac{L-\\tau}{\\tau-1}\\underset{1\\leq&space;i<k\\leq&space;\\tau}{\\sum}|X_{i}-X_{k}|&space;-&space;\\frac{\\tau}{L-\\tau-1}\\underset{\\tau&space;<&space;j<k&space;\\leq&space;L}{\\sum}|Y_{j}-Y_{k}|)" title="\\frac{2}{L}( \\underset{i=1}{\\overset{\\tau}{\\sum}}\\underset{j=\\tau+1}{\\overset{L}{\\sum}}|X_{i}-Y_{j}| -\\frac{L-\\tau}{\\tau-1}\\underset{1\\leq i<k\\leq \\tau}{\\sum}|X_{i}-X_{k}| - \\frac{\\tau}{L-\\tau-1}\\underset{\\tau < j<k \\leq L}{\\sum}|Y_{j}-Y_{k}|)" /></a>\n\n### Multiple Change Points\n\nThe algorithm for finding multiple change points is also simple.\n\nAssuming you have some k known change points:\n1. Partition the series into segments between/around these change points.\n2. Find the maximum value of our divergence metric _within_ each partition.\n3. Take the maximum of the maxima we have just found --> this is our k+1th change point.\n4. Return to step 1 and continue until reaching your stopping criterion.\n\n### Stopping Criterion\n\nIn this package we have implemented a permutation based test as a stopping criterion:\n\nAfter step 3 of the multiple change point procedure above, randomly permute all of the data _within_ each cluster, and\n find the most likely change point for this permuted data using the procedure laid out above. \n \nAfter performing this operation z times, count the number of\n permuted change points z\' that have higher divergence metrics than the change point you calculated with un-permuted data.\n The significance level of your change point is thus z\'/(z+1). \n\nWe allow users to configure a permutation tester with `pvalue`\n and `permutations` representing the significance cutoff for algorithm termination and permutations to perform for each\n test, respectively.\n \n### Example\n```\nfrom signal_processing_algorithms.e_divisive import EDivisive\nfrom signal_processing_algorithms.e_divisive.calculators import cext_calculator\nfrom signal_processing_algorithms.e_divisive.significance_test import QHatPermutationsSignificanceTester\nfrom some_module import series\n\n// Use C-Extension calculator for calculating divergence metrics\ncalculator = cext_calculator\n// Permutation tester with 1% significance threshold performing 100 permutations for each change point candidate\ntester = QHatPermutationsSignificanceTester(\n    calculator=calculator, pvalue=0.01, permutations=100\n)\nalgo = EDivisive(calculator=calculator, significance_tester=tester)\n\nchange_points = algo.get_change_points(series)\n```',
    'author': 'Alexander Costas',
    'author_email': 'alexander.costas@mongodb.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/mongodb/signal-processing-algorithms',
    'package_dir': package_dir,
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.7,<4.0',
}
from build import *
build(setup_kwargs)

setup(**setup_kwargs)
