# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['django_outbox_pattern',
 'django_outbox_pattern.management',
 'django_outbox_pattern.management.commands',
 'django_outbox_pattern.migrations']

package_data = \
{'': ['*']}

install_requires = \
['django>=3.2', 'stomp.py>=8.0.1,<9']

setup_kwargs = {
    'name': 'django-outbox-pattern',
    'version': '0.1.5',
    'description': 'A django application to make it easier to use the transactional outbox pattern',
    'long_description': '\n# Django outbox pattern\n[![Build Status](https://dev.azure.com/juntos-somos-mais-loyalty/python/_apis/build/status/juntossomosmais.django-outbox-pattern?branchName=main)](https://dev.azure.com/juntos-somos-mais-loyalty/python/_build/latest?definitionId=307&branchName=main)\n[![Maintainability Rating](https://sonarcloud.io/api/project_badges/measure?project=juntossomosmais_django-outbox-pattern&metric=sqale_rating)](https://sonarcloud.io/summary/new_code?id=juntossomosmais_django-outbox-pattern)\n[![Coverage](https://sonarcloud.io/api/project_badges/measure?project=juntossomosmais_django-outbox-pattern&metric=coverage)](https://sonarcloud.io/summary/new_code?id=juntossomosmais_django-outbox-pattern)\n[![Code style: black](https://img.shields.io/badge/code%20style-black-black)](https://github.com/ambv/black)\n[![Downloads](https://pepy.tech/badge/django-outbox-pattern)](https://pepy.tech/project/django-outbox-pattern)\n[![Downloads](https://pepy.tech/badge/django-outbox-pattern/month)](https://pepy.tech/project/django-outbox-pattern/month)\n[![Downloads](https://pepy.tech/badge/django-outbox-pattern/week)](https://pepy.tech/project/django-outbox-pattern/week)\n[![PyPI version](https://badge.fury.io/py/django-outbox-pattern.svg)](https://badge.fury.io/py/django-outbox-pattern)\n[![GitHub](https://img.shields.io/github/license/mashape/apistatus.svg)](https://github.com/juntossomosmais/django-outbox-pattern/blob/master/LICENSE)\n\nA django application to make it easier to use the [transactional outbox pattern](https://microservices.io/patterns/data/transactional-outbox.html)\n\n## Installation\n\nInstall django-outbox-pattern with pip\n\n```bash\npip install django-outbox-pattern\n```\n\nAdd to settings\n\n```python\n# settings.py\n\nINSTALLED_APPS = [\n    "django_outbox_pattern",\n]\n\nDJANGO_OUTBOX_PATTERN = {\n  "DEFAULT_STOMP_HOST_AND_PORTS": [("127.0.0.1", 61613)],\n  "DEFAULT_STOMP_USERNAME": "guest",\n  "DEFAULT_STOMP_PASSCODE": "guest",\n}\n\n```\n\n## Usage/Examples\n\nThe `publish` decorator adds the outbox table to the model. `\npublish` accepts three parameters, the `destination` which is required,\nfields which the default are all the `fields` of the model and `serializer` which by default adds the id in the message to be sent.\n`fields` and `serializer` are mutually exclusive.\n\n_**Only destination**_\n\n```python\nfrom django.db import models\nfrom django_outbox_pattern.decorators import publish\n\n\n@publish(destination=\'/topic/my_route_key\')\nclass MyModel(models.Model):\n    field_one = models.CharField(max_length=100)\n    field_two = models.CharField(max_length=100)\n```\n\nThis generates the following data to be sent.\n\n```text\nproducer.send(destination=\'/topic/my_route_key.v1\', body=\'{"id": 1, "field_one": "Field One", "field_two": "Field Two"}\')\n```\n\n_**With fields**_\n\n```python\nfrom django.db import models\nfrom django_outbox_pattern.decorators import publish\n\n\n@publish(destination=\'/topic/my_route_key\', fields=["field_one"])\nclass MyModel(models.Model):\n    field_one = models.CharField(max_length=100)\n    field_two = models.CharField(max_length=100)\n```\n\nThis generates the following data to be sent.\n\n```text\nproducer.send(destination=\'/topic/my_route_key.v1\', body=\'{"id": 1, "field_one": "Field One"}\')\n```\n\n_**With serializer**_\n\n```python\nfrom django.db import models\nfrom django_outbox_pattern.decorators import publish\n\n\n@publish(destination=\'/topic/my_route_key\', serializer=\'my_serializer\')\nclass MyModel(models.Model):\n    field_one = models.CharField(max_length=100)\n    field_two = models.CharField(max_length=100)\n\n    def my_serializer(self):\n        return {\n            "id": self.id,\n            "one": self.field_one,\n            "two": self.field_two\n        }\n```\n\nThis generates the following data to be sent.\n\n```text\nproducer.send(destination=\'/topic/my_route_key.v1\', body=\'{"id": 1, "one": "Field One", "two": "Field Two"}\')\n```\n## Publish/Subscribe commands\n\n##### Publish command\n\nTo send the messages added to the outbox table it is necessary to start the producer.\n\n```shell\npython manage.py publish\n```\n\n##### Subscribe command\n\nDjango outbox pattern also provides a consumer that can be used to receive outgoing messages.\n\nCreate a function that receives an instance of `django_outbox_pattern.payloads.Payload`\n\n```python\n# callbacks.py\n\ndef callback(payload):\n    try:\n        # Do anything\n        payload.ack()\n    except Exception:\n        # Nack is automatically called in case of errors, but you might want to handle the error in another way\n        payload.nack()\n```\n\n```shell\npython manage.py subscribe \'dotted.path.to.callback` \'destination\'\n```\n\n## Settings\n\n**DEFAULT_CONNECTION_CLASS**\n\nThe stomp.py class responsible for connecting to the broker. Default: `stomp.StompConnection12`\n\n**DEFAULT_CONSUMER_LISTENER_CLASS**\n\nThe consumer listener class. Default: `django_outbox_pattern.listeners.ConsumerListener`\n\n**DEFAULT_GENERATE_HEADERS**\n\nA function to add headers to the message. Default: `django_outbox_pattern.headers.generate_headers`\n\n**DEFAULT_MAXIMUM_BACKOFF**:\n\nMaximum wait time for connection attempts in seconds. Default: `3600` (1 hour)\n\n**DEFAULT_MAXIMUM_RETRY_ATTEMPTS**\n\nMaximum number of message resend attempts. Default: `50`\n\n**DEFAULT_PAUSE_FOR_RETRY**\n\nPausing for attempts to resend messages in seconds. Defualt: `240` (4 minutes)\n\n**DEFAULT_WAIT_RETRY**\n\nTime between attempts to send messages after the pause. Default: `60` (1 minute)\n\n**DEFAULT_PRODUCER_LISTENER_CLASS**:\n\nThe producer listener class. Default: `django_outbox_pattern.listeners.ProducerListener`\n\n**DEFAULT_STOMP_HOST_AND_PORTS**\n\nList of host and port tuples to try to connect to the broker. Default `[("127.0.0.1", 61613)]`\n\n**DEFAULT_STOMP_QUEUE_HEADERS**\n\nHeaders for queues. Default: `{"durable": "true", "auto-delete": "false", "prefetch-count": "1"}`\n\n**DEFAULT_STOMP_HEARTBEATS**\n\nTime tuples for input and output heartbeats. Default:  `(10000, 10000)`\n\n**DEFAULT_STOMP_VHOST**\n\nVirtual host. Default: "/"\n\n**DEFAULT_STOMP_USERNAME**\n\nUsername for connection. Default: `"guest"`\n\n**DEFAULT_STOMP_PASSCODE**\n\nPassword for connection. Default: `"guest"`\n',
    'author': 'Hugo Brilhante',
    'author_email': 'hugobrilhante@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/juntossomosmais/django-outbox-pattern',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
