"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const javascript_1 = require("projen/lib/javascript");
const python_1 = require("projen/lib/python");
const typescript_1 = require("projen/lib/typescript");
const synth_1 = require("projen/lib/util/synth");
const src_1 = require("../src");
describe("NX Monorepo Unit Tests", () => {
    it("Empty Monorepo", () => {
        const project = new src_1.NxMonorepoProject({
            defaultReleaseBranch: "mainline",
            name: "Defaults",
        });
        expect(synth_1.synthSnapshot(project)).toMatchSnapshot();
    });
    it("Ignore Patterns", () => {
        const project = new src_1.NxMonorepoProject({
            defaultReleaseBranch: "mainline",
            name: "IgnorePatterns",
            nxConfig: {
                nxIgnore: ["pattern1.txt", "**/foo.ts"],
            },
        });
        expect(synth_1.synthSnapshot(project)).toMatchSnapshot();
    });
    it("Target Dependencies", () => {
        const project = new src_1.NxMonorepoProject({
            defaultReleaseBranch: "mainline",
            name: "TargetDependencies",
            nxConfig: {
                targetDependencies: {
                    test: [
                        {
                            target: "test",
                            projects: src_1.Nx.TargetDependencyProject.DEPENDENCIES,
                        },
                    ],
                    eslint: [
                        { target: "eslint", projects: src_1.Nx.TargetDependencyProject.SELF },
                    ],
                },
            },
        });
        expect(synth_1.synthSnapshot(project)).toMatchSnapshot();
    });
    it("Affected Branch", () => {
        const project = new src_1.NxMonorepoProject({
            defaultReleaseBranch: "main",
            name: "AffectedBranch",
            nxConfig: {
                affectedBranch: "main",
            },
        });
        expect(synth_1.synthSnapshot(project)).toMatchSnapshot();
    });
    it.each([javascript_1.NodePackageManager.PNPM, javascript_1.NodePackageManager.YARN])("Additional Workspace Packages", (packageManager) => {
        const project = new src_1.NxMonorepoProject({
            defaultReleaseBranch: "mainline",
            packageManager,
            name: "AdditionalWorkspacePackages",
            workspaceConfig: {
                additionalPackages: ["my/custom/package"],
            },
        });
        new typescript_1.TypeScriptProject({
            name: "ts-subproject",
            outdir: "packages/ts-subproject",
            parent: project,
            packageManager,
            defaultReleaseBranch: "mainline",
        });
        project.addWorkspacePackages("another/custom/package", "yet/another/package");
        expect(synth_1.synthSnapshot(project)).toMatchSnapshot();
    });
    it("Workspace Package Order", () => {
        const project = new src_1.NxMonorepoProject({
            defaultReleaseBranch: "mainline",
            name: "WorkspacePackageOrder",
            workspaceConfig: {
                additionalPackages: ["packages/one"],
            },
        });
        new typescript_1.TypeScriptProject({
            name: "two",
            outdir: "packages/two",
            parent: project,
            defaultReleaseBranch: "mainline",
        });
        project.addWorkspacePackages("packages/three", "packages/four");
        new typescript_1.TypeScriptProject({
            name: "five",
            outdir: "packages/five",
            parent: project,
            defaultReleaseBranch: "mainline",
        });
        project.addWorkspacePackages("packages/six");
        new typescript_1.TypeScriptProject({
            name: "seven",
            outdir: "packages/seven",
            parent: project,
            defaultReleaseBranch: "mainline",
        });
        expect(synth_1.synthSnapshot(project)).toMatchSnapshot();
    });
    it("PNPM", () => {
        const project = new src_1.NxMonorepoProject({
            defaultReleaseBranch: "mainline",
            name: "PNPM",
            packageManager: javascript_1.NodePackageManager.PNPM,
        });
        new typescript_1.TypeScriptProject({
            name: "ts-subproject",
            outdir: "packages/ts-subproject",
            parent: project,
            packageManager: javascript_1.NodePackageManager.PNPM,
            defaultReleaseBranch: "mainline",
        });
        expect(synth_1.synthSnapshot(project)).toMatchSnapshot();
    });
    it("Validate consistent Package Managers", () => {
        const project = new src_1.NxMonorepoProject({
            defaultReleaseBranch: "mainline",
            name: "PNPM",
            packageManager: javascript_1.NodePackageManager.PNPM,
        });
        new typescript_1.TypeScriptProject({
            name: "ts-subproject",
            outdir: "packages/ts-subproject",
            parent: project,
            defaultReleaseBranch: "mainline",
        });
        expect(() => synth_1.synthSnapshot(project)).toThrow("ts-subproject packageManager does not match the monorepo packageManager: pnpm.");
    });
    it("Composite", () => {
        const project = new src_1.NxMonorepoProject({
            defaultReleaseBranch: "mainline",
            name: "Composite",
        });
        const tsProject = new typescript_1.TypeScriptProject({
            name: "ts-subproject",
            outdir: "packages/ts-subproject",
            parent: project,
            defaultReleaseBranch: "mainline",
        });
        new typescript_1.TypeScriptProject({
            name: "ts-subproject2",
            outdir: "packages/ts-subproject2",
            parent: project,
            defaultReleaseBranch: "mainline",
            devDeps: ["ts-subproject"],
        });
        const pyProject = new python_1.PythonProject({
            parent: project,
            outdir: "packages/py-subproject",
            authorEmail: "test@test.com",
            authorName: "test",
            moduleName: "py_subproject",
            name: "py-subproject",
            version: "0.0.0",
        });
        project.addImplicitDependency(tsProject, pyProject);
        expect(synth_1.synthSnapshot(project)).toMatchSnapshot();
    });
});
//# sourceMappingURL=data:application/json;base64,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