"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const path = require("path");
const pdk_nag_1 = require("../../pdk-nag");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const assertions_1 = require("aws-cdk-lib/assertions");
const aws_s3_1 = require("aws-cdk-lib/aws-s3");
const aws_s3_assets_1 = require("aws-cdk-lib/aws-s3-assets");
const src_1 = require("../src");
describe("PDK Pipeline Unit Tests", () => {
    it("Defaults", () => {
        const app = pdk_nag_1.PDKNag.app();
        const stack = new aws_cdk_lib_1.Stack(app);
        const pipeline = new src_1.PDKPipeline(stack, "Defaults", {
            primarySynthDirectory: "cdk.out",
            repositoryName: "Defaults",
            synth: {},
            crossAccountKeys: false,
            sonarCodeScannerConfig: {
                sonarqubeAuthorizedGroup: "dev",
                sonarqubeDefaultProfileOrGateName: "dev",
                sonarqubeEndpoint: "https://sonar.dev",
                sonarqubeProjectName: "Default",
            },
        });
        const stage = new aws_cdk_lib_1.Stage(app, "Stage");
        const appStack = new aws_cdk_lib_1.Stack(stage, "AppStack");
        new aws_s3_assets_1.Asset(appStack, "Asset", {
            path: path.join(__dirname, "pdk-pipeline.test.ts"),
        });
        pipeline.addStage(stage);
        pipeline.buildPipeline();
        app.synth();
        expect(app.nagResults().length).toEqual(0);
        expect(assertions_1.Template.fromStack(stack)).toMatchSnapshot();
    });
    it("Defaults - using AwsPrototyping NagPack", () => {
        const app = pdk_nag_1.PDKNag.app({ nagPacks: [new pdk_nag_1.AwsPrototypingChecks()] });
        const stack = new aws_cdk_lib_1.Stack(app);
        const pipeline = new src_1.PDKPipeline(stack, "Defaults", {
            primarySynthDirectory: "cdk.out",
            repositoryName: "Defaults",
            synth: {},
            crossAccountKeys: false,
            sonarCodeScannerConfig: {
                sonarqubeAuthorizedGroup: "dev",
                sonarqubeDefaultProfileOrGateName: "dev",
                sonarqubeEndpoint: "https://sonar.dev",
                sonarqubeProjectName: "Default",
            },
        });
        const stage = new aws_cdk_lib_1.Stage(app, "Stage");
        const appStack = new aws_cdk_lib_1.Stack(stage, "AppStack");
        new aws_s3_assets_1.Asset(appStack, "Asset", {
            path: path.join(__dirname, "pdk-pipeline.test.ts"),
        });
        pipeline.addStage(stage);
        pipeline.buildPipeline();
        app.synth();
        expect(app.nagResults().length).toEqual(0);
        expect(assertions_1.Template.fromStack(stack)).toMatchSnapshot();
    });
    it("CrossAccount", () => {
        const app = pdk_nag_1.PDKNag.app();
        const stack = new aws_cdk_lib_1.Stack(app);
        const pipeline = new src_1.PDKPipeline(stack, "CrossAccount", {
            primarySynthDirectory: "cdk.out",
            repositoryName: "Defaults",
            synth: {},
            crossAccountKeys: true,
            sonarCodeScannerConfig: {
                sonarqubeAuthorizedGroup: "dev",
                sonarqubeDefaultProfileOrGateName: "dev",
                sonarqubeEndpoint: "https://sonar.dev",
                sonarqubeProjectName: "Default",
            },
        });
        const stage = new aws_cdk_lib_1.Stage(app, "Stage");
        const appStack = new aws_cdk_lib_1.Stack(stage, "AppStack");
        new aws_s3_assets_1.Asset(appStack, "Asset", {
            path: path.join(__dirname, "pdk-pipeline.test.ts"),
        });
        pipeline.addStage(stage);
        pipeline.buildPipeline();
        app.synth();
        expect(app.nagResults().length).toEqual(0);
        expect(assertions_1.Template.fromStack(stack)).toMatchSnapshot();
    });
    it("CrossAccount - using AwsPrototyping NagPack", () => {
        const app = pdk_nag_1.PDKNag.app({ nagPacks: [new pdk_nag_1.AwsPrototypingChecks()] });
        const stack = new aws_cdk_lib_1.Stack(app);
        const pipeline = new src_1.PDKPipeline(stack, "CrossAccount", {
            primarySynthDirectory: "cdk.out",
            repositoryName: "Defaults",
            synth: {},
            crossAccountKeys: true,
            sonarCodeScannerConfig: {
                sonarqubeAuthorizedGroup: "dev",
                sonarqubeDefaultProfileOrGateName: "dev",
                sonarqubeEndpoint: "https://sonar.dev",
                sonarqubeProjectName: "Default",
            },
        });
        const stage = new aws_cdk_lib_1.Stage(app, "Stage");
        const appStack = new aws_cdk_lib_1.Stack(stage, "AppStack");
        new aws_s3_assets_1.Asset(appStack, "Asset", {
            path: path.join(__dirname, "pdk-pipeline.test.ts"),
        });
        pipeline.addStage(stage);
        pipeline.buildPipeline();
        app.synth();
        expect(app.nagResults().length).toEqual(0);
        expect(assertions_1.Template.fromStack(stack)).toMatchSnapshot();
    });
    it("StageNagRuns", () => {
        const app = pdk_nag_1.PDKNag.app({ failOnError: false });
        const stack = new aws_cdk_lib_1.Stack(app);
        const pipeline = new src_1.PDKPipeline(stack, "StageNagRuns", {
            primarySynthDirectory: "cdk.out",
            repositoryName: "StageNagRuns",
            synth: {},
            sonarCodeScannerConfig: {
                sonarqubeAuthorizedGroup: "dev",
                sonarqubeDefaultProfileOrGateName: "dev",
                sonarqubeEndpoint: "https://sonar.dev",
                sonarqubeProjectName: "Default",
            },
        });
        const stage = new aws_cdk_lib_1.Stage(app, "Stage");
        const appStack = new aws_cdk_lib_1.Stack(stage, "AppStack");
        new aws_s3_1.Bucket(appStack, "Non-Compliant");
        pipeline.addStage(stage);
        pipeline.buildPipeline();
        app.synth();
        expect(app.nagResults()[0].resource).toEqual("Stage/AppStack/Non-Compliant/Resource");
    });
    it("StageNagRuns - using AwsPrototyping NagPack", () => {
        const app = pdk_nag_1.PDKNag.app({
            failOnError: false,
            nagPacks: [new pdk_nag_1.AwsPrototypingChecks()],
        });
        const stack = new aws_cdk_lib_1.Stack(app);
        const pipeline = new src_1.PDKPipeline(stack, "StageNagRuns", {
            primarySynthDirectory: "cdk.out",
            repositoryName: "StageNagRuns",
            synth: {},
            sonarCodeScannerConfig: {
                sonarqubeAuthorizedGroup: "dev",
                sonarqubeDefaultProfileOrGateName: "dev",
                sonarqubeEndpoint: "https://sonar.dev",
                sonarqubeProjectName: "Default",
            },
        });
        const stage = new aws_cdk_lib_1.Stage(app, "Stage");
        const appStack = new aws_cdk_lib_1.Stack(stage, "AppStack");
        new aws_s3_1.Bucket(appStack, "Non-Compliant");
        pipeline.addStage(stage);
        pipeline.buildPipeline();
        app.synth();
        expect(app.nagResults()[0].resource).toEqual("Stage/AppStack/Non-Compliant/Resource");
    });
});
//# sourceMappingURL=data:application/json;base64,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