from sigma.processing.transformations import FieldMappingTransformation, AddFieldnamePrefixTransformation, AddConditionTransformation, ChangeLogsourceTransformation
from sigma.processing.conditions import LogsourceCondition, IncludeFieldCondition, ExcludeFieldCondition, RuleProcessingItemAppliedCondition
from sigma.processing.pipeline import ProcessingItem, ProcessingPipeline

ecs_zeek_beats_category_service_mapping = {
    "accounting": "syslog",
    "firewall": "conn",
    "dns": "dns",
    "proxy": "http",
    "webserver": "http",
}

def ecs_zeek_beats():
    return ProcessingPipeline(
        name="Elastic Common Schema (ECS) for Zeek using filebeat >= 7.6.1",
        priority=20,
        items=[
            ProcessingItem(
                identifier=f"zeek_mapping_category_{ category }_to_service_{ service }",
                transformation=ChangeLogsourceTransformation(product="zeek", service=service),
                rule_conditions=[
                    LogsourceCondition(category=category),
                ],
            )
            for category, service in ecs_zeek_beats_category_service_mapping.items()
        ] + [
            ProcessingItem(
                identifier="zeek_ecs_event_dataset",
                transformation=AddConditionTransformation(
                    conditions={ "event.dataset": "zeek.$service" },
                    template=True,
                ),
                rule_conditions=[
                    LogsourceCondition(product="zeek"),
                ],
            ),
            ProcessingItem(     # Field mappings
                identifier="zeek_ecs_field_mapping",
                transformation=FieldMappingTransformation({
                    "dst": [ "destination.address", "destination.ip", ],
                    "dst_ip": [ "destination.address", "destination.ip", ],
                    "dst_port": "destination.port",
                    "network_application": "network.protocol",
                    "network_community_id": "network.community_id",
                    "network_protocol": "network.transport",
                    "proto": "network.transport",
                    "src": [ "source.address", "source.ip", ],
                    "src_ip": "source.ip",
                    "src_port": "source.port",
                    "action": "zeek.smb_files.action",
                    "mqtt_action": "smb.action",
                    "smb_action": "smb.action",
                    "tunnel_action": "tunnel.action",
                    "addl": "zeek.weird.additional_info",
                    "dns_addl": "zeek.dns.addl",
                    "weird_addl": "zeek.weird.additional_info",
                    "arg": "zeek.*.arg",
                    "ftp_arg": "zeek.ftp.arg",
                    "mysql_arg": "zeek.mysql.arg",
                    "pop3_arg": "zeek.pop3.arg",
                    "auth": "zeek.*.auth*",
                    "cipher": "zeek.*.cipher",
                    "kerberos_cipher": "zeek.kerberos.cipher",
                    "ssl_cipher": "zeek.ssl.cipher",
                    "tls_cipher": "zeek.ssl.cipher",
                    "client": "zeek.*.client*",
                    "command": "zeek.*.command",
                    "ftp_command": "zeek.irc.command",
                    "irc_command": "zeek.ftp.command",
                    "pop3_command": "zeek.pop3.command",
                    "date": "zeek.*.date",
                    "duration": "event.duration",
                    "from": "zeek.*.from",
                    "kerberos_from": "zeek.smtp.from",
                    "smtp_from": "zeek.kerberos.valid.from",
                    "is_orig": "zeek.*.is_orig",
                    "local_orig": "zeek.*.local_orig",
                    "method": "http.request.method",
                    "http_method": "http.request.method",
                    "sip_method": "zeek.sip.sequence.method",
                    "name": "zeek.smb_files.name",
                    "smb_files_name": "zeek.smb_files.name",
                    "software_name": "zeek.software.name",
                    "weird_name": "zeek.weird.name",
                    "path": "zeek.*.path",
                    "smb_mapping_path": "zeek.smb_mapping.path",
                    "smb_files_path": "zeek.smb_files.path",
                    "smtp_files_path": "zeek.smtp.path",
                    "password": "zeek.*.password",
                    "reply_msg": "zeek.*.reply*msg",
                    "reply_to": "zeek.*.reply_to",
                    "response_body_len": "http.response.body.bytes",
                    "request_body_len": "http.request.body.bytes",
                    "rtt": "event.duration",
                    "status_code": "http.response.status_code",
                    "status_msg": "zeek.*status*msg",
                    "service": "zeek.*.service",
                    "kerberos_service": "zeek.kerberos.service",
                    "smb_mapping_kerberos": "zeek.smb_mapping.service",
                    "subject": [ "zeek.*.subject", ],
                    "known_certs_subject": "zeek.known_certs.subject",
                    "sip_subject": "zeek.sip.subject",
                    "smtp_subject": "zeek.smtp.subject",
                    "ssl_subject": "zeek.ssl.subject  ",
                    "trans_depth": "zeek.*.trans*depth",
                    "username": "zeek.*.username",
                    "uri": "url.original",
                    "user": "zeek.*user*",
                    "user_agent": [ "zeek.*user_agent*", ],
                    "http_user_agent": "user_agent.original",
                    "gquic_user_agent": "zeek.gquic.user_agent",
                    "sip_user_agent": "zeek.sip.user_agent",
                    "smtp_user_agent": "zeek.smtp.user_agent",
                    "version": "zeek.*.version",
                    "gquic_version": "zeek.gquic.version",
                    "http_version": "http.version",
                    "ntp_version": "zeek.ntp.version",
                    "socks_version": "zeek.socks.version",
                    "snmp_version": "zeek.snmp.version",
                    "ssh_version": "zeek.ssh.version",
                    "tls_version": "zeek.ssl.version",
                    "answer": "dns.answers.name",
                    "question_length": "labels.dns.query_length",
                    "record_type": "dns.question.type",
                    "parent_domain": "dns.question.registered_domain",
                    "cs-bytes": "http.request.body.bytes",
                    "cs-cookie": "http.cookie_vars",
                    "r-dns": [ "url.domain", "destination.domain", ],
                    "sc-bytes": "http.response.body.bytes",
                    "sc-status": "http.response.status_code",
                    "c-uri": "url.original",
                    "c-uri-extension": "url.extension",
                    "c-uri-query": "url.query",
                    "c-uri-stem": "url.original",
                    "c-useragent": "user_agent.original",
                    "cs-host": [ "url.domain", "destination.domain", ],
                    "cs-method": "http.request.method",
                    "cs-referer": "http.request.referrer",
                    "cs-version": "http.version",
                    "uid": "zeek.session_id",
                    "cache_add_rx_ev": "zeek.connection.cache_add_rx_ev",
                    "cache_add_rx_mpg": "zeek.connection.cache_add_rx_mpg",
                    "cache_add_rx_new": "zeek.connection.cache_add_rx_new",
                    "cache_add_tx_ev": "zeek.connection.cache_add_tx_ev",
                    "cache_add_tx_mpg": "zeek.connection.cache_add_tx_mpg",
                    "cache_del_mpg": "zeek.connection.cache_del_mpg",
                    "cache_entries": "zeek.connection.cache_entries",
                    "conn_state": "zeek.connection.conn_state",
                    "conn_uids": "zeek.files.session_ids",
                    "corelight_shunted": "zeek.connection.corelight_shunted",
                    "history": "zeek.connection.history",
                    "id.orig_h.name_src": "zeek.connection.id.orig_h_name.src",
                    "id.orig_h.names_vals": "zeek.connection.id.orig_h_names.vals",
                    "id.resp_h.name_src": "zeek.connection.id.resp_h_name.src",
                    "id.resp_h.name_vals": "zeek.connection.id.resp_h_name.vals",
                    "local_resp": "zeek.connection.local_resp",
                    "missed_bytes": "zeek.connection.missed_bytes",
                    "orig_bytes": "source.bytes",
                    "orig_cc": "source.geo.country_iso_code",
                    "orig_ip_bytes": "zeek.connection.orig_ip_bytes",
                    "orig_l2_addr": "source.mac",
                    "orig_pkts": "source.packets",
                    "resp_bytes": "destination.bytes",
                    "resp_cc": "destination.geo.country_iso_code",
                    "resp_ip_bytes": "zeek.connection.resp.ip_bytes",
                    "resp_l2_addr": "destination.mac",
                    "resp_pkts": "destination.packets",
                    "endpoint": "zeek.dce_rpc.endpoint",
                    "named_pipe": "zeek.dce_rpc.named_pipe",
                    "operation": "zeek.dce_rpc.operation",
                    "domain": "zeek.dhcp.domain",
                    "host_name": "zeek.dhcp.hostname",
                    "lease_time": "zeek.dhcp.lease_time",
                    "agent_remote_id": "zeek.dhcp.agent_remote_id",
                    "assigned_addr": "zeek.dhcp.assigned_addr",
                    "circuit_id": "zeek.dhcp.circuit_id",
                    "client_message": "zeek.dhcp.client_message",
                    "client_software": "zeek.dhcp.client_software",
                    "client_fqdn": "zeek.dhcp.client_fqdn",
                    "msg_orig": "zeek.dhcp.msg_orig",
                    "msg_types": "zeek.dhcp.msg_types",
                    "requested_addr": "zeek.dhcp.requested_addr",
                    "server_addr": "destination.ip",
                    "server_message": "zeek.dhcp.server_message",
                    "server_software": "zeek.dhcp.server_software",
                    "subscriber_id": "zeek.dhcp.subscriber_id",
                    "AA": "zeek.dns.AA",
                    "answers": "dns.answers.name",
                    "TTLs": "dns.answers.ttl",
                    "RA": "zeek.dns.RA",
                    "RD": "zeek.dns.RD",
                    "rejected": "zeek.dns.rejected",
                    "TC": "zeek.dns.TC",
                    "Z": "zeek.dns.Z",
                    "qclass": "zeek.dns.qclass",
                    "qclass_name": "dns.question.class",
                    "qtype": "zeek.dns.qtype",
                    "qtype_name": "dns.question.type",
                    "query": "dns.question.name",
                    "rcode_name": "dns.response_code",
                    "rcode": "zeek.dns.rcode",
                    "trans_id": "dns.id",
                    "fc_reply": "dnp3.function.reply",
                    "fc_request": "dnp3.function.request",
                    "iin": "dnp3.inn",
                    "failure_reason": "dpd.failure_reason",
                    "packet_segment": "dpd.packet_segment",
                    "rx_hosts": "zeek.files.rx_host",
                    "tx_hosts": "zeek.files.tx_host",
                    "depth": "zeek.files.depth",
                    "extracted": "zeek.files.extracted",
                    "extracted_cutoff": "zeek.files.extracted_cutoff",
                    "extracted_size": "zeek.files.extracted_size",
                    "entropy": "zeek.files.entropy",
                    "md5": "zeek.files.md5",
                    "sha1": "zeek.files.sha1",
                    "sha256": "zeek.files.sha256",
                    "missing_bytes": "zeek.files.missing_bytes",
                    "filename": "zeek.files.filename",
                    "overflow_bytes": "zeek.files.overflow_bytes",
                    "seen_bytes": "zeek.files.seen_bytes",
                    "total_bytes": "zeek.files.total_bytes",
                    "timedout": "zeek.files.timedout",
                    "cyu": "gquic.cyu",
                    "cyutags": "gquic.cyutags",
                    "tag_count": "gquic.tag_count",
                    "cwd": "zeek.ftp.cwd",
                    "data_channel.passive": "zeek.ftp.data_channel.passive",
                    "data_channel.orig_h": "zeek.ftp.data_channel.originating_host",
                    "data_channel.resp_h": "zeek.ftp.data_channel.response_host",
                    "data_channel.resp_p": "zeek.ftp.data_channel.response_port",
                    "file_size": "zeek.ftp.file.size",
                    "passive": "zeek.ftp.passive",
                    "reply_code": "zeek.ftp.reply.code",
                    "client_header_names": "zeek.http.client_header_names",
                    "cookie_vars": "zeek.http.cookie_vars",
                    "flash_version": "zeek.http.flash_version",
                    "info_code": "zeek.http.info_code",
                    "info_msg": "zeek.http.info_msg",
                    "omniture": "zeek.http.omniture",
                    "orig_filenames": "zeek.http.orig_filenames",
                    "orig_mime_types": "zeek.http.orig_mime_types",
                    "origin": "zeek.http.origin",
                    "referrer": "http.request.referrer",
                    "post_body": "zeek.http.post_body",
                    "proxied": "zeek.http.proxied",
                    "resp_filenames": "zeek.http.resp_filenames",
                    "resp_mime_types": "zeek.http.resp_mime_types",
                    "server_header_names": "zeek.http.server_header_names",
                    "uri_vars": "zeek.http.uri_vars",
                    "file_mime_type": "zeek.intel.mime_type",
                    "file_desc": "zeek.intel.file_desc",
                    "host": "zeek.intel.seen.host",
                    "matched": "zeek.intel.matched",
                    "indicator": "zeek.intel.seen.indicator",
                    "indicator_type": "zeek.intel.seen.indicator_type",
                    "node": "zeek.intel.seen.node",
                    "where": "zeek.intel.seen.where",
                    "sources": "zeek.intel.seen.sources",
                    "dcc_file_name": "zeek.irc.dcc.file.name",
                    "dcc_file_size": "zeek.irc.dcc.file.size",
                    "dcc_mime_type": "zeek.irc.dcc.mime_type",
                    "nick": "zeek.irc.nick",
                    "value": "zeek.irc.command",
                    "auth_ticket": "zeek.kerberos.ticket.auth",
                    "client_cert_subject": "zeek.kerberos.cert.client.subject",
                    "error_code": "zeek.kerberos.error.code",
                    "error_msg": "zeek.kerberos.error.msg",
                    "forwardable": "zeek.kerberos.forwardable",
                    "new_ticket": "zeek.kerberos.ticket.new",
                    "renewable": "zeek.kerberos.renewable",
                    "request_type": "zeek.kerberos.request_type",
                    "success": "zeek.kerberos.success",
                    "server_cert_subject": "zeek.kerberos.cert.server.subject",
                    "till": "zeek.kerberos.valid.until",
                    "delta": "zeek.modbus.delta",
                    "new_val": "zeek.modbus.new_val",
                    "old_val": "zeek.modbus.old_val",
                    "register": "zeek.modbus.register",
                    "func": "zeek.modbus.function",
                    "exception": "zeek.modbus.exception",
                    "track_address": "zeek.modbus.track_address",
                    "cmd": "zeek.mysql.cmd",
                    "response": "zeek.mysql.response",
                    "rows": "zeek.mysql.rows",
                    "actions": "zeek.notice.actions",
                    "dropped": "zeek.notice.dropped",
                    "email_body_sections": "zeek.notice.email_body_sections",
                    "email_delay_tokens": "zeek.notice.email_delay_tokens",
                    "identifier": "zeek.notice.identifier",
                    "msg": "zeek.notice.msg",
                    "n": "zeek.notice.n",
                    "note": "zeek.notice.note",
                    "p": "destination.port",
                    "peer_descr": "zeek.notice.peer_descr",
                    "peer_name": "zeek.notice.peer_name",
                    "sub": "zeek.notice.sub",
                    "subpress_for": "zeek.notice.subpress_for",
                    "domainname": "zeek.ntlm.domain",
                    "hostname": "zeek.ntlm.hostname",
                    "server_nb_computer_name": "zeek.ntlm.server.name.netbios",
                    "server_tree_name": "zeek.ntlm.server.name.tree",
                    "server_dns_computer_name": "zeek.ntlm.server.name.dns",
                    "certStatus": "zeek.ocsp.status",
                    "hashAlgorithm": "zeek.ocsp.hash.algorithm",
                    "issuerKeyHash": "zeek.ocsp.hash.issuer.key",
                    "issuerNameHash": "zeek.ocsp.hash.issuer.name",
                    "nextUpdate": "zeek.ocsp.update.next",
                    "revokereason": "zeek.ocsp.revoke.reason",
                    "revoketime": "zeek.ocsp.revoke.date",
                    "serialNumber": "zeek.ocsp.serial_number",
                    "thisUpdate": "zeek.ocsp.update.this",
                    "compile_ts": "zeek.pe.compile_time",
                    "has_cert_table": "zeek.pe.has_cert_table",
                    "has_debug_data": "zeek.pe.has_debug_data",
                    "has_import_table": "zeek.pe.has_import_table",
                    "has_export_table": "zeek.pe.has_export_table",
                    "is_64bit": "zeek.pe.is_64bit",
                    "is_exe": "zeek.pe.is_exe",
                    "machine": "zeek.pe.machine",
                    "os": "zeek.pe.os",
                    "section_names": "zeek.pe.section_names",
                    "subsystem": "zeek.pe.subsystem",
                    "uses_aslr": "zeek.pe.uses_aslr",
                    "uses_code_integrity": "zeek.pe.uses_code_integrity",
                    "uses_dep": "zeek.pe.uses_dep",
                    "uses_seh": "zeek.pe.uses_seh",
                    "connect_info": "zeek.radius.connect_info",
                    "framed_addr": "zeek.radius.framed_addr",
                    "mac": "zeek.radius.mac",
                    "result": "zeek.radius.result",
                    "ttl": "zeek.radius.ttl",
                    "tunnel_client": "zeek.radius.tunnel_client",
                    "cert_count": "zeek.rdp.cert.count",
                    "cert_permanent": "zeek.rdp.cert.permanent",
                    "cert_type": "zeek.rdp.cert.type",
                    "client_build": "zeek.rdp.client.build",
                    "client_dig_product_id": "zeek.rdp.client.product_id",
                    "client_name": "zeek.rdp.client.name",
                    "cookie": "zeek.rdp.cookie",
                    "desktop_height": "zeek.rdp.desktop.height",
                    "desktop_width": "zeek.rdp.desktop.width",
                    "encryption_level": "zeek.rdp.encryption.level",
                    "encryption_method": "zeek.rdp.encryption.method",
                    "keyboard_layout": "zeek.rdp.keyboard_layout",
                    "requested_color_depth": "zeek.rdp.desktop.color_depth",
                    "security_protocol": "zeek.rdp.security_protocol",
                    "ssl": "zeek.rdp.ssl",
                    "authentication_method": "zeek.rfb.auth.method",
                    "client_major_version": "zeek.rfb.version.client.major",
                    "client_minor_version": "zeek.rfb.version.client.minor",
                    "desktop_name": "zeek.rfb.desktop_name",
                    "height": "zeek.rfb.height",
                    "server_major_version": "zeek.rfb.version.server.major",
                    "server_minor_version": "zeek.rfb.version.server.minor",
                    "share_flag": "zeek.rfb.share_flag",
                    "width": "zeek.rfb.width",
                    "call_id": "zeek.sip.call_id",
                    "content_type": "zeek.sip.content_type",
                    "request_from": "zeek.sip.request.from",
                    "request_path": "zeek.sip.request.path",
                    "request_to": "zeek.sip.request.to",
                    "response_from": "zeek.sip.response.from",
                    "response_path": "zeek.sip.response.path",
                    "response_to": "zeek.sip.response.to",
                    "seq": "zeek.sip.seq",
                    "warning": "zeek.sip.warning",
                    "prev_name": "zeek.smb_files.prev_name",
                    "size": "zeek.smb_files.size",
                    "times_accessed": "zeek.smb_files.accessed",
                    "times_changed": "zeek.smb_files.ctime",
                    "times_created": "zeek.smb_files.created",
                    "times_modified": "zeek.smb_files.mtime",
                    "native_file_system": "zeek.smb_mapping.native_file_system",
                    "share_type": "zeek.smb_mapping.share_type",
                    "cc": "zeek.smtp.cc",
                    "first_received": "zeek.smtp.first_received",
                    "helo": "zeek.smtp.helo",
                    "in_reply_to": "zeek.smtp.in_reply_to",
                    "is_webmail": "zeek.smtp.is_webmail",
                    "last_reply": "zeek.smtp.last_reply",
                    "mailfrom": "zeek.smtp.mail_from",
                    "msg_id": "zeek.smtp.msg_id",
                    "rcptto": "zeek.smtp.rcpt_to",
                    "second_received": "zeek.smtp.second_received",
                    "tls": "zeek.smtp.tls",
                    "to": "zeek.smtp.to",
                    "x_originating_ip": "zeek.smtp.x_originating_ip",
                    "community": "zeek.snmp.community",
                    "display_string": "zeek.snmp.display_string",
                    "get_bulk_requests": "zeek.snmp.get.bulk_requests",
                    "get_requests": "zeek.snmp.get.requests",
                    "set_requests": "zeek.snmp.set.requests",
                    "up_since": "zeek.snmp.up_since",
                    "bound_host": "zeek.socks.bound_host",
                    "bound_name": "zeek.socks.bound_name",
                    "bound_p": "zeek.socks.bound.port",
                    "request_host": "zeek.socks.request_host",
                    "request_name": "zeek.socks.request.host",
                    "request_p": "zeek.socks.request.port",
                    "status": "zeek.socks.status",
                    "auth_attempts": "zeek.ssh.auth.attempts",
                    "auth_success": "zeek.ssh.auth.success",
                    "cipher_alg": "zeek.ssh.algorithm.cipher",
                    "compression_alg": "zeek.ssh.algorithm.compression",
                    "cshka": "zeek.ssh.cshka",
                    "direction": "zeek.ssh.direction",
                    "hassh": "zeek.ssh.hassh",
                    "hasshAlgorithms": "zeek.ssh.hasshAlgorithms",
                    "hasshServer": "zeek.ssh.hasshServer",
                    "hasshServerAlgorithms": "zeek.ssh.hasshServerAlgorithms",
                    "hasshVersion": "zeek.ssh.hasshVersion",
                    "host_key": "zeek.ssh.host_key",
                    "host_key_alg": "zeek.ssh.algorithm.host_key",
                    "kex_alg": "zeek.ssh.algorithm.key_exchange",
                    "mac_alg": "zeek.ssh.algorithm.mac",
                    "server": "zeek.ssh.server",
                    "client_issuer": "zeek.ssl.client_issuer",
                    "client_subject": "zeek.ssl.client_subject",
                    "curve": "zeek.ssl.curve",
                    "established": "zeek.ssl.established",
                    "issuer": "zeek.ssl.issuer",
                    "ja3": "zeek.ssl.ja3",
                    "ja3s": "zeek.ssl.ja3s",
                    "last_alert": "zeek.ssl.last_alert",
                    "next_protocol": "zeek.ssl.next_protocol",
                    "notary": "zeek.ssl.notary",
                    "ocsp_status": "zeek.ssl.oscp_status",
                    "orig_certificate_sha1": "zeek.ssl.orig_certificate_sha1",
                    "resp_certificate_sha1": "zeek.ssl.resp_certificate_sha1",
                    "resumed": "zeek.ssl.resumed",
                    "server_name": "zeek.ssl.server.name",
                    "valid_ct_logs": "zeek.ssl.valid_ct_logs",
                    "valid_ct_operators": "zeek.ssl.validct_operators",
                    "valid_ct_operators_list": "zeek.ssl.valid_ct_operators_list",
                    "validation_status": "zeek.ssl.validation_status",
                    "version_num": "zeek.ssl.version_num",
                    "facility": "zeek.syslog.facility",
                    "severity": "zeek.syslog.severity",
                    "message": "zeek.syslog.msg # why did Elastic beats do this",
                    "tunnel_type": "zeek.tunnel.type",
                    "notice": "zeek.weird.notice",
                    "peer": "zeek.weird.peer",
                    "basic_constraints.ca": "zeek.x509.certificate.basic_constraints.certificate_authority",
                    "basic_constraints.path_len": "zeek.x509.certificate.basic_constraints.path_length",
                    "certificate.cn": "zeek.x509.certificate.common_name",
                    "certificate.curve": "zeek.x509.certificate.curve",
                    "certificate.exponent": "zeek.x509.certificate.exponent",
                    "certificate.issuer": "zeek.x509.certificate.iss # why did Elastic beats do this",
                    "certificate.key_alg": "zeek.x509.certificate.key.algorithm",
                    "certificate.key_length": "zeek.x509.certificate.key.length",
                    "certificate.key_type": "zeek.x509.certificate.key.type",
                    "certificate.not_valid_after": "zeek.x509.certificate.valid.until",
                    "certificate.not_valid_before": "zeek.x509.certificate.valid.from",
                    "certificate.serial": "zeek.x509.certificate.serial",
                    "certificate.sig_alg": "zeek.x509.certificate.signature_algorithm",
                    "certificate.subject": "zeek.x509.certificate.sub  # why did Elastic beats do this",
                    "certificate.version": "zeek.x509.certificate.version",
                    "logcert": "zeek.x509.logcert",
                    "san.dns": "zeek.x509.san.dns",
                    "san.email": "zeek.x509.san.email",
                    "san.ip": "zeek.x509.san.ip",
                    "san.uri": "zeek.x509.san.url",
                    "agent.version": "version",
                    "c-cookie": "http.cookie_vars",
                    "c-ip": "source.ip",
                    "cs-uri": "url.original",
                    "clientip": "source.ip",
                    "clientIP": "source.io",
                    "dest_domain": [ "query", "host", "server_name", ],
                    "dest_ip": "destination.ip",
                    "dest_port": "destination.port",
                }),
                rule_conditions=[
                    LogsourceCondition(product="zeek"),
                ],
            ),
        ],
    )

def ecs_zeek_corelight():
    return ProcessingPipeline(
        name="Elastic Common Schema (ECS) mapping from Corelight",
        priority=20,
        items=[
            ProcessingItem(
                identifier=f"zeek_mapping_category_{ category }_to_service_{ service }",
                transformation=ChangeLogsourceTransformation(product="zeek", service=service),
                rule_conditions=[
                    LogsourceCondition(category=category),
                ],
            )
            for category, service in ecs_zeek_beats_category_service_mapping.items()
        ] + [
            ProcessingItem(
                identifier="zeek_ecs_event_dataset",
                transformation=AddConditionTransformation(
                    conditions={ "event.dataset": "zeek.$service" },
                    template=True,
                ),
                rule_conditions=[
                    LogsourceCondition(product="zeek"),
                ],
            ),
            ProcessingItem(     # Field mappings
                identifier="zeek_ecs_field_mapping",
                transformation=FieldMappingTransformation({
                    "dst": [ "destination.address", "destination.ip", ],
                    "dst_ip": [ "destination.address", "destination.ip", ],
                    "dst_port": "destination.port",
                    "network_application": "network.protocol",
                    "network_community_id": "network.community_id",
                    "network_protocol": "network.transport",
                    "proto": "network.transport",
                    "src": [ "source.address", "source.ip", ],
                    "src_ip": "source.ip",
                    "src_port": "source.port",
                    "action": "zeek.smb_files.action",
                    "mqtt_action": "smb.action",
                    "smb_action": "smb.action",
                    "tunnel_action": "tunnel.action",
                    "addl": "zeek.weird.additional_info",
                    "dns_addl": "zeek.dns.addl",
                    "weird_addl": "zeek.weird.additional_info",
                    "arg": "zeek.*.arg",
                    "ftp_arg": "zeek.ftp.arg",
                    "mysql_arg": "zeek.mysql.arg",
                    "pop3_arg": "zeek.pop3.arg",
                    "auth": "zeek.*.auth*",
                    "cipher": "zeek.*.cipher",
                    "kerberos_cipher": "zeek.kerberos.cipher",
                    "ssl_cipher": "zeek.ssl.cipher",
                    "tls_cipher": "zeek.ssl.cipher",
                    "client": "zeek.*.client*",
                    "command": "zeek.*.command",
                    "ftp_command": "zeek.irc.command",
                    "irc_command": "zeek.ftp.command",
                    "pop3_command": "zeek.pop3.command",
                    "date": "zeek.*.date",
                    "duration": "event.duration",
                    "from": "zeek.*.from",
                    "kerberos_from": "zeek.smtp.from",
                    "smtp_from": "zeek.kerberos.valid.from",
                    "is_orig": "zeek.*.is_orig",
                    "local_orig": "zeek.*.local_orig",
                    "method": "http.request.method",
                    "http_method": "http.request.method",
                    "sip_method": "zeek.sip.sequence.method",
                    "name": "zeek.smb_files.name",
                    "smb_files_name": "zeek.smb_files.name",
                    "software_name": "zeek.software.name",
                    "weird_name": "zeek.weird.name",
                    "path": "zeek.*.path",
                    "smb_mapping_path": "zeek.smb_mapping.path",
                    "smb_files_path": "zeek.smb_files.path",
                    "smtp_files_path": "zeek.smtp.path",
                    "password": "zeek.*.password",
                    "reply_msg": "zeek.*.reply*msg",
                    "reply_to": "zeek.*.reply_to",
                    "response_body_len": "http.response.body.bytes",
                    "request_body_len": "http.request.body.bytes",
                    "rtt": "event.duration",
                    "status_code": "http.response.status_code",
                    "status_msg": "zeek.*status*msg",
                    "service": "zeek.*.service",
                    "kerberos_service": "zeek.kerberos.service",
                    "smb_mapping_kerberos": "zeek.smb_mapping.service",
                    "subject": [ "zeek.*.subject", ],
                    "known_certs_subject": "zeek.known_certs.subject",
                    "sip_subject": "zeek.sip.subject",
                    "smtp_subject": "zeek.smtp.subject",
                    "ssl_subject": "zeek.ssl.subject  ",
                    "trans_depth": "zeek.*.trans*depth",
                    "username": "zeek.*.username",
                    "uri": "url.original",
                    "user": "zeek.*user*",
                    "user_agent": [ "zeek.*user_agent*", ],
                    "http_user_agent": "user_agent.original",
                    "gquic_user_agent": "zeek.gquic.user_agent",
                    "sip_user_agent": "zeek.sip.user_agent",
                    "smtp_user_agent": "zeek.smtp.user_agent",
                    "version": "zeek.*.version",
                    "gquic_version": "zeek.gquic.version",
                    "http_version": "http.version",
                    "ntp_version": "zeek.ntp.version",
                    "socks_version": "zeek.socks.version",
                    "snmp_version": "zeek.snmp.version",
                    "ssh_version": "zeek.ssh.version",
                    "tls_version": "zeek.ssl.version",
                    "answer": "dns.answers.name",
                    "question_length": "labels.dns.query_length",
                    "record_type": "dns.question.type",
                    "parent_domain": "dns.question.registered_domain",
                    "cs-bytes": "http.request.body.bytes",
                    "cs-cookie": "http.cookie_vars",
                    "r-dns": [ "url.domain", "destination.domain", ],
                    "sc-bytes": "http.response.body.bytes",
                    "sc-status": "http.response.status_code",
                    "c-uri": "url.original",
                    "c-uri-extension": "url.extension",
                    "c-uri-query": "url.query",
                    "c-uri-stem": "url.original",
                    "c-useragent": "user_agent.original",
                    "cs-host": [ "url.domain", "destination.domain", ],
                    "cs-method": "http.request.method",
                    "cs-referer": "http.request.referrer",
                    "cs-version": "http.version",
                    "uid": "zeek.session_id",
                    "cache_add_rx_ev": "zeek.connection.cache_add_rx_ev",
                    "cache_add_rx_mpg": "zeek.connection.cache_add_rx_mpg",
                    "cache_add_rx_new": "zeek.connection.cache_add_rx_new",
                    "cache_add_tx_ev": "zeek.connection.cache_add_tx_ev",
                    "cache_add_tx_mpg": "zeek.connection.cache_add_tx_mpg",
                    "cache_del_mpg": "zeek.connection.cache_del_mpg",
                    "cache_entries": "zeek.connection.cache_entries",
                    "conn_state": "zeek.connection.conn_state",
                    "conn_uids": "zeek.files.session_ids",
                    "corelight_shunted": "zeek.connection.corelight_shunted",
                    "history": "zeek.connection.history",
                    "id.orig_h.name_src": "zeek.connection.id.orig_h_name.src",
                    "id.orig_h.names_vals": "zeek.connection.id.orig_h_names.vals",
                    "id.resp_h.name_src": "zeek.connection.id.resp_h_name.src",
                    "id.resp_h.name_vals": "zeek.connection.id.resp_h_name.vals",
                    "local_resp": "zeek.connection.local_resp",
                    "missed_bytes": "zeek.connection.missed_bytes",
                    "orig_bytes": "source.bytes",
                    "orig_cc": "source.geo.country_iso_code",
                    "orig_ip_bytes": "zeek.connection.orig_ip_bytes",
                    "orig_l2_addr": "source.mac",
                    "orig_pkts": "source.packets",
                    "resp_bytes": "destination.bytes",
                    "resp_cc": "destination.geo.country_iso_code",
                    "resp_ip_bytes": "zeek.connection.resp.ip_bytes",
                    "resp_l2_addr": "destination.mac",
                    "resp_pkts": "destination.packets",
                    "endpoint": "zeek.dce_rpc.endpoint",
                    "named_pipe": "zeek.dce_rpc.named_pipe",
                    "operation": "zeek.dce_rpc.operation",
                    "domain": "zeek.dhcp.domain",
                    "host_name": "zeek.dhcp.hostname",
                    "lease_time": "zeek.dhcp.lease_time",
                    "agent_remote_id": "zeek.dhcp.agent_remote_id",
                    "assigned_addr": "zeek.dhcp.assigned_addr",
                    "circuit_id": "zeek.dhcp.circuit_id",
                    "client_message": "zeek.dhcp.client_message",
                    "client_software": "zeek.dhcp.client_software",
                    "client_fqdn": "zeek.dhcp.client_fqdn",
                    "msg_orig": "zeek.dhcp.msg_orig",
                    "msg_types": "zeek.dhcp.msg_types",
                    "requested_addr": "zeek.dhcp.requested_addr",
                    "server_addr": "destination.ip",
                    "server_message": "zeek.dhcp.server_message",
                    "server_software": "zeek.dhcp.server_software",
                    "subscriber_id": "zeek.dhcp.subscriber_id",
                    "AA": "zeek.dns.AA",
                    "answers": "dns.answers.name",
                    "TTLs": "dns.answers.ttl",
                    "RA": "zeek.dns.RA",
                    "RD": "zeek.dns.RD",
                    "rejected": "zeek.dns.rejected",
                    "TC": "zeek.dns.TC",
                    "Z": "zeek.dns.Z",
                    "qclass": "zeek.dns.qclass",
                    "qclass_name": "dns.question.class",
                    "qtype": "zeek.dns.qtype",
                    "qtype_name": "dns.question.type",
                    "query": "dns.question.name",
                    "rcode_name": "dns.response_code",
                    "rcode": "zeek.dns.rcode",
                    "trans_id": "dns.id",
                    "fc_reply": "dnp3.function.reply",
                    "fc_request": "dnp3.function.request",
                    "iin": "dnp3.inn",
                    "failure_reason": "dpd.failure_reason",
                    "packet_segment": "dpd.packet_segment",
                    "rx_hosts": "zeek.files.rx_host",
                    "tx_hosts": "zeek.files.tx_host",
                    "depth": "zeek.files.depth",
                    "extracted": "zeek.files.extracted",
                    "extracted_cutoff": "zeek.files.extracted_cutoff",
                    "extracted_size": "zeek.files.extracted_size",
                    "entropy": "zeek.files.entropy",
                    "md5": "zeek.files.md5",
                    "sha1": "zeek.files.sha1",
                    "sha256": "zeek.files.sha256",
                    "missing_bytes": "zeek.files.missing_bytes",
                    "filename": "zeek.files.filename",
                    "overflow_bytes": "zeek.files.overflow_bytes",
                    "seen_bytes": "zeek.files.seen_bytes",
                    "total_bytes": "zeek.files.total_bytes",
                    "timedout": "zeek.files.timedout",
                    "cyu": "gquic.cyu",
                    "cyutags": "gquic.cyutags",
                    "tag_count": "gquic.tag_count",
                    "cwd": "zeek.ftp.cwd",
                    "data_channel.passive": "zeek.ftp.data_channel.passive",
                    "data_channel.orig_h": "zeek.ftp.data_channel.originating_host",
                    "data_channel.resp_h": "zeek.ftp.data_channel.response_host",
                    "data_channel.resp_p": "zeek.ftp.data_channel.response_port",
                    "file_size": "zeek.ftp.file.size",
                    "passive": "zeek.ftp.passive",
                    "reply_code": "zeek.ftp.reply.code",
                    "client_header_names": "zeek.http.client_header_names",
                    "cookie_vars": "zeek.http.cookie_vars",
                    "flash_version": "zeek.http.flash_version",
                    "info_code": "zeek.http.info_code",
                    "info_msg": "zeek.http.info_msg",
                    "omniture": "zeek.http.omniture",
                    "orig_filenames": "zeek.http.orig_filenames",
                    "orig_mime_types": "zeek.http.orig_mime_types",
                    "origin": "zeek.http.origin",
                    "referrer": "http.request.referrer",
                    "post_body": "zeek.http.post_body",
                    "proxied": "zeek.http.proxied",
                    "resp_filenames": "zeek.http.resp_filenames",
                    "resp_mime_types": "zeek.http.resp_mime_types",
                    "server_header_names": "zeek.http.server_header_names",
                    "uri_vars": "zeek.http.uri_vars",
                    "file_mime_type": "zeek.intel.mime_type",
                    "file_desc": "zeek.intel.file_desc",
                    "host": "zeek.intel.seen.host",
                    "matched": "zeek.intel.matched",
                    "indicator": "zeek.intel.seen.indicator",
                    "indicator_type": "zeek.intel.seen.indicator_type",
                    "node": "zeek.intel.seen.node",
                    "where": "zeek.intel.seen.where",
                    "sources": "zeek.intel.seen.sources",
                    "dcc_file_name": "zeek.irc.dcc.file.name",
                    "dcc_file_size": "zeek.irc.dcc.file.size",
                    "dcc_mime_type": "zeek.irc.dcc.mime_type",
                    "nick": "zeek.irc.nick",
                    "value": "zeek.irc.command",
                    "auth_ticket": "zeek.kerberos.ticket.auth",
                    "client_cert_subject": "zeek.kerberos.cert.client.subject",
                    "error_code": "zeek.kerberos.error.code",
                    "error_msg": "zeek.kerberos.error.msg",
                    "forwardable": "zeek.kerberos.forwardable",
                    "new_ticket": "zeek.kerberos.ticket.new",
                    "renewable": "zeek.kerberos.renewable",
                    "request_type": "zeek.kerberos.request_type",
                    "success": "zeek.kerberos.success",
                    "server_cert_subject": "zeek.kerberos.cert.server.subject",
                    "till": "zeek.kerberos.valid.until",
                    "delta": "zeek.modbus.delta",
                    "new_val": "zeek.modbus.new_val",
                    "old_val": "zeek.modbus.old_val",
                    "register": "zeek.modbus.register",
                    "func": "zeek.modbus.function",
                    "exception": "zeek.modbus.exception",
                    "track_address": "zeek.modbus.track_address",
                    "cmd": "zeek.mysql.cmd",
                    "response": "zeek.mysql.response",
                    "rows": "zeek.mysql.rows",
                    "actions": "zeek.notice.actions",
                    "dropped": "zeek.notice.dropped",
                    "email_body_sections": "zeek.notice.email_body_sections",
                    "email_delay_tokens": "zeek.notice.email_delay_tokens",
                    "identifier": "zeek.notice.identifier",
                    "msg": "zeek.notice.msg",
                    "n": "zeek.notice.n",
                    "note": "zeek.notice.note",
                    "p": "destination.port",
                    "peer_descr": "zeek.notice.peer_descr",
                    "peer_name": "zeek.notice.peer_name",
                    "sub": "zeek.notice.sub",
                    "subpress_for": "zeek.notice.subpress_for",
                    "domainname": "zeek.ntlm.domain",
                    "hostname": "zeek.ntlm.hostname",
                    "server_nb_computer_name": "zeek.ntlm.server.name.netbios",
                    "server_tree_name": "zeek.ntlm.server.name.tree",
                    "server_dns_computer_name": "zeek.ntlm.server.name.dns",
                    "certStatus": "zeek.ocsp.status",
                    "hashAlgorithm": "zeek.ocsp.hash.algorithm",
                    "issuerKeyHash": "zeek.ocsp.hash.issuer.key",
                    "issuerNameHash": "zeek.ocsp.hash.issuer.name",
                    "nextUpdate": "zeek.ocsp.update.next",
                    "revokereason": "zeek.ocsp.revoke.reason",
                    "revoketime": "zeek.ocsp.revoke.date",
                    "serialNumber": "zeek.ocsp.serial_number",
                    "thisUpdate": "zeek.ocsp.update.this",
                    "compile_ts": "zeek.pe.compile_time",
                    "has_cert_table": "zeek.pe.has_cert_table",
                    "has_debug_data": "zeek.pe.has_debug_data",
                    "has_import_table": "zeek.pe.has_import_table",
                    "has_export_table": "zeek.pe.has_export_table",
                    "is_64bit": "zeek.pe.is_64bit",
                    "is_exe": "zeek.pe.is_exe",
                    "machine": "zeek.pe.machine",
                    "os": "zeek.pe.os",
                    "section_names": "zeek.pe.section_names",
                    "subsystem": "zeek.pe.subsystem",
                    "uses_aslr": "zeek.pe.uses_aslr",
                    "uses_code_integrity": "zeek.pe.uses_code_integrity",
                    "uses_dep": "zeek.pe.uses_dep",
                    "uses_seh": "zeek.pe.uses_seh",
                    "connect_info": "zeek.radius.connect_info",
                    "framed_addr": "zeek.radius.framed_addr",
                    "mac": "zeek.radius.mac",
                    "result": "zeek.radius.result",
                    "ttl": "zeek.radius.ttl",
                    "tunnel_client": "zeek.radius.tunnel_client",
                    "cert_count": "zeek.rdp.cert.count",
                    "cert_permanent": "zeek.rdp.cert.permanent",
                    "cert_type": "zeek.rdp.cert.type",
                    "client_build": "zeek.rdp.client.build",
                    "client_dig_product_id": "zeek.rdp.client.product_id",
                    "client_name": "zeek.rdp.client.name",
                    "cookie": "zeek.rdp.cookie",
                    "desktop_height": "zeek.rdp.desktop.height",
                    "desktop_width": "zeek.rdp.desktop.width",
                    "encryption_level": "zeek.rdp.encryption.level",
                    "encryption_method": "zeek.rdp.encryption.method",
                    "keyboard_layout": "zeek.rdp.keyboard_layout",
                    "requested_color_depth": "zeek.rdp.desktop.color_depth",
                    "security_protocol": "zeek.rdp.security_protocol",
                    "ssl": "zeek.rdp.ssl",
                    "authentication_method": "zeek.rfb.auth.method",
                    "client_major_version": "zeek.rfb.version.client.major",
                    "client_minor_version": "zeek.rfb.version.client.minor",
                    "desktop_name": "zeek.rfb.desktop_name",
                    "height": "zeek.rfb.height",
                    "server_major_version": "zeek.rfb.version.server.major",
                    "server_minor_version": "zeek.rfb.version.server.minor",
                    "share_flag": "zeek.rfb.share_flag",
                    "width": "zeek.rfb.width",
                    "call_id": "zeek.sip.call_id",
                    "content_type": "zeek.sip.content_type",
                    "request_from": "zeek.sip.request.from",
                    "request_path": "zeek.sip.request.path",
                    "request_to": "zeek.sip.request.to",
                    "response_from": "zeek.sip.response.from",
                    "response_path": "zeek.sip.response.path",
                    "response_to": "zeek.sip.response.to",
                    "seq": "zeek.sip.seq",
                    "warning": "zeek.sip.warning",
                    "prev_name": "zeek.smb_files.prev_name",
                    "size": "zeek.smb_files.size",
                    "times_accessed": "zeek.smb_files.accessed",
                    "times_changed": "zeek.smb_files.ctime",
                    "times_created": "zeek.smb_files.created",
                    "times_modified": "zeek.smb_files.mtime",
                    "native_file_system": "zeek.smb_mapping.native_file_system",
                    "share_type": "zeek.smb_mapping.share_type",
                    "cc": "zeek.smtp.cc",
                    "first_received": "zeek.smtp.first_received",
                    "helo": "zeek.smtp.helo",
                    "in_reply_to": "zeek.smtp.in_reply_to",
                    "is_webmail": "zeek.smtp.is_webmail",
                    "last_reply": "zeek.smtp.last_reply",
                    "mailfrom": "zeek.smtp.mail_from",
                    "msg_id": "zeek.smtp.msg_id",
                    "rcptto": "zeek.smtp.rcpt_to",
                    "second_received": "zeek.smtp.second_received",
                    "tls": "zeek.smtp.tls",
                    "to": "zeek.smtp.to",
                    "x_originating_ip": "zeek.smtp.x_originating_ip",
                    "community": "zeek.snmp.community",
                    "display_string": "zeek.snmp.display_string",
                    "get_bulk_requests": "zeek.snmp.get.bulk_requests",
                    "get_requests": "zeek.snmp.get.requests",
                    "set_requests": "zeek.snmp.set.requests",
                    "up_since": "zeek.snmp.up_since",
                    "bound_host": "zeek.socks.bound_host",
                    "bound_name": "zeek.socks.bound_name",
                    "bound_p": "zeek.socks.bound.port",
                    "request_host": "zeek.socks.request_host",
                    "request_name": "zeek.socks.request.host",
                    "request_p": "zeek.socks.request.port",
                    "status": "zeek.socks.status",
                    "auth_attempts": "zeek.ssh.auth.attempts",
                    "auth_success": "zeek.ssh.auth.success",
                    "cipher_alg": "zeek.ssh.algorithm.cipher",
                    "compression_alg": "zeek.ssh.algorithm.compression",
                    "cshka": "zeek.ssh.cshka",
                    "direction": "zeek.ssh.direction",
                    "hassh": "zeek.ssh.hassh",
                    "hasshAlgorithms": "zeek.ssh.hasshAlgorithms",
                    "hasshServer": "zeek.ssh.hasshServer",
                    "hasshServerAlgorithms": "zeek.ssh.hasshServerAlgorithms",
                    "hasshVersion": "zeek.ssh.hasshVersion",
                    "host_key": "zeek.ssh.host_key",
                    "host_key_alg": "zeek.ssh.algorithm.host_key",
                    "kex_alg": "zeek.ssh.algorithm.key_exchange",
                    "mac_alg": "zeek.ssh.algorithm.mac",
                    "server": "zeek.ssh.server",
                    "client_issuer": "zeek.ssl.client_issuer",
                    "client_subject": "zeek.ssl.client_subject",
                    "curve": "zeek.ssl.curve",
                    "established": "zeek.ssl.established",
                    "issuer": "zeek.ssl.issuer",
                    "ja3": "zeek.ssl.ja3",
                    "ja3s": "zeek.ssl.ja3s",
                    "last_alert": "zeek.ssl.last_alert",
                    "next_protocol": "zeek.ssl.next_protocol",
                    "notary": "zeek.ssl.notary",
                    "ocsp_status": "zeek.ssl.oscp_status",
                    "orig_certificate_sha1": "zeek.ssl.orig_certificate_sha1",
                    "resp_certificate_sha1": "zeek.ssl.resp_certificate_sha1",
                    "resumed": "zeek.ssl.resumed",
                    "server_name": "zeek.ssl.server.name",
                    "valid_ct_logs": "zeek.ssl.valid_ct_logs",
                    "valid_ct_operators": "zeek.ssl.validct_operators",
                    "valid_ct_operators_list": "zeek.ssl.valid_ct_operators_list",
                    "validation_status": "zeek.ssl.validation_status",
                    "version_num": "zeek.ssl.version_num",
                    "facility": "zeek.syslog.facility",
                    "severity": "zeek.syslog.severity",
                    "message": "zeek.syslog.msg # why did Elastic beats do this",
                    "tunnel_type": "zeek.tunnel.type",
                    "notice": "zeek.weird.notice",
                    "peer": "zeek.weird.peer",
                    "basic_constraints.ca": "zeek.x509.certificate.basic_constraints.certificate_authority",
                    "basic_constraints.path_len": "zeek.x509.certificate.basic_constraints.path_length",
                    "certificate.cn": "zeek.x509.certificate.common_name",
                    "certificate.curve": "zeek.x509.certificate.curve",
                    "certificate.exponent": "zeek.x509.certificate.exponent",
                    "certificate.issuer": "zeek.x509.certificate.iss # why did Elastic beats do this",
                    "certificate.key_alg": "zeek.x509.certificate.key.algorithm",
                    "certificate.key_length": "zeek.x509.certificate.key.length",
                    "certificate.key_type": "zeek.x509.certificate.key.type",
                    "certificate.not_valid_after": "zeek.x509.certificate.valid.until",
                    "certificate.not_valid_before": "zeek.x509.certificate.valid.from",
                    "certificate.serial": "zeek.x509.certificate.serial",
                    "certificate.sig_alg": "zeek.x509.certificate.signature_algorithm",
                    "certificate.subject": "zeek.x509.certificate.sub  # why did Elastic beats do this",
                    "certificate.version": "zeek.x509.certificate.version",
                    "logcert": "zeek.x509.logcert",
                    "san.dns": "zeek.x509.san.dns",
                    "san.email": "zeek.x509.san.email",
                    "san.ip": "zeek.x509.san.ip",
                    "san.uri": "zeek.x509.san.url",
                    "agent.version": "version",
                    "c-cookie": "http.cookie_vars",
                    "c-ip": "source.ip",
                    "cs-uri": "url.original",
                    "clientip": "source.ip",
                    "clientIP": "source.io",
                    "dest_domain": [ "query", "host", "server_name", ],
                    "dest_ip": "destination.ip",
                    "dest_port": "destination.port",
                }),
                rule_conditions=[
                LogsourceCondition(product="zeek"),
            ],
        ),
    ],
)

def zeek_raw():
    return ProcessingPipeline(
        name="Zeek raw JSON field naming",
        priority=20,
        items=[
            ProcessingItem(
                identifier=f"zeek_mapping_category_{ category }_to_service_{ service }",
                transformation=ChangeLogsourceTransformation(product="zeek", service=service),
                rule_conditions=[
                    LogsourceCondition(category=category),
                ],
            )
            for category, service in ecs_zeek_beats_category_service_mapping.items()
        ] + [
            ProcessingItem(
                identifier="zeek_event_dataset",
                transformation=AddConditionTransformation(
                    conditions={ "@stream": "$service" },
                    template=True,
                ),
                rule_conditions=[
                    LogsourceCondition(product="zeek"),
                ],
            ),
            ProcessingItem(     # Field mappings
                identifier="zeek_field_mapping",
                transformation=FieldMappingTransformation({
                    "dst_ip": "id.resp_h",
                    "dst_port": "id.resp_p",
                    "network_protocol": "proto",
                    "src_ip": "id.orig_h",
                    "src_port": "id.orig_p",
                    "answer": "answers",
                    "record_type": "qtype_name",
                    "cs-bytes": "request_body_len",
                    "cs-cookie": "cookie",
                    "r-dns": "host",
                    "sc-bytes": "response_body_len",
                    "sc-status": "status_code",
                    "c-uri": "uri",
                    "c-uri-extension": "uri",
                    "c-uri-query": "uri",
                    "c-uri-stem": "uri",
                    "c-useragent": "user_agent",
                    "cs-host": "host",
                    "cs-method": "method",
                    "cs-referer": "referrer",
                    "cs-version": "version",
                    "id_orig_h": "id.orig_h",
                    "id_orig_p": "id.orig_p",
                    "id_resp_h": "id.resp_h",
                    "id_resp_p": "id.resp_p",
                    "agent.version": "version",
                    "c-cookie": "cookie",
                    "c-ip": "id.orig_h",
                    "cs-uri": "uri",
                    "clientip": "id.orig_h",
                    "clientIP": "id.orig_h",
                    "dest_domain": [ "query", "host", "server_name", ],
                    "dest_ip": "id.resp_h",
                    "dest_port": "id.resp_p",
                    "destination.hostname": [ "query", "host", "server_name", ],
                    "DestinationAddress": "id.resp_h",
                    "DestinationHostname": [ "host", "query", "server_name", ],
                    "DestinationIp": "id.resp_h",
                    "DestinationIP": "id.resp_h",
                    "DestinationPort": "id.resp_p",
                    "dst-ip": "id.resp_h",
                    "dstip": "id.resp_h",
                    "dstport": "id.resp_p",
                    "Host": [ "host", "query", "server_name", ],
                    "HostVersion": "http.version",
                    "http_host": [ "host", "query", "server_name", ],
                    "http_uri": "uri",
                    "http_url": "uri",
                    "http_user_agent": "user_agent",
                    "http.request.url-query-params": "uri",
                    "HttpMethod": "method",
                    "in_url": "uri",
                    "post_url_parameter": "uri",
                    "Request_Url": "uri",
                    "request_url": "uri",
                    "request_URL": "uri",
                    "RequestUrl": "uri",
                    "resource.url": "uri",
                    "resource.URL": "uri",
                    "sc_status": "status_code",
                    "sender_domain": [ "query", "server_name", ],
                    "service.response_code": "status_code",
                    "source": "id.orig_h",
                    "SourceAddr": "id.orig_h",
                    "SourceAddress": "id.orig_h",
                    "SourceIP": "id.orig_h",
                    "SourceIp": "id.orig_h",
                    "SourceNetworkAddress": "id.orig_h",
                    "SourcePort": "id.orig_p",
                    "srcip": "id.orig_h",
                    "Status": "status_code",
                    "status": "status_code",
                    "url": "uri",
                    "URL": "uri",
                    "url_query": "uri",
                    "url.query": "uri",
                    "uri_path": "uri",
                    "user_agent": "user_agent",
                    "user_agent.name": "user_agent",
                    "user-agent": "user_agent",
                    "User-Agent": "user_agent",
                    "useragent": "user_agent",
                    "UserAgent": "user_agent",
                    "User_Agent": "user_agent",
                    "web_dest": [ "host", "query", "server_name", ],
                    "web.dest": [ "host", "query", "server_name", ],
                    "Web.dest": [ "host", "query", "server_name", ],
                    "web.host": [ "host", "query", "server_name", ],
                    "Web.host": [ "host", "query", "server_name", ],
                    "web_method": "method",
                    "Web_method": "method",
                    "web.method": "method",
                    "Web.method": "method",
                    "web_src": "id.orig_h",
                    "web_status": "status_code",
                    "Web_status": "status_code",
                    "web.status": "status_code",
                    "Web.status": "status_code",
                    "web_uri": "uri",
                    "web_url": "uri",
                    "destination.ip": "id.resp_h",
                    "destination.port": "id.resp_p",
                    "http.request.body.content": "post_body",
                    "source.ip": "id.orig_h",
                    "source.port": "id.orig_p",
                }),
                rule_conditions=[
                LogsourceCondition(product="zeek"),
            ],
        ),
    ],
)