"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*********************************************************************************************************************
 Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.

 Licensed under the Apache License, Version 2.0 (the "License").
 You may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 ******************************************************************************************************************** */
const crypto = require("crypto");
const aws_sdk_1 = require("aws-sdk"); // eslint-disable-line
const prepare_spec_1 = require("./prepare-spec");
const s3 = new aws_sdk_1.S3();
/**
 * Prepare the api spec for API Gateway
 * @param inputSpecLocation location of the specification to prepare
 * @param outputSpecLocation location to write the prepared spec to
 * @param options integrations, authorizers etc to apply
 * @return the output location of the prepared spec
 */
const prepare = async ({ inputSpecLocation, outputSpecLocation, ...options }) => {
    // Read the spec from the s3 input location
    const inputSpec = JSON.parse((await s3
        .getObject({
        Bucket: inputSpecLocation.bucket,
        Key: inputSpecLocation.key,
    })
        .promise()).Body.toString("utf-8"));
    // Prepare the spec
    const preparedSpec = prepare_spec_1.prepareApiSpec(inputSpec, options);
    const preparedSpecHash = crypto
        .createHash("sha256")
        .update(JSON.stringify(preparedSpec))
        .digest("hex");
    const outputLocation = {
        bucket: outputSpecLocation.bucket,
        key: `${outputSpecLocation.key}/${preparedSpecHash}.json`,
    };
    // Write the spec to the s3 output location
    await s3
        .putObject({
        Bucket: outputLocation.bucket,
        Key: outputLocation.key,
        Body: JSON.stringify(preparedSpec),
    })
        .promise();
    return outputLocation;
};
exports.handler = async (event) => {
    switch (event.RequestType) {
        case "Create":
        case "Update":
            // Prepare the spec on create
            const outputLocation = await prepare(event.ResourceProperties);
            return {
                PhysicalResourceId: outputLocation.key,
                Status: "SUCCESS",
                Data: {
                    outputSpecKey: outputLocation.key,
                },
            };
        case "Delete":
        // Nothing to do for delete
        default:
            break;
    }
    return {
        PhysicalResourceId: event.PhysicalResourceId,
        Status: "SUCCESS",
    };
};
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaW5kZXguanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi8uLi8uLi9zcmMvY29uc3RydWN0L3ByZXBhcmUtc3BlYy1ldmVudC1oYW5kbGVyL2luZGV4LnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7O0FBQUE7Ozs7Ozs7Ozs7Ozs7O3dIQWN3SDtBQUN4SCxpQ0FBaUM7QUFDakMscUNBQTZCLENBQUMsc0JBQXNCO0FBQ3BELGlEQUF1RTtBQXlFdkUsTUFBTSxFQUFFLEdBQUcsSUFBSSxZQUFFLEVBQUUsQ0FBQztBQUVwQjs7Ozs7O0dBTUc7QUFDSCxNQUFNLE9BQU8sR0FBRyxLQUFLLEVBQUUsRUFDckIsaUJBQWlCLEVBQ2pCLGtCQUFrQixFQUNsQixHQUFHLE9BQU8sRUFDNkIsRUFBdUIsRUFBRTtJQUNoRSwyQ0FBMkM7SUFDM0MsTUFBTSxTQUFTLEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FDMUIsQ0FDRSxNQUFNLEVBQUU7U0FDTCxTQUFTLENBQUM7UUFDVCxNQUFNLEVBQUUsaUJBQWlCLENBQUMsTUFBTTtRQUNoQyxHQUFHLEVBQUUsaUJBQWlCLENBQUMsR0FBRztLQUMzQixDQUFDO1NBQ0QsT0FBTyxFQUFFLENBQ2IsQ0FBQyxJQUFLLENBQUMsUUFBUSxDQUFDLE9BQU8sQ0FBQyxDQUMxQixDQUFDO0lBRUYsbUJBQW1CO0lBQ25CLE1BQU0sWUFBWSxHQUFHLDZCQUFjLENBQUMsU0FBUyxFQUFFLE9BQU8sQ0FBQyxDQUFDO0lBQ3hELE1BQU0sZ0JBQWdCLEdBQUcsTUFBTTtTQUM1QixVQUFVLENBQUMsUUFBUSxDQUFDO1NBQ3BCLE1BQU0sQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLFlBQVksQ0FBQyxDQUFDO1NBQ3BDLE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQztJQUVqQixNQUFNLGNBQWMsR0FBZTtRQUNqQyxNQUFNLEVBQUUsa0JBQWtCLENBQUMsTUFBTTtRQUNqQyxHQUFHLEVBQUUsR0FBRyxrQkFBa0IsQ0FBQyxHQUFHLElBQUksZ0JBQWdCLE9BQU87S0FDMUQsQ0FBQztJQUVGLDJDQUEyQztJQUMzQyxNQUFNLEVBQUU7U0FDTCxTQUFTLENBQUM7UUFDVCxNQUFNLEVBQUUsY0FBYyxDQUFDLE1BQU07UUFDN0IsR0FBRyxFQUFFLGNBQWMsQ0FBQyxHQUFHO1FBQ3ZCLElBQUksRUFBRSxJQUFJLENBQUMsU0FBUyxDQUFDLFlBQVksQ0FBQztLQUNuQyxDQUFDO1NBQ0QsT0FBTyxFQUFFLENBQUM7SUFFYixPQUFPLGNBQWMsQ0FBQztBQUN4QixDQUFDLENBQUM7QUFFRixPQUFPLENBQUMsT0FBTyxHQUFHLEtBQUssRUFBRSxLQUFxQixFQUE0QixFQUFFO0lBQzFFLFFBQVEsS0FBSyxDQUFDLFdBQVcsRUFBRTtRQUN6QixLQUFLLFFBQVEsQ0FBQztRQUNkLEtBQUssUUFBUTtZQUNYLDZCQUE2QjtZQUM3QixNQUFNLGNBQWMsR0FBRyxNQUFNLE9BQU8sQ0FBQyxLQUFLLENBQUMsa0JBQWtCLENBQUMsQ0FBQztZQUMvRCxPQUFPO2dCQUNMLGtCQUFrQixFQUFFLGNBQWMsQ0FBQyxHQUFHO2dCQUN0QyxNQUFNLEVBQUUsU0FBUztnQkFDakIsSUFBSSxFQUFFO29CQUNKLGFBQWEsRUFBRSxjQUFjLENBQUMsR0FBRztpQkFDbEM7YUFDRixDQUFDO1FBQ0osS0FBSyxRQUFRLENBQUM7UUFDZCwyQkFBMkI7UUFDM0I7WUFDRSxNQUFNO0tBQ1Q7SUFFRCxPQUFPO1FBQ0wsa0JBQWtCLEVBQUUsS0FBSyxDQUFDLGtCQUFtQjtRQUM3QyxNQUFNLEVBQUUsU0FBUztLQUNsQixDQUFDO0FBQ0osQ0FBQyxDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiLyoqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKlxuIENvcHlyaWdodCBBbWF6b24uY29tLCBJbmMuIG9yIGl0cyBhZmZpbGlhdGVzLiBBbGwgUmlnaHRzIFJlc2VydmVkLlxuXG4gTGljZW5zZWQgdW5kZXIgdGhlIEFwYWNoZSBMaWNlbnNlLCBWZXJzaW9uIDIuMCAodGhlIFwiTGljZW5zZVwiKS5cbiBZb3UgbWF5IG5vdCB1c2UgdGhpcyBmaWxlIGV4Y2VwdCBpbiBjb21wbGlhbmNlIHdpdGggdGhlIExpY2Vuc2UuXG4gWW91IG1heSBvYnRhaW4gYSBjb3B5IG9mIHRoZSBMaWNlbnNlIGF0XG5cbiBodHRwOi8vd3d3LmFwYWNoZS5vcmcvbGljZW5zZXMvTElDRU5TRS0yLjBcblxuIFVubGVzcyByZXF1aXJlZCBieSBhcHBsaWNhYmxlIGxhdyBvciBhZ3JlZWQgdG8gaW4gd3JpdGluZywgc29mdHdhcmVcbiBkaXN0cmlidXRlZCB1bmRlciB0aGUgTGljZW5zZSBpcyBkaXN0cmlidXRlZCBvbiBhbiBcIkFTIElTXCIgQkFTSVMsXG4gV0lUSE9VVCBXQVJSQU5USUVTIE9SIENPTkRJVElPTlMgT0YgQU5ZIEtJTkQsIGVpdGhlciBleHByZXNzIG9yIGltcGxpZWQuXG4gU2VlIHRoZSBMaWNlbnNlIGZvciB0aGUgc3BlY2lmaWMgbGFuZ3VhZ2UgZ292ZXJuaW5nIHBlcm1pc3Npb25zIGFuZFxuIGxpbWl0YXRpb25zIHVuZGVyIHRoZSBMaWNlbnNlLlxuICoqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqICovXG5pbXBvcnQgKiBhcyBjcnlwdG8gZnJvbSBcImNyeXB0b1wiO1xuaW1wb3J0IHsgUzMgfSBmcm9tIFwiYXdzLXNka1wiOyAvLyBlc2xpbnQtZGlzYWJsZS1saW5lXG5pbXBvcnQgeyBwcmVwYXJlQXBpU3BlYywgUHJlcGFyZUFwaVNwZWNPcHRpb25zIH0gZnJvbSBcIi4vcHJlcGFyZS1zcGVjXCI7XG5cbi8qKlxuICogUmVwcmVzZW50cyBhbiBvYmplY3QgbG9jYXRpb24gaW4gYW4gczMgYnVja2V0XG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgUzNMb2NhdGlvbiB7XG4gIC8qKlxuICAgKiBUaGUgYnVja2V0IGluIHdoaWNoIHRoZSBvYmplY3QgcmVzaWRlc1xuICAgKi9cbiAgcmVhZG9ubHkgYnVja2V0OiBzdHJpbmc7XG4gIC8qKlxuICAgKiBUaGUgb2JqZWN0IGtleVxuICAgKi9cbiAgcmVhZG9ubHkga2V5OiBzdHJpbmc7XG59XG5cbi8qKlxuICogUHJvcGVydGllcyByZXF1aXJlZCB0byBwcmVwYXJlIHRoZSBhcGkgc3BlY2lmaWNhdGlvbiB3aXRoIHRoZSBnaXZlbiBpbnRlZ3JhdGlvbnMsIGF1dGhvcml6ZXJzLCBldGNcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBQcmVwYXJlQXBpU3BlY0N1c3RvbVJlc291cmNlUHJvcGVydGllc1xuICBleHRlbmRzIFByZXBhcmVBcGlTcGVjT3B0aW9ucyB7XG4gIC8qKlxuICAgKiBUaGUgbG9jYXRpb24gZnJvbSB3aGljaCB0byByZWFkIHRoZSBzcGVjIHRvIHByZXBhcmVcbiAgICovXG4gIHJlYWRvbmx5IGlucHV0U3BlY0xvY2F0aW9uOiBTM0xvY2F0aW9uO1xuICAvKipcbiAgICogVGhlIGxvY2F0aW9uIHRvIHdyaXRlIHRoZSBwcmVwYXJlZCBzcGVjLiBOb3RlIHRoYXQgdGhlIGtleSBpcyB1c2VkIGFzIGEgcHJlZml4IGFuZCB0aGUgb3V0cHV0IGxvY2F0aW9uIHdpbGxcbiAgICogaW5jbHVkZSBhIGhhc2guXG4gICAqL1xuICByZWFkb25seSBvdXRwdXRTcGVjTG9jYXRpb246IFMzTG9jYXRpb247XG59XG5cbi8qKlxuICogQ2xvdWRmb3JtYXRpb24gZXZlbnQgdHlwZSBmb3IgY3VzdG9tIHJlc291cmNlXG4gKi9cbmludGVyZmFjZSBPbkV2ZW50UmVxdWVzdCB7XG4gIC8qKlxuICAgKiBUaGUgdHlwZSBvZiBjbG91ZGZvcm1hdGlvbiByZXF1ZXN0XG4gICAqL1xuICByZWFkb25seSBSZXF1ZXN0VHlwZTogXCJDcmVhdGVcIiB8IFwiVXBkYXRlXCIgfCBcIkRlbGV0ZVwiO1xuICAvKipcbiAgICogUGh5c2ljYWwgcmVzb3VyY2UgaWQgb2YgdGhlIGN1c3RvbSByZXNvdXJjZVxuICAgKi9cbiAgcmVhZG9ubHkgUGh5c2ljYWxSZXNvdXJjZUlkPzogc3RyaW5nO1xuICAvKipcbiAgICogUHJvcGVydGllcyBmb3IgcHJlcGFyaW5nIHRoZSBhcGlcbiAgICovXG4gIHJlYWRvbmx5IFJlc291cmNlUHJvcGVydGllczogUHJlcGFyZUFwaVNwZWNDdXN0b21SZXNvdXJjZVByb3BlcnRpZXM7XG59XG5cbi8qKlxuICogQ3VzdG9tIHJlc291cmNlIHJlc3BvbnNlXG4gKi9cbmludGVyZmFjZSBPbkV2ZW50UmVzcG9uc2Uge1xuICAvKipcbiAgICogUGh5c2ljYWwgcmVzb3VyY2UgaWQgb2YgdGhlIGN1c3RvbSByZXNvdXJjZVxuICAgKi9cbiAgcmVhZG9ubHkgUGh5c2ljYWxSZXNvdXJjZUlkOiBzdHJpbmc7XG4gIC8qKlxuICAgKiBTdGF0dXMgb2YgdGhlIGN1c3RvbSByZXNvdXJjZVxuICAgKi9cbiAgcmVhZG9ubHkgU3RhdHVzOiBcIlNVQ0NFU1NcIiB8IFwiRkFJTEVEXCI7XG4gIC8qKlxuICAgKiBEYXRhIHJldHVybmVkIGJ5IHRoZSBjdXN0b20gcmVzb3VyY2VcbiAgICovXG4gIHJlYWRvbmx5IERhdGE/OiB7XG4gICAgLyoqXG4gICAgICogVGhlIGtleSBmb3IgdGhlIG91dHB1dCBzcGVjIGluIHRoZSBvdXRwdXQgYnVja2V0XG4gICAgICovXG4gICAgcmVhZG9ubHkgb3V0cHV0U3BlY0tleTogc3RyaW5nO1xuICB9O1xufVxuXG5jb25zdCBzMyA9IG5ldyBTMygpO1xuXG4vKipcbiAqIFByZXBhcmUgdGhlIGFwaSBzcGVjIGZvciBBUEkgR2F0ZXdheVxuICogQHBhcmFtIGlucHV0U3BlY0xvY2F0aW9uIGxvY2F0aW9uIG9mIHRoZSBzcGVjaWZpY2F0aW9uIHRvIHByZXBhcmVcbiAqIEBwYXJhbSBvdXRwdXRTcGVjTG9jYXRpb24gbG9jYXRpb24gdG8gd3JpdGUgdGhlIHByZXBhcmVkIHNwZWMgdG9cbiAqIEBwYXJhbSBvcHRpb25zIGludGVncmF0aW9ucywgYXV0aG9yaXplcnMgZXRjIHRvIGFwcGx5XG4gKiBAcmV0dXJuIHRoZSBvdXRwdXQgbG9jYXRpb24gb2YgdGhlIHByZXBhcmVkIHNwZWNcbiAqL1xuY29uc3QgcHJlcGFyZSA9IGFzeW5jICh7XG4gIGlucHV0U3BlY0xvY2F0aW9uLFxuICBvdXRwdXRTcGVjTG9jYXRpb24sXG4gIC4uLm9wdGlvbnNcbn06IFByZXBhcmVBcGlTcGVjQ3VzdG9tUmVzb3VyY2VQcm9wZXJ0aWVzKTogUHJvbWlzZTxTM0xvY2F0aW9uPiA9PiB7XG4gIC8vIFJlYWQgdGhlIHNwZWMgZnJvbSB0aGUgczMgaW5wdXQgbG9jYXRpb25cbiAgY29uc3QgaW5wdXRTcGVjID0gSlNPTi5wYXJzZShcbiAgICAoXG4gICAgICBhd2FpdCBzM1xuICAgICAgICAuZ2V0T2JqZWN0KHtcbiAgICAgICAgICBCdWNrZXQ6IGlucHV0U3BlY0xvY2F0aW9uLmJ1Y2tldCxcbiAgICAgICAgICBLZXk6IGlucHV0U3BlY0xvY2F0aW9uLmtleSxcbiAgICAgICAgfSlcbiAgICAgICAgLnByb21pc2UoKVxuICAgICkuQm9keSEudG9TdHJpbmcoXCJ1dGYtOFwiKVxuICApO1xuXG4gIC8vIFByZXBhcmUgdGhlIHNwZWNcbiAgY29uc3QgcHJlcGFyZWRTcGVjID0gcHJlcGFyZUFwaVNwZWMoaW5wdXRTcGVjLCBvcHRpb25zKTtcbiAgY29uc3QgcHJlcGFyZWRTcGVjSGFzaCA9IGNyeXB0b1xuICAgIC5jcmVhdGVIYXNoKFwic2hhMjU2XCIpXG4gICAgLnVwZGF0ZShKU09OLnN0cmluZ2lmeShwcmVwYXJlZFNwZWMpKVxuICAgIC5kaWdlc3QoXCJoZXhcIik7XG5cbiAgY29uc3Qgb3V0cHV0TG9jYXRpb246IFMzTG9jYXRpb24gPSB7XG4gICAgYnVja2V0OiBvdXRwdXRTcGVjTG9jYXRpb24uYnVja2V0LFxuICAgIGtleTogYCR7b3V0cHV0U3BlY0xvY2F0aW9uLmtleX0vJHtwcmVwYXJlZFNwZWNIYXNofS5qc29uYCxcbiAgfTtcblxuICAvLyBXcml0ZSB0aGUgc3BlYyB0byB0aGUgczMgb3V0cHV0IGxvY2F0aW9uXG4gIGF3YWl0IHMzXG4gICAgLnB1dE9iamVjdCh7XG4gICAgICBCdWNrZXQ6IG91dHB1dExvY2F0aW9uLmJ1Y2tldCxcbiAgICAgIEtleTogb3V0cHV0TG9jYXRpb24ua2V5LFxuICAgICAgQm9keTogSlNPTi5zdHJpbmdpZnkocHJlcGFyZWRTcGVjKSxcbiAgICB9KVxuICAgIC5wcm9taXNlKCk7XG5cbiAgcmV0dXJuIG91dHB1dExvY2F0aW9uO1xufTtcblxuZXhwb3J0cy5oYW5kbGVyID0gYXN5bmMgKGV2ZW50OiBPbkV2ZW50UmVxdWVzdCk6IFByb21pc2U8T25FdmVudFJlc3BvbnNlPiA9PiB7XG4gIHN3aXRjaCAoZXZlbnQuUmVxdWVzdFR5cGUpIHtcbiAgICBjYXNlIFwiQ3JlYXRlXCI6XG4gICAgY2FzZSBcIlVwZGF0ZVwiOlxuICAgICAgLy8gUHJlcGFyZSB0aGUgc3BlYyBvbiBjcmVhdGVcbiAgICAgIGNvbnN0IG91dHB1dExvY2F0aW9uID0gYXdhaXQgcHJlcGFyZShldmVudC5SZXNvdXJjZVByb3BlcnRpZXMpO1xuICAgICAgcmV0dXJuIHtcbiAgICAgICAgUGh5c2ljYWxSZXNvdXJjZUlkOiBvdXRwdXRMb2NhdGlvbi5rZXksXG4gICAgICAgIFN0YXR1czogXCJTVUNDRVNTXCIsXG4gICAgICAgIERhdGE6IHtcbiAgICAgICAgICBvdXRwdXRTcGVjS2V5OiBvdXRwdXRMb2NhdGlvbi5rZXksXG4gICAgICAgIH0sXG4gICAgICB9O1xuICAgIGNhc2UgXCJEZWxldGVcIjpcbiAgICAvLyBOb3RoaW5nIHRvIGRvIGZvciBkZWxldGVcbiAgICBkZWZhdWx0OlxuICAgICAgYnJlYWs7XG4gIH1cblxuICByZXR1cm4ge1xuICAgIFBoeXNpY2FsUmVzb3VyY2VJZDogZXZlbnQuUGh5c2ljYWxSZXNvdXJjZUlkISxcbiAgICBTdGF0dXM6IFwiU1VDQ0VTU1wiLFxuICB9O1xufTtcbiJdfQ==