"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.OpenApiGatewayLambdaApi = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*********************************************************************************************************************
 Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.

 Licensed under the Apache License, Version 2.0 (the "License").
 You may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 ******************************************************************************************************************** */
const pdk_nag_1 = require("@aws-prototyping-sdk/pdk-nag");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_apigateway_1 = require("aws-cdk-lib/aws-apigateway");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const cdk_nag_1 = require("cdk-nag");
const api_gateway_integrations_1 = require("./spec/api-gateway-integrations");
/**
 * A construct for creating an api gateway api based on the definition in the OpenAPI spec.
 */
class OpenApiGatewayLambdaApi extends aws_apigateway_1.SpecRestApi {
    constructor(scope, id, props) {
        const { integrations, spec, ...options } = props;
        // Create the api gateway resources from the spec, augmenting the spec with the properties specific to api gateway
        // such as integrations or auth types
        super(scope, id, {
            apiDefinition: aws_apigateway_1.ApiDefinition.fromInline(api_gateway_integrations_1.prepareApiSpec(scope, spec, props)),
            deployOptions: {
                accessLogDestination: new aws_apigateway_1.LogGroupLogDestination(new aws_logs_1.LogGroup(scope, `AccessLogs`)),
                accessLogFormat: aws_apigateway_1.AccessLogFormat.clf(),
                loggingLevel: aws_apigateway_1.MethodLoggingLevel.INFO,
            },
            ...options,
        });
        // Grant API Gateway permission to invoke each lambda which implements an integration or custom authorizer
        const stack = aws_cdk_lib_1.Stack.of(this);
        api_gateway_integrations_1.getLabelledFunctions(props).forEach(({ label, function: lambda }) => {
            new aws_lambda_1.CfnPermission(this, `LambdaPermission-${label}`, {
                action: "lambda:InvokeFunction",
                principal: "apigateway.amazonaws.com",
                functionName: lambda.functionArn,
                sourceArn: stack.formatArn({
                    service: "execute-api",
                    resource: this.restApiId,
                    resourceName: "*/*",
                }),
            });
        });
        cdk_nag_1.NagSuppressions.addResourceSuppressions(this, [
            {
                id: "AwsSolutions-IAM4",
                reason: "Cloudwatch Role requires access to create/read groups at the root level.",
                appliesTo: [
                    {
                        regex: `/^Policy::arn:${pdk_nag_1.PDKNag.getStackPartitionRegex(stack)}:iam::aws:policy/service-role/AmazonAPIGatewayPushToCloudWatchLogs$/g`,
                    },
                ],
            },
        ], true);
        cdk_nag_1.NagSuppressions.addResourceSuppressions(this, [
            {
                id: "AwsSolutions-APIG2",
                reason: "This construct implements fine grained validation via OpenApi.",
            },
        ], true);
    }
}
exports.OpenApiGatewayLambdaApi = OpenApiGatewayLambdaApi;
_a = JSII_RTTI_SYMBOL_1;
OpenApiGatewayLambdaApi[_a] = { fqn: "@aws-prototyping-sdk/open-api-gateway.OpenApiGatewayLambdaApi", version: "0.3.14" };
//# sourceMappingURL=data:application/json;base64,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