"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const child_process_1 = require("child_process");
const os_1 = require("os");
const path_1 = require("path");
const fs_extra_1 = require("fs-extra");
const tasks_1 = require("../../tasks");
const util_1 = require("../util");
test('minimal case (just a shell command)', () => {
    // GIVEN
    const p = new util_1.TestProject();
    // WHEN
    p.addTask('test', {
        exec: 'echo hello_tasks!',
    });
    // THEN
    expect(executeTask(p, 'test')).toEqual(['hello_tasks!']);
});
test('fails if the step fails', () => {
    // GIVEN
    const p = new util_1.TestProject();
    // WHEN
    p.addTask('test', {
        exec: 'false',
    });
    // THEN
    expect(() => executeTask(p, 'test')).toThrow(/Task \"test\" failed when executing \"false\"/);
});
test('multiple steps', () => {
    // GIVEN
    const p = new util_1.TestProject();
    const t = p.addTask('test');
    // WHEN
    t.exec('echo step1');
    t.exec('echo step2');
    t.exec('echo step3');
    // THEN
    expect(executeTask(p, 'test')).toEqual([
        'step1',
        'step2',
        'step3',
    ]);
});
test('execution stops if a step fails', () => {
    // GIVEN
    const p = new util_1.TestProject();
    const t = p.addTask('test');
    // WHEN
    t.exec('echo step1');
    t.exec('echo step2');
    t.exec('echo step3');
    t.exec('echo failing && false');
    t.exec('echo step4');
    // THEN
    expect(() => executeTask(p, 'test')).toThrow(/Task \"test\" failed when executing \"echo failing && false\"/);
});
describe('condition', () => {
    test('zero exit code means that steps should be executed', () => {
        // GIVEN
        const p = new util_1.TestProject();
        // WHEN
        const t = p.addTask('foo', {
            condition: 'echo evaluating_condition',
        });
        t.exec('echo step1');
        t.exec('echo step2');
        // THEN
        expect(executeTask(p, 'foo')).toEqual([
            'evaluating_condition',
            'step1',
            'step2',
        ]);
    });
    test('non-zero exit code means steps should not be executed', () => {
        // GIVEN
        const p = new util_1.TestProject();
        // WHEN
        const t = p.addTask('foo', {
            condition: 'echo failing_condition && false',
        });
        t.exec('echo step1');
        t.exec('echo step2');
        // THEN
        expect(executeTask(p, 'foo')).toEqual([
            'failing_condition',
        ]);
    });
});
describe('cwd', () => {
    test('default cwd is project root', () => {
        const p = new util_1.TestProject();
        p.addTask('test', { exec: 'echo cwd is $PWD' });
        expect(executeTask(p, 'test')[0].includes(path_1.basename(p.outdir))).toBeTruthy();
    });
    test('if a step changes cwd, it will not affect next steps', () => {
        const p = new util_1.TestProject();
        const task = p.addTask('test');
        task.exec('cd /tmp');
        task.exec('echo $PWD');
        expect(executeTask(p, 'test')[0].includes(path_1.basename(p.outdir))).toBeTruthy();
    });
    test('cwd can be set at the task level', () => {
        const p = new util_1.TestProject();
        const cwd = path_1.join(p.outdir, 'mypwd');
        fs_extra_1.mkdirpSync(cwd);
        const task = p.addTask('test', {
            cwd,
        });
        task.exec('echo step1=$PWD');
        task.exec('echo step2=$PWD');
        for (const line of executeTask(p, 'test')) {
            expect(line.includes('mypwd')).toBeTruthy();
        }
    });
    test('cwd can be set at step level', () => {
        const p = new util_1.TestProject();
        const taskcwd = path_1.join(p.outdir, 'mypwd');
        const stepcwd = path_1.join(p.outdir, 'yourpwd');
        fs_extra_1.mkdirpSync(taskcwd);
        fs_extra_1.mkdirpSync(stepcwd);
        const task = p.addTask('test', { cwd: taskcwd });
        task.exec('echo step1=$PWD');
        task.exec('echo step2=$PWD', { cwd: stepcwd });
        const lines = executeTask(p, 'test');
        expect(lines[0].includes('mypwd')).toBeTruthy();
        expect(lines[1].includes('yourpwd')).toBeTruthy();
    });
    test('fails gracefully if cwd does not exist (task level)', () => {
        const p = new util_1.TestProject();
        p.addTask('test', {
            cwd: path_1.join(p.outdir, 'not-found'),
            exec: 'echo hi',
        });
        expect(() => executeTask(p, 'test')).toThrow(/invalid workdir/);
    });
    test('fails gracefully if cwd does not exist (step level)', () => {
        const p = new util_1.TestProject();
        const task = p.addTask('test');
        task.exec('echo step', { cwd: path_1.join(p.outdir, 'mystep') });
        expect(() => executeTask(p, 'test')).toThrow(/must be an existing directory/);
    });
});
describe('say', () => {
    test('"say" can be used to print an info log during execution', () => {
        const p = new util_1.TestProject();
        const task = p.addTask('say');
        task.say('hello, world');
        p.synth();
        const rt = new tasks_1.TaskRuntime(p.outdir);
        expect(rt.tasks).toStrictEqual([{
                name: 'say',
                steps: [{ say: 'hello, world' }],
            }]);
    });
});
function executeTask(p, taskName) {
    p.synth();
    const args = [
        require.resolve('../../cli'),
        taskName,
    ].map(x => `"${x}"`);
    const result = child_process_1.spawnSync(`"${process.execPath}"`, args, { cwd: p.outdir, shell: true, env: { ...process.env } });
    if (result.status !== 0) {
        throw new Error(`non-zero exit code: ${result.stderr.toString('utf-8')}`);
    }
    return result.stdout.toString('utf-8').trim().split(os_1.EOL);
}
//# sourceMappingURL=data:application/json;base64,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