"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.OpenApiGatewayRestApi = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const path = require("path");
const pdk_nag_1 = require("@aws-prototyping-sdk/pdk-nag");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_apigateway_1 = require("aws-cdk-lib/aws-apigateway");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const aws_s3_assets_1 = require("aws-cdk-lib/aws-s3-assets");
const custom_resources_1 = require("aws-cdk-lib/custom-resources");
const cdk_nag_1 = require("cdk-nag");
const constructs_1 = require("constructs");
const prepare_spec_1 = require("./prepare-spec-event-handler/prepare-spec");
const api_gateway_auth_1 = require("./spec/api-gateway-auth");
const api_gateway_integrations_1 = require("./spec/api-gateway-integrations");
const open_api_gateway_web_acl_1 = require("./waf/open-api-gateway-web-acl");
/**
 * A construct for creating an api gateway rest api based on the definition in the OpenAPI spec.
 */
class OpenApiGatewayRestApi extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const { integrations, spec, specPath, operationLookup, defaultAuthorizer, corsOptions, ...options } = props;
        // Upload the spec to s3 as an asset
        const inputSpecAsset = new aws_s3_assets_1.Asset(this, "InputSpec", {
            path: specPath,
        });
        // We'll output the prepared spec in the same asset bucket
        const preparedSpecOutputKeyPrefix = `${inputSpecAsset.s3ObjectKey}-prepared`;
        const stack = aws_cdk_lib_1.Stack.of(this);
        const prepareSpecLambdaName = `${pdk_nag_1.PDKNag.getStackPrefix(stack)
            .split("/")
            .join("-")}PrepareSpec`;
        const prepareSpecRole = new aws_iam_1.Role(this, "PrepareSpecRole", {
            assumedBy: new aws_iam_1.ServicePrincipal("lambda.amazonaws.com"),
            inlinePolicies: {
                logs: new aws_iam_1.PolicyDocument({
                    statements: [
                        new aws_iam_1.PolicyStatement({
                            effect: aws_iam_1.Effect.ALLOW,
                            actions: [
                                "logs:CreateLogGroup",
                                "logs:CreateLogStream",
                                "logs:PutLogEvents",
                            ],
                            resources: [
                                `arn:aws:logs:${stack.region}:${stack.account}:log-group:/aws/lambda/${prepareSpecLambdaName}`,
                                `arn:aws:logs:${stack.region}:${stack.account}:log-group:/aws/lambda/${prepareSpecLambdaName}:*`,
                            ],
                        }),
                    ],
                }),
                s3: new aws_iam_1.PolicyDocument({
                    statements: [
                        new aws_iam_1.PolicyStatement({
                            effect: aws_iam_1.Effect.ALLOW,
                            actions: ["s3:getObject"],
                            resources: [
                                inputSpecAsset.bucket.arnForObjects(inputSpecAsset.s3ObjectKey),
                            ],
                        }),
                        new aws_iam_1.PolicyStatement({
                            effect: aws_iam_1.Effect.ALLOW,
                            actions: ["s3:putObject"],
                            resources: [
                                // The output file will include a hash of the prepared spec, which is not known until deploy time since
                                // tokens must be resolved
                                inputSpecAsset.bucket.arnForObjects(`${preparedSpecOutputKeyPrefix}/*`),
                            ],
                        }),
                    ],
                }),
            },
        });
        ["AwsSolutions-IAM5", "AwsPrototyping-IAMNoWildcardPermissions"].forEach((RuleId) => {
            cdk_nag_1.NagSuppressions.addResourceSuppressions(prepareSpecRole, [
                {
                    id: RuleId,
                    reason: "Cloudwatch resources have been scoped down to the LogGroup level, however * is still needed as stream names are created just in time.",
                    appliesTo: [
                        {
                            regex: `/^Resource::arn:aws:logs:${pdk_nag_1.PDKNag.getStackRegionRegex(stack)}:${pdk_nag_1.PDKNag.getStackAccountRegex(stack)}:log-group:/aws/lambda/${prepareSpecLambdaName}:\*/g`,
                        },
                    ],
                },
                {
                    id: RuleId,
                    reason: "S3 resources have been scoped down to the appropriate prefix in the CDK asset bucket, however * is still needed as since the prepared spec hash is not known until deploy time.",
                    appliesTo: [
                        {
                            regex: `/^Resource::arn:${pdk_nag_1.PDKNag.getStackPartitionRegex(stack)}:s3:.*/${preparedSpecOutputKeyPrefix}/\*/g`,
                        },
                    ],
                },
            ], true);
        });
        // Create a custom resource for preparing the spec for deployment (adding integrations, authorizers, etc)
        const prepareSpec = new aws_lambda_1.Function(this, "PrepareSpec", {
            handler: "index.handler",
            runtime: aws_lambda_1.Runtime.NODEJS_18_X,
            code: aws_lambda_1.Code.fromAsset(path.join(__dirname, "../../lib/construct/prepare-spec-event-handler")),
            timeout: aws_cdk_lib_1.Duration.seconds(30),
            role: prepareSpecRole,
            functionName: prepareSpecLambdaName,
        });
        const providerFunctionName = `${prepareSpecLambdaName}-Provider`;
        const providerRole = new aws_iam_1.Role(this, "PrepareSpecProviderRole", {
            assumedBy: new aws_iam_1.ServicePrincipal("lambda.amazonaws.com"),
            inlinePolicies: {
                logs: new aws_iam_1.PolicyDocument({
                    statements: [
                        new aws_iam_1.PolicyStatement({
                            effect: aws_iam_1.Effect.ALLOW,
                            actions: [
                                "logs:CreateLogGroup",
                                "logs:CreateLogStream",
                                "logs:PutLogEvents",
                            ],
                            resources: [
                                `arn:aws:logs:${stack.region}:${stack.account}:log-group:/aws/lambda/${providerFunctionName}`,
                                `arn:aws:logs:${stack.region}:${stack.account}:log-group:/aws/lambda/${providerFunctionName}:*`,
                            ],
                        }),
                    ],
                }),
            },
        });
        const provider = new custom_resources_1.Provider(this, "PrepareSpecProvider", {
            onEventHandler: prepareSpec,
            role: providerRole,
            providerFunctionName,
        });
        ["AwsSolutions-IAM5", "AwsPrototyping-IAMNoWildcardPermissions"].forEach((RuleId) => {
            cdk_nag_1.NagSuppressions.addResourceSuppressions(providerRole, [
                {
                    id: RuleId,
                    reason: "Cloudwatch resources have been scoped down to the LogGroup level, however * is still needed as stream names are created just in time.",
                },
            ], true);
        });
        ["AwsSolutions-L1", "AwsPrototyping-LambdaLatestVersion"].forEach((RuleId) => {
            cdk_nag_1.NagSuppressions.addResourceSuppressions(provider, [
                {
                    id: RuleId,
                    reason: "Latest runtime cannot be configured. CDK will need to upgrade the Provider construct accordingly.",
                },
            ], true);
        });
        const prepareSpecOptions = {
            defaultAuthorizerReference: api_gateway_auth_1.serializeAsAuthorizerReference(defaultAuthorizer),
            integrations: Object.fromEntries(Object.entries(integrations).map(([operationId, { authorizer, integration }]) => [
                operationId,
                {
                    integration: integration.render({
                        operationId,
                        scope: this,
                        ...operationLookup[operationId],
                    }),
                    methodAuthorizer: api_gateway_auth_1.serializeAsAuthorizerReference(authorizer),
                },
            ])),
            securitySchemes: api_gateway_auth_1.prepareSecuritySchemes(this, integrations, defaultAuthorizer),
            corsOptions: corsOptions && {
                allowHeaders: corsOptions.allowHeaders || aws_apigateway_1.Cors.DEFAULT_HEADERS,
                allowMethods: corsOptions.allowMethods || aws_apigateway_1.Cors.ALL_METHODS,
                allowOrigins: corsOptions.allowOrigins,
                statusCode: corsOptions.statusCode || 204,
            },
            operationLookup,
        };
        // Spec preparation will happen in a custom resource lambda so that references to lambda integrations etc can be
        // resolved. However, we also prepare inline to perform some additional validation at synth time.
        const preparedSpec = prepare_spec_1.prepareApiSpec(spec, prepareSpecOptions);
        const prepareApiSpecCustomResourceProperties = {
            inputSpecLocation: {
                bucket: inputSpecAsset.bucket.bucketName,
                key: inputSpecAsset.s3ObjectKey,
            },
            outputSpecLocation: {
                bucket: inputSpecAsset.bucket.bucketName,
                key: preparedSpecOutputKeyPrefix,
            },
            ...prepareSpecOptions,
        };
        const prepareSpecCustomResource = new aws_cdk_lib_1.CustomResource(this, "PrepareSpecCustomResource", {
            serviceToken: provider.serviceToken,
            properties: prepareApiSpecCustomResourceProperties,
        });
        // Create the api gateway resources from the spec, augmenting the spec with the properties specific to api gateway
        // such as integrations or auth types
        this.api = new aws_apigateway_1.SpecRestApi(this, id, {
            apiDefinition: aws_apigateway_1.ApiDefinition.fromBucket(inputSpecAsset.bucket, prepareSpecCustomResource.getAttString("outputSpecKey")),
            deployOptions: {
                accessLogDestination: new aws_apigateway_1.LogGroupLogDestination(new aws_logs_1.LogGroup(scope, `AccessLogs`)),
                accessLogFormat: aws_apigateway_1.AccessLogFormat.clf(),
                loggingLevel: aws_apigateway_1.MethodLoggingLevel.INFO,
            },
            ...options,
        });
        this.api.node.addDependency(prepareSpecCustomResource);
        // While the api will be updated when the output path from the custom resource changes, CDK still needs to know when
        // to redeploy the api. This is achieved by including a hash of the spec in the logical id (internalised in the
        // addToLogicalId method since this is how changes of individual resources/methods etc trigger redeployments in CDK)
        this.api.latestDeployment?.addToLogicalId(preparedSpec);
        // Grant API Gateway permission to invoke the integrations
        Object.keys(integrations).forEach((operationId) => {
            integrations[operationId].integration.grant({
                operationId,
                scope: this,
                api: this.api,
                ...operationLookup[operationId],
            });
        });
        // Grant API Gateway permission to invoke each custom authorizer lambda (if any)
        api_gateway_integrations_1.getAuthorizerFunctions(props).forEach(({ label, function: lambda }) => {
            new aws_lambda_1.CfnPermission(this, `LambdaPermission-${label}`, {
                action: "lambda:InvokeFunction",
                principal: "apigateway.amazonaws.com",
                functionName: lambda.functionArn,
                sourceArn: stack.formatArn({
                    service: "execute-api",
                    resource: this.api.restApiId,
                    resourceName: "*/*",
                }),
            });
        });
        // Create and associate the web acl if not disabled
        if (!props.webAclOptions?.disable) {
            const acl = new open_api_gateway_web_acl_1.OpenApiGatewayWebAcl(this, `${id}-Acl`, {
                ...props.webAclOptions,
                apiDeploymentStageArn: this.api.deploymentStage.stageArn,
            });
            this.webAcl = acl.webAcl;
            this.ipSet = acl.ipSet;
            this.webAclAssociation = acl.webAclAssociation;
        }
        ["AwsSolutions-IAM4", "AwsPrototyping-IAMNoManagedPolicies"].forEach((RuleId) => {
            cdk_nag_1.NagSuppressions.addResourceSuppressions(this, [
                {
                    id: RuleId,
                    reason: "Cloudwatch Role requires access to create/read groups at the root level.",
                    appliesTo: [
                        {
                            regex: `/^Policy::arn:${pdk_nag_1.PDKNag.getStackPartitionRegex(stack)}:iam::aws:policy/service-role/AmazonAPIGatewayPushToCloudWatchLogs$/g`,
                        },
                    ],
                },
            ], true);
        });
        ["AwsSolutions-APIG2", "AwsPrototyping-APIGWRequestValidation"].forEach((RuleId) => {
            cdk_nag_1.NagSuppressions.addResourceSuppressions(this, [
                {
                    id: RuleId,
                    reason: "This construct implements fine grained validation via OpenApi.",
                },
            ], true);
        });
    }
}
exports.OpenApiGatewayRestApi = OpenApiGatewayRestApi;
_a = JSII_RTTI_SYMBOL_1;
OpenApiGatewayRestApi[_a] = { fqn: "@aws-prototyping-sdk/open-api-gateway.OpenApiGatewayRestApi", version: "0.14.16" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoib3Blbi1hcGktZ2F0ZXdheS1yZXN0LWFwaS5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uLy4uL3NyYy9jb25zdHJ1Y3Qvb3Blbi1hcGktZ2F0ZXdheS1yZXN0LWFwaS50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7OztBQUFBO3NDQUNzQztBQUN0Qyw2QkFBNkI7QUFDN0IsMERBQXNEO0FBQ3RELDZDQUE4RDtBQUM5RCwrREFRb0M7QUFDcEMsaURBTTZCO0FBQzdCLHVEQUtnQztBQUNoQyxtREFBZ0Q7QUFDaEQsNkRBQWtEO0FBTWxELG1FQUF3RDtBQUN4RCxxQ0FBMEM7QUFDMUMsMkNBQXVDO0FBRXZDLDRFQUdtRDtBQUVuRCw4REFHaUM7QUFDakMsOEVBQXlFO0FBQ3pFLDZFQUFzRTtBQXdCdEU7O0dBRUc7QUFDSCxNQUFhLHFCQUFzQixTQUFRLHNCQUFTO0lBTWxELFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQUUsS0FBaUM7UUFDekUsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztRQUVqQixNQUFNLEVBQ0osWUFBWSxFQUNaLElBQUksRUFDSixRQUFRLEVBQ1IsZUFBZSxFQUNmLGlCQUFpQixFQUNqQixXQUFXLEVBQ1gsR0FBRyxPQUFPLEVBQ1gsR0FBRyxLQUFLLENBQUM7UUFFVixvQ0FBb0M7UUFDcEMsTUFBTSxjQUFjLEdBQUcsSUFBSSxxQkFBSyxDQUFDLElBQUksRUFBRSxXQUFXLEVBQUU7WUFDbEQsSUFBSSxFQUFFLFFBQVE7U0FDZixDQUFDLENBQUM7UUFDSCwwREFBMEQ7UUFDMUQsTUFBTSwyQkFBMkIsR0FBRyxHQUFHLGNBQWMsQ0FBQyxXQUFXLFdBQVcsQ0FBQztRQUU3RSxNQUFNLEtBQUssR0FBRyxtQkFBSyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUU3QixNQUFNLHFCQUFxQixHQUFHLEdBQUcsZ0JBQU0sQ0FBQyxjQUFjLENBQUMsS0FBSyxDQUFDO2FBQzFELEtBQUssQ0FBQyxHQUFHLENBQUM7YUFDVixJQUFJLENBQUMsR0FBRyxDQUFDLGFBQWEsQ0FBQztRQUMxQixNQUFNLGVBQWUsR0FBRyxJQUFJLGNBQUksQ0FBQyxJQUFJLEVBQUUsaUJBQWlCLEVBQUU7WUFDeEQsU0FBUyxFQUFFLElBQUksMEJBQWdCLENBQUMsc0JBQXNCLENBQUM7WUFDdkQsY0FBYyxFQUFFO2dCQUNkLElBQUksRUFBRSxJQUFJLHdCQUFjLENBQUM7b0JBQ3ZCLFVBQVUsRUFBRTt3QkFDVixJQUFJLHlCQUFlLENBQUM7NEJBQ2xCLE1BQU0sRUFBRSxnQkFBTSxDQUFDLEtBQUs7NEJBQ3BCLE9BQU8sRUFBRTtnQ0FDUCxxQkFBcUI7Z0NBQ3JCLHNCQUFzQjtnQ0FDdEIsbUJBQW1COzZCQUNwQjs0QkFDRCxTQUFTLEVBQUU7Z0NBQ1QsZ0JBQWdCLEtBQUssQ0FBQyxNQUFNLElBQUksS0FBSyxDQUFDLE9BQU8sMEJBQTBCLHFCQUFxQixFQUFFO2dDQUM5RixnQkFBZ0IsS0FBSyxDQUFDLE1BQU0sSUFBSSxLQUFLLENBQUMsT0FBTywwQkFBMEIscUJBQXFCLElBQUk7NkJBQ2pHO3lCQUNGLENBQUM7cUJBQ0g7aUJBQ0YsQ0FBQztnQkFDRixFQUFFLEVBQUUsSUFBSSx3QkFBYyxDQUFDO29CQUNyQixVQUFVLEVBQUU7d0JBQ1YsSUFBSSx5QkFBZSxDQUFDOzRCQUNsQixNQUFNLEVBQUUsZ0JBQU0sQ0FBQyxLQUFLOzRCQUNwQixPQUFPLEVBQUUsQ0FBQyxjQUFjLENBQUM7NEJBQ3pCLFNBQVMsRUFBRTtnQ0FDVCxjQUFjLENBQUMsTUFBTSxDQUFDLGFBQWEsQ0FBQyxjQUFjLENBQUMsV0FBVyxDQUFDOzZCQUNoRTt5QkFDRixDQUFDO3dCQUNGLElBQUkseUJBQWUsQ0FBQzs0QkFDbEIsTUFBTSxFQUFFLGdCQUFNLENBQUMsS0FBSzs0QkFDcEIsT0FBTyxFQUFFLENBQUMsY0FBYyxDQUFDOzRCQUN6QixTQUFTLEVBQUU7Z0NBQ1QsdUdBQXVHO2dDQUN2RywwQkFBMEI7Z0NBQzFCLGNBQWMsQ0FBQyxNQUFNLENBQUMsYUFBYSxDQUNqQyxHQUFHLDJCQUEyQixJQUFJLENBQ25DOzZCQUNGO3lCQUNGLENBQUM7cUJBQ0g7aUJBQ0YsQ0FBQzthQUNIO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsQ0FBQyxtQkFBbUIsRUFBRSx5Q0FBeUMsQ0FBQyxDQUFDLE9BQU8sQ0FDdEUsQ0FBQyxNQUFNLEVBQUUsRUFBRTtZQUNULHlCQUFlLENBQUMsdUJBQXVCLENBQ3JDLGVBQWUsRUFDZjtnQkFDRTtvQkFDRSxFQUFFLEVBQUUsTUFBTTtvQkFDVixNQUFNLEVBQ0osdUlBQXVJO29CQUN6SSxTQUFTLEVBQUU7d0JBQ1Q7NEJBQ0UsS0FBSyxFQUFFLDRCQUE0QixnQkFBTSxDQUFDLG1CQUFtQixDQUMzRCxLQUFLLENBQ04sSUFBSSxnQkFBTSxDQUFDLG9CQUFvQixDQUM5QixLQUFLLENBQ04sMEJBQTBCLHFCQUFxQixPQUFPO3lCQUN4RDtxQkFDRjtpQkFDRjtnQkFDRDtvQkFDRSxFQUFFLEVBQUUsTUFBTTtvQkFDVixNQUFNLEVBQ0osaUxBQWlMO29CQUNuTCxTQUFTLEVBQUU7d0JBQ1Q7NEJBQ0UsS0FBSyxFQUFFLG1CQUFtQixnQkFBTSxDQUFDLHNCQUFzQixDQUNyRCxLQUFLLENBQ04sVUFBVSwyQkFBMkIsT0FBTzt5QkFDOUM7cUJBQ0Y7aUJBQ0Y7YUFDRixFQUNELElBQUksQ0FDTCxDQUFDO1FBQ0osQ0FBQyxDQUNGLENBQUM7UUFFRix5R0FBeUc7UUFDekcsTUFBTSxXQUFXLEdBQUcsSUFBSSxxQkFBYyxDQUFDLElBQUksRUFBRSxhQUFhLEVBQUU7WUFDMUQsT0FBTyxFQUFFLGVBQWU7WUFDeEIsT0FBTyxFQUFFLG9CQUFPLENBQUMsV0FBVztZQUM1QixJQUFJLEVBQUUsaUJBQUksQ0FBQyxTQUFTLENBQ2xCLElBQUksQ0FBQyxJQUFJLENBQUMsU0FBUyxFQUFFLGdEQUFnRCxDQUFDLENBQ3ZFO1lBQ0QsT0FBTyxFQUFFLHNCQUFRLENBQUMsT0FBTyxDQUFDLEVBQUUsQ0FBQztZQUM3QixJQUFJLEVBQUUsZUFBZTtZQUNyQixZQUFZLEVBQUUscUJBQXFCO1NBQ3BDLENBQUMsQ0FBQztRQUVILE1BQU0sb0JBQW9CLEdBQUcsR0FBRyxxQkFBcUIsV0FBVyxDQUFDO1FBQ2pFLE1BQU0sWUFBWSxHQUFHLElBQUksY0FBSSxDQUFDLElBQUksRUFBRSx5QkFBeUIsRUFBRTtZQUM3RCxTQUFTLEVBQUUsSUFBSSwwQkFBZ0IsQ0FBQyxzQkFBc0IsQ0FBQztZQUN2RCxjQUFjLEVBQUU7Z0JBQ2QsSUFBSSxFQUFFLElBQUksd0JBQWMsQ0FBQztvQkFDdkIsVUFBVSxFQUFFO3dCQUNWLElBQUkseUJBQWUsQ0FBQzs0QkFDbEIsTUFBTSxFQUFFLGdCQUFNLENBQUMsS0FBSzs0QkFDcEIsT0FBTyxFQUFFO2dDQUNQLHFCQUFxQjtnQ0FDckIsc0JBQXNCO2dDQUN0QixtQkFBbUI7NkJBQ3BCOzRCQUNELFNBQVMsRUFBRTtnQ0FDVCxnQkFBZ0IsS0FBSyxDQUFDLE1BQU0sSUFBSSxLQUFLLENBQUMsT0FBTywwQkFBMEIsb0JBQW9CLEVBQUU7Z0NBQzdGLGdCQUFnQixLQUFLLENBQUMsTUFBTSxJQUFJLEtBQUssQ0FBQyxPQUFPLDBCQUEwQixvQkFBb0IsSUFBSTs2QkFDaEc7eUJBQ0YsQ0FBQztxQkFDSDtpQkFDRixDQUFDO2FBQ0g7U0FDRixDQUFDLENBQUM7UUFFSCxNQUFNLFFBQVEsR0FBRyxJQUFJLDJCQUFRLENBQUMsSUFBSSxFQUFFLHFCQUFxQixFQUFFO1lBQ3pELGNBQWMsRUFBRSxXQUFXO1lBQzNCLElBQUksRUFBRSxZQUFZO1lBQ2xCLG9CQUFvQjtTQUNyQixDQUFDLENBQUM7UUFFSCxDQUFDLG1CQUFtQixFQUFFLHlDQUF5QyxDQUFDLENBQUMsT0FBTyxDQUN0RSxDQUFDLE1BQU0sRUFBRSxFQUFFO1lBQ1QseUJBQWUsQ0FBQyx1QkFBdUIsQ0FDckMsWUFBWSxFQUNaO2dCQUNFO29CQUNFLEVBQUUsRUFBRSxNQUFNO29CQUNWLE1BQU0sRUFDSix1SUFBdUk7aUJBQzFJO2FBQ0YsRUFDRCxJQUFJLENBQ0wsQ0FBQztRQUNKLENBQUMsQ0FDRixDQUFDO1FBRUYsQ0FBQyxpQkFBaUIsRUFBRSxvQ0FBb0MsQ0FBQyxDQUFDLE9BQU8sQ0FDL0QsQ0FBQyxNQUFNLEVBQUUsRUFBRTtZQUNULHlCQUFlLENBQUMsdUJBQXVCLENBQ3JDLFFBQVEsRUFDUjtnQkFDRTtvQkFDRSxFQUFFLEVBQUUsTUFBTTtvQkFDVixNQUFNLEVBQ0osbUdBQW1HO2lCQUN0RzthQUNGLEVBQ0QsSUFBSSxDQUNMLENBQUM7UUFDSixDQUFDLENBQ0YsQ0FBQztRQUVGLE1BQU0sa0JBQWtCLEdBQTBCO1lBQ2hELDBCQUEwQixFQUN4QixpREFBOEIsQ0FBQyxpQkFBaUIsQ0FBQztZQUNuRCxZQUFZLEVBQUUsTUFBTSxDQUFDLFdBQVcsQ0FDOUIsTUFBTSxDQUFDLE9BQU8sQ0FBQyxZQUFZLENBQUMsQ0FBQyxHQUFHLENBQzlCLENBQUMsQ0FBQyxXQUFXLEVBQUUsRUFBRSxVQUFVLEVBQUUsV0FBVyxFQUFFLENBQUMsRUFBRSxFQUFFLENBQUM7Z0JBQzlDLFdBQVc7Z0JBQ1g7b0JBQ0UsV0FBVyxFQUFFLFdBQVcsQ0FBQyxNQUFNLENBQUM7d0JBQzlCLFdBQVc7d0JBQ1gsS0FBSyxFQUFFLElBQUk7d0JBQ1gsR0FBRyxlQUFlLENBQUMsV0FBVyxDQUFDO3FCQUNoQyxDQUFDO29CQUNGLGdCQUFnQixFQUFFLGlEQUE4QixDQUFDLFVBQVUsQ0FBQztpQkFDN0Q7YUFDRixDQUNGLENBQ0Y7WUFDRCxlQUFlLEVBQUUseUNBQXNCLENBQ3JDLElBQUksRUFDSixZQUFZLEVBQ1osaUJBQWlCLENBQ2xCO1lBQ0QsV0FBVyxFQUFFLFdBQVcsSUFBSTtnQkFDMUIsWUFBWSxFQUFFLFdBQVcsQ0FBQyxZQUFZLElBQUkscUJBQUksQ0FBQyxlQUFlO2dCQUM5RCxZQUFZLEVBQUUsV0FBVyxDQUFDLFlBQVksSUFBSSxxQkFBSSxDQUFDLFdBQVc7Z0JBQzFELFlBQVksRUFBRSxXQUFXLENBQUMsWUFBWTtnQkFDdEMsVUFBVSxFQUFFLFdBQVcsQ0FBQyxVQUFVLElBQUksR0FBRzthQUMxQztZQUNELGVBQWU7U0FDaEIsQ0FBQztRQUVGLGdIQUFnSDtRQUNoSCxpR0FBaUc7UUFDakcsTUFBTSxZQUFZLEdBQUcsNkJBQWMsQ0FBQyxJQUFJLEVBQUUsa0JBQWtCLENBQUMsQ0FBQztRQUU5RCxNQUFNLHNDQUFzQyxHQUMxQztZQUNFLGlCQUFpQixFQUFFO2dCQUNqQixNQUFNLEVBQUUsY0FBYyxDQUFDLE1BQU0sQ0FBQyxVQUFVO2dCQUN4QyxHQUFHLEVBQUUsY0FBYyxDQUFDLFdBQVc7YUFDaEM7WUFDRCxrQkFBa0IsRUFBRTtnQkFDbEIsTUFBTSxFQUFFLGNBQWMsQ0FBQyxNQUFNLENBQUMsVUFBVTtnQkFDeEMsR0FBRyxFQUFFLDJCQUEyQjthQUNqQztZQUNELEdBQUcsa0JBQWtCO1NBQ3RCLENBQUM7UUFFSixNQUFNLHlCQUF5QixHQUFHLElBQUksNEJBQWMsQ0FDbEQsSUFBSSxFQUNKLDJCQUEyQixFQUMzQjtZQUNFLFlBQVksRUFBRSxRQUFRLENBQUMsWUFBWTtZQUNuQyxVQUFVLEVBQUUsc0NBQXNDO1NBQ25ELENBQ0YsQ0FBQztRQUVGLGtIQUFrSDtRQUNsSCxxQ0FBcUM7UUFDckMsSUFBSSxDQUFDLEdBQUcsR0FBRyxJQUFJLDRCQUFXLENBQUMsSUFBSSxFQUFFLEVBQUUsRUFBRTtZQUNuQyxhQUFhLEVBQUUsOEJBQWEsQ0FBQyxVQUFVLENBQ3JDLGNBQWMsQ0FBQyxNQUFNLEVBQ3JCLHlCQUF5QixDQUFDLFlBQVksQ0FBQyxlQUFlLENBQUMsQ0FDeEQ7WUFDRCxhQUFhLEVBQUU7Z0JBQ2Isb0JBQW9CLEVBQUUsSUFBSSx1Q0FBc0IsQ0FDOUMsSUFBSSxtQkFBUSxDQUFDLEtBQUssRUFBRSxZQUFZLENBQUMsQ0FDbEM7Z0JBQ0QsZUFBZSxFQUFFLGdDQUFlLENBQUMsR0FBRyxFQUFFO2dCQUN0QyxZQUFZLEVBQUUsbUNBQWtCLENBQUMsSUFBSTthQUN0QztZQUNELEdBQUcsT0FBTztTQUNYLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxHQUFHLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyx5QkFBeUIsQ0FBQyxDQUFDO1FBRXZELG9IQUFvSDtRQUNwSCwrR0FBK0c7UUFDL0csb0hBQW9IO1FBQ3BILElBQUksQ0FBQyxHQUFHLENBQUMsZ0JBQWdCLEVBQUUsY0FBYyxDQUFDLFlBQVksQ0FBQyxDQUFDO1FBRXhELDBEQUEwRDtRQUMxRCxNQUFNLENBQUMsSUFBSSxDQUFDLFlBQVksQ0FBQyxDQUFDLE9BQU8sQ0FBQyxDQUFDLFdBQVcsRUFBRSxFQUFFO1lBQ2hELFlBQVksQ0FBQyxXQUFXLENBQUMsQ0FBQyxXQUFXLENBQUMsS0FBSyxDQUFDO2dCQUMxQyxXQUFXO2dCQUNYLEtBQUssRUFBRSxJQUFJO2dCQUNYLEdBQUcsRUFBRSxJQUFJLENBQUMsR0FBRztnQkFDYixHQUFHLGVBQWUsQ0FBQyxXQUFXLENBQUM7YUFDaEMsQ0FBQyxDQUFDO1FBQ0wsQ0FBQyxDQUFDLENBQUM7UUFFSCxnRkFBZ0Y7UUFDaEYsaURBQXNCLENBQUMsS0FBSyxDQUFDLENBQUMsT0FBTyxDQUFDLENBQUMsRUFBRSxLQUFLLEVBQUUsUUFBUSxFQUFFLE1BQU0sRUFBRSxFQUFFLEVBQUU7WUFDcEUsSUFBSSwwQkFBYSxDQUFDLElBQUksRUFBRSxvQkFBb0IsS0FBSyxFQUFFLEVBQUU7Z0JBQ25ELE1BQU0sRUFBRSx1QkFBdUI7Z0JBQy9CLFNBQVMsRUFBRSwwQkFBMEI7Z0JBQ3JDLFlBQVksRUFBRSxNQUFNLENBQUMsV0FBVztnQkFDaEMsU0FBUyxFQUFFLEtBQUssQ0FBQyxTQUFTLENBQUM7b0JBQ3pCLE9BQU8sRUFBRSxhQUFhO29CQUN0QixRQUFRLEVBQUUsSUFBSSxDQUFDLEdBQUcsQ0FBQyxTQUFTO29CQUM1QixZQUFZLEVBQUUsS0FBSztpQkFDcEIsQ0FBQzthQUNILENBQUMsQ0FBQztRQUNMLENBQUMsQ0FBQyxDQUFDO1FBRUgsbURBQW1EO1FBQ25ELElBQUksQ0FBQyxLQUFLLENBQUMsYUFBYSxFQUFFLE9BQU8sRUFBRTtZQUNqQyxNQUFNLEdBQUcsR0FBRyxJQUFJLCtDQUFvQixDQUFDLElBQUksRUFBRSxHQUFHLEVBQUUsTUFBTSxFQUFFO2dCQUN0RCxHQUFHLEtBQUssQ0FBQyxhQUFhO2dCQUN0QixxQkFBcUIsRUFBRSxJQUFJLENBQUMsR0FBRyxDQUFDLGVBQWUsQ0FBQyxRQUFRO2FBQ3pELENBQUMsQ0FBQztZQUVILElBQUksQ0FBQyxNQUFNLEdBQUcsR0FBRyxDQUFDLE1BQU0sQ0FBQztZQUN6QixJQUFJLENBQUMsS0FBSyxHQUFHLEdBQUcsQ0FBQyxLQUFLLENBQUM7WUFDdkIsSUFBSSxDQUFDLGlCQUFpQixHQUFHLEdBQUcsQ0FBQyxpQkFBaUIsQ0FBQztTQUNoRDtRQUVELENBQUMsbUJBQW1CLEVBQUUscUNBQXFDLENBQUMsQ0FBQyxPQUFPLENBQ2xFLENBQUMsTUFBTSxFQUFFLEVBQUU7WUFDVCx5QkFBZSxDQUFDLHVCQUF1QixDQUNyQyxJQUFJLEVBQ0o7Z0JBQ0U7b0JBQ0UsRUFBRSxFQUFFLE1BQU07b0JBQ1YsTUFBTSxFQUNKLDBFQUEwRTtvQkFDNUUsU0FBUyxFQUFFO3dCQUNUOzRCQUNFLEtBQUssRUFBRSxpQkFBaUIsZ0JBQU0sQ0FBQyxzQkFBc0IsQ0FDbkQsS0FBSyxDQUNOLHVFQUF1RTt5QkFDekU7cUJBQ0Y7aUJBQ0Y7YUFDRixFQUNELElBQUksQ0FDTCxDQUFDO1FBQ0osQ0FBQyxDQUNGLENBQUM7UUFFRixDQUFDLG9CQUFvQixFQUFFLHVDQUF1QyxDQUFDLENBQUMsT0FBTyxDQUNyRSxDQUFDLE1BQU0sRUFBRSxFQUFFO1lBQ1QseUJBQWUsQ0FBQyx1QkFBdUIsQ0FDckMsSUFBSSxFQUNKO2dCQUNFO29CQUNFLEVBQUUsRUFBRSxNQUFNO29CQUNWLE1BQU0sRUFDSixnRUFBZ0U7aUJBQ25FO2FBQ0YsRUFDRCxJQUFJLENBQ0wsQ0FBQztRQUNKLENBQUMsQ0FDRixDQUFDO0lBQ0osQ0FBQzs7QUFyVkgsc0RBc1ZDIiwic291cmNlc0NvbnRlbnQiOlsiLyohIENvcHlyaWdodCBbQW1hem9uLmNvbV0oaHR0cDovL2FtYXpvbi5jb20vKSwgSW5jLiBvciBpdHMgYWZmaWxpYXRlcy4gQWxsIFJpZ2h0cyBSZXNlcnZlZC5cblNQRFgtTGljZW5zZS1JZGVudGlmaWVyOiBBcGFjaGUtMi4wICovXG5pbXBvcnQgKiBhcyBwYXRoIGZyb20gXCJwYXRoXCI7XG5pbXBvcnQgeyBQREtOYWcgfSBmcm9tIFwiQGF3cy1wcm90b3R5cGluZy1zZGsvcGRrLW5hZ1wiO1xuaW1wb3J0IHsgQ3VzdG9tUmVzb3VyY2UsIER1cmF0aW9uLCBTdGFjayB9IGZyb20gXCJhd3MtY2RrLWxpYlwiO1xuaW1wb3J0IHtcbiAgQWNjZXNzTG9nRm9ybWF0LFxuICBBcGlEZWZpbml0aW9uLFxuICBDb3JzLFxuICBMb2dHcm91cExvZ0Rlc3RpbmF0aW9uLFxuICBNZXRob2RMb2dnaW5nTGV2ZWwsXG4gIFJlc3RBcGlCYXNlUHJvcHMsXG4gIFNwZWNSZXN0QXBpLFxufSBmcm9tIFwiYXdzLWNkay1saWIvYXdzLWFwaWdhdGV3YXlcIjtcbmltcG9ydCB7XG4gIEVmZmVjdCxcbiAgUG9saWN5RG9jdW1lbnQsXG4gIFBvbGljeVN0YXRlbWVudCxcbiAgUm9sZSxcbiAgU2VydmljZVByaW5jaXBhbCxcbn0gZnJvbSBcImF3cy1jZGstbGliL2F3cy1pYW1cIjtcbmltcG9ydCB7XG4gIENmblBlcm1pc3Npb24sXG4gIENvZGUsXG4gIEZ1bmN0aW9uIGFzIExhbWJkYUZ1bmN0aW9uLFxuICBSdW50aW1lLFxufSBmcm9tIFwiYXdzLWNkay1saWIvYXdzLWxhbWJkYVwiO1xuaW1wb3J0IHsgTG9nR3JvdXAgfSBmcm9tIFwiYXdzLWNkay1saWIvYXdzLWxvZ3NcIjtcbmltcG9ydCB7IEFzc2V0IH0gZnJvbSBcImF3cy1jZGstbGliL2F3cy1zMy1hc3NldHNcIjtcbmltcG9ydCB7XG4gIENmbklQU2V0LFxuICBDZm5XZWJBQ0wsXG4gIENmbldlYkFDTEFzc29jaWF0aW9uLFxufSBmcm9tIFwiYXdzLWNkay1saWIvYXdzLXdhZnYyXCI7XG5pbXBvcnQgeyBQcm92aWRlciB9IGZyb20gXCJhd3MtY2RrLWxpYi9jdXN0b20tcmVzb3VyY2VzXCI7XG5pbXBvcnQgeyBOYWdTdXBwcmVzc2lvbnMgfSBmcm9tIFwiY2RrLW5hZ1wiO1xuaW1wb3J0IHsgQ29uc3RydWN0IH0gZnJvbSBcImNvbnN0cnVjdHNcIjtcbmltcG9ydCB7IFByZXBhcmVBcGlTcGVjQ3VzdG9tUmVzb3VyY2VQcm9wZXJ0aWVzIH0gZnJvbSBcIi4vcHJlcGFyZS1zcGVjLWV2ZW50LWhhbmRsZXJcIjtcbmltcG9ydCB7XG4gIHByZXBhcmVBcGlTcGVjLFxuICBQcmVwYXJlQXBpU3BlY09wdGlvbnMsXG59IGZyb20gXCIuL3ByZXBhcmUtc3BlYy1ldmVudC1oYW5kbGVyL3ByZXBhcmUtc3BlY1wiO1xuaW1wb3J0IHsgT3BlbkFwaU9wdGlvbnMgfSBmcm9tIFwiLi9zcGVjXCI7XG5pbXBvcnQge1xuICBwcmVwYXJlU2VjdXJpdHlTY2hlbWVzLFxuICBzZXJpYWxpemVBc0F1dGhvcml6ZXJSZWZlcmVuY2UsXG59IGZyb20gXCIuL3NwZWMvYXBpLWdhdGV3YXktYXV0aFwiO1xuaW1wb3J0IHsgZ2V0QXV0aG9yaXplckZ1bmN0aW9ucyB9IGZyb20gXCIuL3NwZWMvYXBpLWdhdGV3YXktaW50ZWdyYXRpb25zXCI7XG5pbXBvcnQgeyBPcGVuQXBpR2F0ZXdheVdlYkFjbCB9IGZyb20gXCIuL3dhZi9vcGVuLWFwaS1nYXRld2F5LXdlYi1hY2xcIjtcbmltcG9ydCB7IE9wZW5BcGlHYXRld2F5V2ViQWNsT3B0aW9ucyB9IGZyb20gXCIuL3dhZi90eXBlc1wiO1xuXG4vKipcbiAqIENvbmZpZ3VyYXRpb24gZm9yIHRoZSBPcGVuQXBpR2F0ZXdheVJlc3RBcGkgY29uc3RydWN0XG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgT3BlbkFwaUdhdGV3YXlSZXN0QXBpUHJvcHNcbiAgZXh0ZW5kcyBSZXN0QXBpQmFzZVByb3BzLFxuICAgIE9wZW5BcGlPcHRpb25zIHtcbiAgLyoqXG4gICAqIFRoZSBwYXJzZWQgT3BlbkFQSSBzcGVjaWZpY2F0aW9uXG4gICAqL1xuICByZWFkb25seSBzcGVjOiBhbnk7IC8vIFR5cGUgaXMgT3BlbkFQSVYzLkRvY3VtZW50IC0gaG93ZXZlciBub3QgdHJhbnNwaWxlYWJsZSB2aWEganNpaSwgc28gd2UgdXNlIGFueS5cbiAgLyoqXG4gICAqIFBhdGggdG8gdGhlIEpTT04gb3BlbiBhcGkgc3BlY1xuICAgKi9cbiAgcmVhZG9ubHkgc3BlY1BhdGg6IHN0cmluZztcbiAgLyoqXG4gICAqIE9wdGlvbnMgZm9yIHRoZSBBV1MgV0FGIHYyIFdlYkFDTCBhc3NvY2lhdGVkIHdpdGggdGhlIGFwaS4gQnkgZGVmYXVsdCwgYSBXZWIgQUNMIHdpdGggdGhlIEFXUyBkZWZhdWx0IG1hbmFnZWRcbiAgICogcnVsZSBzZXQgd2lsbCBiZSBhc3NvY2lhdGVkIHdpdGggdGhlIEFQSS4gVGhlc2Ugb3B0aW9ucyBtYXkgZGlzYWJsZSBvciBvdmVycmlkZSB0aGUgZGVmYXVsdHMuXG4gICAqL1xuICByZWFkb25seSB3ZWJBY2xPcHRpb25zPzogT3BlbkFwaUdhdGV3YXlXZWJBY2xPcHRpb25zO1xufVxuXG4vKipcbiAqIEEgY29uc3RydWN0IGZvciBjcmVhdGluZyBhbiBhcGkgZ2F0ZXdheSByZXN0IGFwaSBiYXNlZCBvbiB0aGUgZGVmaW5pdGlvbiBpbiB0aGUgT3BlbkFQSSBzcGVjLlxuICovXG5leHBvcnQgY2xhc3MgT3BlbkFwaUdhdGV3YXlSZXN0QXBpIGV4dGVuZHMgQ29uc3RydWN0IHtcbiAgcHVibGljIHJlYWRvbmx5IGFwaTogU3BlY1Jlc3RBcGk7XG4gIHJlYWRvbmx5IHdlYkFjbD86IENmbldlYkFDTDtcbiAgcmVhZG9ubHkgaXBTZXQ/OiBDZm5JUFNldDtcbiAgcmVhZG9ubHkgd2ViQWNsQXNzb2NpYXRpb24/OiBDZm5XZWJBQ0xBc3NvY2lhdGlvbjtcblxuICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogT3BlbkFwaUdhdGV3YXlSZXN0QXBpUHJvcHMpIHtcbiAgICBzdXBlcihzY29wZSwgaWQpO1xuXG4gICAgY29uc3Qge1xuICAgICAgaW50ZWdyYXRpb25zLFxuICAgICAgc3BlYyxcbiAgICAgIHNwZWNQYXRoLFxuICAgICAgb3BlcmF0aW9uTG9va3VwLFxuICAgICAgZGVmYXVsdEF1dGhvcml6ZXIsXG4gICAgICBjb3JzT3B0aW9ucyxcbiAgICAgIC4uLm9wdGlvbnNcbiAgICB9ID0gcHJvcHM7XG5cbiAgICAvLyBVcGxvYWQgdGhlIHNwZWMgdG8gczMgYXMgYW4gYXNzZXRcbiAgICBjb25zdCBpbnB1dFNwZWNBc3NldCA9IG5ldyBBc3NldCh0aGlzLCBcIklucHV0U3BlY1wiLCB7XG4gICAgICBwYXRoOiBzcGVjUGF0aCxcbiAgICB9KTtcbiAgICAvLyBXZSdsbCBvdXRwdXQgdGhlIHByZXBhcmVkIHNwZWMgaW4gdGhlIHNhbWUgYXNzZXQgYnVja2V0XG4gICAgY29uc3QgcHJlcGFyZWRTcGVjT3V0cHV0S2V5UHJlZml4ID0gYCR7aW5wdXRTcGVjQXNzZXQuczNPYmplY3RLZXl9LXByZXBhcmVkYDtcblxuICAgIGNvbnN0IHN0YWNrID0gU3RhY2sub2YodGhpcyk7XG5cbiAgICBjb25zdCBwcmVwYXJlU3BlY0xhbWJkYU5hbWUgPSBgJHtQREtOYWcuZ2V0U3RhY2tQcmVmaXgoc3RhY2spXG4gICAgICAuc3BsaXQoXCIvXCIpXG4gICAgICAuam9pbihcIi1cIil9UHJlcGFyZVNwZWNgO1xuICAgIGNvbnN0IHByZXBhcmVTcGVjUm9sZSA9IG5ldyBSb2xlKHRoaXMsIFwiUHJlcGFyZVNwZWNSb2xlXCIsIHtcbiAgICAgIGFzc3VtZWRCeTogbmV3IFNlcnZpY2VQcmluY2lwYWwoXCJsYW1iZGEuYW1hem9uYXdzLmNvbVwiKSxcbiAgICAgIGlubGluZVBvbGljaWVzOiB7XG4gICAgICAgIGxvZ3M6IG5ldyBQb2xpY3lEb2N1bWVudCh7XG4gICAgICAgICAgc3RhdGVtZW50czogW1xuICAgICAgICAgICAgbmV3IFBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgICAgICAgIGVmZmVjdDogRWZmZWN0LkFMTE9XLFxuICAgICAgICAgICAgICBhY3Rpb25zOiBbXG4gICAgICAgICAgICAgICAgXCJsb2dzOkNyZWF0ZUxvZ0dyb3VwXCIsXG4gICAgICAgICAgICAgICAgXCJsb2dzOkNyZWF0ZUxvZ1N0cmVhbVwiLFxuICAgICAgICAgICAgICAgIFwibG9nczpQdXRMb2dFdmVudHNcIixcbiAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgICAgcmVzb3VyY2VzOiBbXG4gICAgICAgICAgICAgICAgYGFybjphd3M6bG9nczoke3N0YWNrLnJlZ2lvbn06JHtzdGFjay5hY2NvdW50fTpsb2ctZ3JvdXA6L2F3cy9sYW1iZGEvJHtwcmVwYXJlU3BlY0xhbWJkYU5hbWV9YCxcbiAgICAgICAgICAgICAgICBgYXJuOmF3czpsb2dzOiR7c3RhY2sucmVnaW9ufToke3N0YWNrLmFjY291bnR9OmxvZy1ncm91cDovYXdzL2xhbWJkYS8ke3ByZXBhcmVTcGVjTGFtYmRhTmFtZX06KmAsXG4gICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICB9KSxcbiAgICAgICAgICBdLFxuICAgICAgICB9KSxcbiAgICAgICAgczM6IG5ldyBQb2xpY3lEb2N1bWVudCh7XG4gICAgICAgICAgc3RhdGVtZW50czogW1xuICAgICAgICAgICAgbmV3IFBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgICAgICAgIGVmZmVjdDogRWZmZWN0LkFMTE9XLFxuICAgICAgICAgICAgICBhY3Rpb25zOiBbXCJzMzpnZXRPYmplY3RcIl0sXG4gICAgICAgICAgICAgIHJlc291cmNlczogW1xuICAgICAgICAgICAgICAgIGlucHV0U3BlY0Fzc2V0LmJ1Y2tldC5hcm5Gb3JPYmplY3RzKGlucHV0U3BlY0Fzc2V0LnMzT2JqZWN0S2V5KSxcbiAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgIH0pLFxuICAgICAgICAgICAgbmV3IFBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgICAgICAgIGVmZmVjdDogRWZmZWN0LkFMTE9XLFxuICAgICAgICAgICAgICBhY3Rpb25zOiBbXCJzMzpwdXRPYmplY3RcIl0sXG4gICAgICAgICAgICAgIHJlc291cmNlczogW1xuICAgICAgICAgICAgICAgIC8vIFRoZSBvdXRwdXQgZmlsZSB3aWxsIGluY2x1ZGUgYSBoYXNoIG9mIHRoZSBwcmVwYXJlZCBzcGVjLCB3aGljaCBpcyBub3Qga25vd24gdW50aWwgZGVwbG95IHRpbWUgc2luY2VcbiAgICAgICAgICAgICAgICAvLyB0b2tlbnMgbXVzdCBiZSByZXNvbHZlZFxuICAgICAgICAgICAgICAgIGlucHV0U3BlY0Fzc2V0LmJ1Y2tldC5hcm5Gb3JPYmplY3RzKFxuICAgICAgICAgICAgICAgICAgYCR7cHJlcGFyZWRTcGVjT3V0cHV0S2V5UHJlZml4fS8qYFxuICAgICAgICAgICAgICAgICksXG4gICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICB9KSxcbiAgICAgICAgICBdLFxuICAgICAgICB9KSxcbiAgICAgIH0sXG4gICAgfSk7XG5cbiAgICBbXCJBd3NTb2x1dGlvbnMtSUFNNVwiLCBcIkF3c1Byb3RvdHlwaW5nLUlBTU5vV2lsZGNhcmRQZXJtaXNzaW9uc1wiXS5mb3JFYWNoKFxuICAgICAgKFJ1bGVJZCkgPT4ge1xuICAgICAgICBOYWdTdXBwcmVzc2lvbnMuYWRkUmVzb3VyY2VTdXBwcmVzc2lvbnMoXG4gICAgICAgICAgcHJlcGFyZVNwZWNSb2xlLFxuICAgICAgICAgIFtcbiAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgaWQ6IFJ1bGVJZCxcbiAgICAgICAgICAgICAgcmVhc29uOlxuICAgICAgICAgICAgICAgIFwiQ2xvdWR3YXRjaCByZXNvdXJjZXMgaGF2ZSBiZWVuIHNjb3BlZCBkb3duIHRvIHRoZSBMb2dHcm91cCBsZXZlbCwgaG93ZXZlciAqIGlzIHN0aWxsIG5lZWRlZCBhcyBzdHJlYW0gbmFtZXMgYXJlIGNyZWF0ZWQganVzdCBpbiB0aW1lLlwiLFxuICAgICAgICAgICAgICBhcHBsaWVzVG86IFtcbiAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICByZWdleDogYC9eUmVzb3VyY2U6OmFybjphd3M6bG9nczoke1BES05hZy5nZXRTdGFja1JlZ2lvblJlZ2V4KFxuICAgICAgICAgICAgICAgICAgICBzdGFja1xuICAgICAgICAgICAgICAgICAgKX06JHtQREtOYWcuZ2V0U3RhY2tBY2NvdW50UmVnZXgoXG4gICAgICAgICAgICAgICAgICAgIHN0YWNrXG4gICAgICAgICAgICAgICAgICApfTpsb2ctZ3JvdXA6L2F3cy9sYW1iZGEvJHtwcmVwYXJlU3BlY0xhbWJkYU5hbWV9OlxcKi9nYCxcbiAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgaWQ6IFJ1bGVJZCxcbiAgICAgICAgICAgICAgcmVhc29uOlxuICAgICAgICAgICAgICAgIFwiUzMgcmVzb3VyY2VzIGhhdmUgYmVlbiBzY29wZWQgZG93biB0byB0aGUgYXBwcm9wcmlhdGUgcHJlZml4IGluIHRoZSBDREsgYXNzZXQgYnVja2V0LCBob3dldmVyICogaXMgc3RpbGwgbmVlZGVkIGFzIHNpbmNlIHRoZSBwcmVwYXJlZCBzcGVjIGhhc2ggaXMgbm90IGtub3duIHVudGlsIGRlcGxveSB0aW1lLlwiLFxuICAgICAgICAgICAgICBhcHBsaWVzVG86IFtcbiAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICByZWdleDogYC9eUmVzb3VyY2U6OmFybjoke1BES05hZy5nZXRTdGFja1BhcnRpdGlvblJlZ2V4KFxuICAgICAgICAgICAgICAgICAgICBzdGFja1xuICAgICAgICAgICAgICAgICAgKX06czM6LiovJHtwcmVwYXJlZFNwZWNPdXRwdXRLZXlQcmVmaXh9L1xcKi9nYCxcbiAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICBdLFxuICAgICAgICAgIHRydWVcbiAgICAgICAgKTtcbiAgICAgIH1cbiAgICApO1xuXG4gICAgLy8gQ3JlYXRlIGEgY3VzdG9tIHJlc291cmNlIGZvciBwcmVwYXJpbmcgdGhlIHNwZWMgZm9yIGRlcGxveW1lbnQgKGFkZGluZyBpbnRlZ3JhdGlvbnMsIGF1dGhvcml6ZXJzLCBldGMpXG4gICAgY29uc3QgcHJlcGFyZVNwZWMgPSBuZXcgTGFtYmRhRnVuY3Rpb24odGhpcywgXCJQcmVwYXJlU3BlY1wiLCB7XG4gICAgICBoYW5kbGVyOiBcImluZGV4LmhhbmRsZXJcIixcbiAgICAgIHJ1bnRpbWU6IFJ1bnRpbWUuTk9ERUpTXzE4X1gsXG4gICAgICBjb2RlOiBDb2RlLmZyb21Bc3NldChcbiAgICAgICAgcGF0aC5qb2luKF9fZGlybmFtZSwgXCIuLi8uLi9saWIvY29uc3RydWN0L3ByZXBhcmUtc3BlYy1ldmVudC1oYW5kbGVyXCIpXG4gICAgICApLFxuICAgICAgdGltZW91dDogRHVyYXRpb24uc2Vjb25kcygzMCksXG4gICAgICByb2xlOiBwcmVwYXJlU3BlY1JvbGUsXG4gICAgICBmdW5jdGlvbk5hbWU6IHByZXBhcmVTcGVjTGFtYmRhTmFtZSxcbiAgICB9KTtcblxuICAgIGNvbnN0IHByb3ZpZGVyRnVuY3Rpb25OYW1lID0gYCR7cHJlcGFyZVNwZWNMYW1iZGFOYW1lfS1Qcm92aWRlcmA7XG4gICAgY29uc3QgcHJvdmlkZXJSb2xlID0gbmV3IFJvbGUodGhpcywgXCJQcmVwYXJlU3BlY1Byb3ZpZGVyUm9sZVwiLCB7XG4gICAgICBhc3N1bWVkQnk6IG5ldyBTZXJ2aWNlUHJpbmNpcGFsKFwibGFtYmRhLmFtYXpvbmF3cy5jb21cIiksXG4gICAgICBpbmxpbmVQb2xpY2llczoge1xuICAgICAgICBsb2dzOiBuZXcgUG9saWN5RG9jdW1lbnQoe1xuICAgICAgICAgIHN0YXRlbWVudHM6IFtcbiAgICAgICAgICAgIG5ldyBQb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgICAgICAgICBlZmZlY3Q6IEVmZmVjdC5BTExPVyxcbiAgICAgICAgICAgICAgYWN0aW9uczogW1xuICAgICAgICAgICAgICAgIFwibG9nczpDcmVhdGVMb2dHcm91cFwiLFxuICAgICAgICAgICAgICAgIFwibG9nczpDcmVhdGVMb2dTdHJlYW1cIixcbiAgICAgICAgICAgICAgICBcImxvZ3M6UHV0TG9nRXZlbnRzXCIsXG4gICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICAgIHJlc291cmNlczogW1xuICAgICAgICAgICAgICAgIGBhcm46YXdzOmxvZ3M6JHtzdGFjay5yZWdpb259OiR7c3RhY2suYWNjb3VudH06bG9nLWdyb3VwOi9hd3MvbGFtYmRhLyR7cHJvdmlkZXJGdW5jdGlvbk5hbWV9YCxcbiAgICAgICAgICAgICAgICBgYXJuOmF3czpsb2dzOiR7c3RhY2sucmVnaW9ufToke3N0YWNrLmFjY291bnR9OmxvZy1ncm91cDovYXdzL2xhbWJkYS8ke3Byb3ZpZGVyRnVuY3Rpb25OYW1lfToqYCxcbiAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgIH0pLFxuICAgICAgICAgIF0sXG4gICAgICAgIH0pLFxuICAgICAgfSxcbiAgICB9KTtcblxuICAgIGNvbnN0IHByb3ZpZGVyID0gbmV3IFByb3ZpZGVyKHRoaXMsIFwiUHJlcGFyZVNwZWNQcm92aWRlclwiLCB7XG4gICAgICBvbkV2ZW50SGFuZGxlcjogcHJlcGFyZVNwZWMsXG4gICAgICByb2xlOiBwcm92aWRlclJvbGUsXG4gICAgICBwcm92aWRlckZ1bmN0aW9uTmFtZSxcbiAgICB9KTtcblxuICAgIFtcIkF3c1NvbHV0aW9ucy1JQU01XCIsIFwiQXdzUHJvdG90eXBpbmctSUFNTm9XaWxkY2FyZFBlcm1pc3Npb25zXCJdLmZvckVhY2goXG4gICAgICAoUnVsZUlkKSA9PiB7XG4gICAgICAgIE5hZ1N1cHByZXNzaW9ucy5hZGRSZXNvdXJjZVN1cHByZXNzaW9ucyhcbiAgICAgICAgICBwcm92aWRlclJvbGUsXG4gICAgICAgICAgW1xuICAgICAgICAgICAge1xuICAgICAgICAgICAgICBpZDogUnVsZUlkLFxuICAgICAgICAgICAgICByZWFzb246XG4gICAgICAgICAgICAgICAgXCJDbG91ZHdhdGNoIHJlc291cmNlcyBoYXZlIGJlZW4gc2NvcGVkIGRvd24gdG8gdGhlIExvZ0dyb3VwIGxldmVsLCBob3dldmVyICogaXMgc3RpbGwgbmVlZGVkIGFzIHN0cmVhbSBuYW1lcyBhcmUgY3JlYXRlZCBqdXN0IGluIHRpbWUuXCIsXG4gICAgICAgICAgICB9LFxuICAgICAgICAgIF0sXG4gICAgICAgICAgdHJ1ZVxuICAgICAgICApO1xuICAgICAgfVxuICAgICk7XG5cbiAgICBbXCJBd3NTb2x1dGlvbnMtTDFcIiwgXCJBd3NQcm90b3R5cGluZy1MYW1iZGFMYXRlc3RWZXJzaW9uXCJdLmZvckVhY2goXG4gICAgICAoUnVsZUlkKSA9PiB7XG4gICAgICAgIE5hZ1N1cHByZXNzaW9ucy5hZGRSZXNvdXJjZVN1cHByZXNzaW9ucyhcbiAgICAgICAgICBwcm92aWRlcixcbiAgICAgICAgICBbXG4gICAgICAgICAgICB7XG4gICAgICAgICAgICAgIGlkOiBSdWxlSWQsXG4gICAgICAgICAgICAgIHJlYXNvbjpcbiAgICAgICAgICAgICAgICBcIkxhdGVzdCBydW50aW1lIGNhbm5vdCBiZSBjb25maWd1cmVkLiBDREsgd2lsbCBuZWVkIHRvIHVwZ3JhZGUgdGhlIFByb3ZpZGVyIGNvbnN0cnVjdCBhY2NvcmRpbmdseS5cIixcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgXSxcbiAgICAgICAgICB0cnVlXG4gICAgICAgICk7XG4gICAgICB9XG4gICAgKTtcblxuICAgIGNvbnN0IHByZXBhcmVTcGVjT3B0aW9uczogUHJlcGFyZUFwaVNwZWNPcHRpb25zID0ge1xuICAgICAgZGVmYXVsdEF1dGhvcml6ZXJSZWZlcmVuY2U6XG4gICAgICAgIHNlcmlhbGl6ZUFzQXV0aG9yaXplclJlZmVyZW5jZShkZWZhdWx0QXV0aG9yaXplciksXG4gICAgICBpbnRlZ3JhdGlvbnM6IE9iamVjdC5mcm9tRW50cmllcyhcbiAgICAgICAgT2JqZWN0LmVudHJpZXMoaW50ZWdyYXRpb25zKS5tYXAoXG4gICAgICAgICAgKFtvcGVyYXRpb25JZCwgeyBhdXRob3JpemVyLCBpbnRlZ3JhdGlvbiB9XSkgPT4gW1xuICAgICAgICAgICAgb3BlcmF0aW9uSWQsXG4gICAgICAgICAgICB7XG4gICAgICAgICAgICAgIGludGVncmF0aW9uOiBpbnRlZ3JhdGlvbi5yZW5kZXIoe1xuICAgICAgICAgICAgICAgIG9wZXJhdGlvbklkLFxuICAgICAgICAgICAgICAgIHNjb3BlOiB0aGlzLFxuICAgICAgICAgICAgICAgIC4uLm9wZXJhdGlvbkxvb2t1cFtvcGVyYXRpb25JZF0sXG4gICAgICAgICAgICAgIH0pLFxuICAgICAgICAgICAgICBtZXRob2RBdXRob3JpemVyOiBzZXJpYWxpemVBc0F1dGhvcml6ZXJSZWZlcmVuY2UoYXV0aG9yaXplciksXG4gICAgICAgICAgICB9LFxuICAgICAgICAgIF1cbiAgICAgICAgKVxuICAgICAgKSxcbiAgICAgIHNlY3VyaXR5U2NoZW1lczogcHJlcGFyZVNlY3VyaXR5U2NoZW1lcyhcbiAgICAgICAgdGhpcyxcbiAgICAgICAgaW50ZWdyYXRpb25zLFxuICAgICAgICBkZWZhdWx0QXV0aG9yaXplclxuICAgICAgKSxcbiAgICAgIGNvcnNPcHRpb25zOiBjb3JzT3B0aW9ucyAmJiB7XG4gICAgICAgIGFsbG93SGVhZGVyczogY29yc09wdGlvbnMuYWxsb3dIZWFkZXJzIHx8IENvcnMuREVGQVVMVF9IRUFERVJTLFxuICAgICAgICBhbGxvd01ldGhvZHM6IGNvcnNPcHRpb25zLmFsbG93TWV0aG9kcyB8fCBDb3JzLkFMTF9NRVRIT0RTLFxuICAgICAgICBhbGxvd09yaWdpbnM6IGNvcnNPcHRpb25zLmFsbG93T3JpZ2lucyxcbiAgICAgICAgc3RhdHVzQ29kZTogY29yc09wdGlvbnMuc3RhdHVzQ29kZSB8fCAyMDQsXG4gICAgICB9LFxuICAgICAgb3BlcmF0aW9uTG9va3VwLFxuICAgIH07XG5cbiAgICAvLyBTcGVjIHByZXBhcmF0aW9uIHdpbGwgaGFwcGVuIGluIGEgY3VzdG9tIHJlc291cmNlIGxhbWJkYSBzbyB0aGF0IHJlZmVyZW5jZXMgdG8gbGFtYmRhIGludGVncmF0aW9ucyBldGMgY2FuIGJlXG4gICAgLy8gcmVzb2x2ZWQuIEhvd2V2ZXIsIHdlIGFsc28gcHJlcGFyZSBpbmxpbmUgdG8gcGVyZm9ybSBzb21lIGFkZGl0aW9uYWwgdmFsaWRhdGlvbiBhdCBzeW50aCB0aW1lLlxuICAgIGNvbnN0IHByZXBhcmVkU3BlYyA9IHByZXBhcmVBcGlTcGVjKHNwZWMsIHByZXBhcmVTcGVjT3B0aW9ucyk7XG5cbiAgICBjb25zdCBwcmVwYXJlQXBpU3BlY0N1c3RvbVJlc291cmNlUHJvcGVydGllczogUHJlcGFyZUFwaVNwZWNDdXN0b21SZXNvdXJjZVByb3BlcnRpZXMgPVxuICAgICAge1xuICAgICAgICBpbnB1dFNwZWNMb2NhdGlvbjoge1xuICAgICAgICAgIGJ1Y2tldDogaW5wdXRTcGVjQXNzZXQuYnVja2V0LmJ1Y2tldE5hbWUsXG4gICAgICAgICAga2V5OiBpbnB1dFNwZWNBc3NldC5zM09iamVjdEtleSxcbiAgICAgICAgfSxcbiAgICAgICAgb3V0cHV0U3BlY0xvY2F0aW9uOiB7XG4gICAgICAgICAgYnVja2V0OiBpbnB1dFNwZWNBc3NldC5idWNrZXQuYnVja2V0TmFtZSxcbiAgICAgICAgICBrZXk6IHByZXBhcmVkU3BlY091dHB1dEtleVByZWZpeCxcbiAgICAgICAgfSxcbiAgICAgICAgLi4ucHJlcGFyZVNwZWNPcHRpb25zLFxuICAgICAgfTtcblxuICAgIGNvbnN0IHByZXBhcmVTcGVjQ3VzdG9tUmVzb3VyY2UgPSBuZXcgQ3VzdG9tUmVzb3VyY2UoXG4gICAgICB0aGlzLFxuICAgICAgXCJQcmVwYXJlU3BlY0N1c3RvbVJlc291cmNlXCIsXG4gICAgICB7XG4gICAgICAgIHNlcnZpY2VUb2tlbjogcHJvdmlkZXIuc2VydmljZVRva2VuLFxuICAgICAgICBwcm9wZXJ0aWVzOiBwcmVwYXJlQXBpU3BlY0N1c3RvbVJlc291cmNlUHJvcGVydGllcyxcbiAgICAgIH1cbiAgICApO1xuXG4gICAgLy8gQ3JlYXRlIHRoZSBhcGkgZ2F0ZXdheSByZXNvdXJjZXMgZnJvbSB0aGUgc3BlYywgYXVnbWVudGluZyB0aGUgc3BlYyB3aXRoIHRoZSBwcm9wZXJ0aWVzIHNwZWNpZmljIHRvIGFwaSBnYXRld2F5XG4gICAgLy8gc3VjaCBhcyBpbnRlZ3JhdGlvbnMgb3IgYXV0aCB0eXBlc1xuICAgIHRoaXMuYXBpID0gbmV3IFNwZWNSZXN0QXBpKHRoaXMsIGlkLCB7XG4gICAgICBhcGlEZWZpbml0aW9uOiBBcGlEZWZpbml0aW9uLmZyb21CdWNrZXQoXG4gICAgICAgIGlucHV0U3BlY0Fzc2V0LmJ1Y2tldCxcbiAgICAgICAgcHJlcGFyZVNwZWNDdXN0b21SZXNvdXJjZS5nZXRBdHRTdHJpbmcoXCJvdXRwdXRTcGVjS2V5XCIpXG4gICAgICApLFxuICAgICAgZGVwbG95T3B0aW9uczoge1xuICAgICAgICBhY2Nlc3NMb2dEZXN0aW5hdGlvbjogbmV3IExvZ0dyb3VwTG9nRGVzdGluYXRpb24oXG4gICAgICAgICAgbmV3IExvZ0dyb3VwKHNjb3BlLCBgQWNjZXNzTG9nc2ApXG4gICAgICAgICksXG4gICAgICAgIGFjY2Vzc0xvZ0Zvcm1hdDogQWNjZXNzTG9nRm9ybWF0LmNsZigpLFxuICAgICAgICBsb2dnaW5nTGV2ZWw6IE1ldGhvZExvZ2dpbmdMZXZlbC5JTkZPLFxuICAgICAgfSxcbiAgICAgIC4uLm9wdGlvbnMsXG4gICAgfSk7XG5cbiAgICB0aGlzLmFwaS5ub2RlLmFkZERlcGVuZGVuY3kocHJlcGFyZVNwZWNDdXN0b21SZXNvdXJjZSk7XG5cbiAgICAvLyBXaGlsZSB0aGUgYXBpIHdpbGwgYmUgdXBkYXRlZCB3aGVuIHRoZSBvdXRwdXQgcGF0aCBmcm9tIHRoZSBjdXN0b20gcmVzb3VyY2UgY2hhbmdlcywgQ0RLIHN0aWxsIG5lZWRzIHRvIGtub3cgd2hlblxuICAgIC8vIHRvIHJlZGVwbG95IHRoZSBhcGkuIFRoaXMgaXMgYWNoaWV2ZWQgYnkgaW5jbHVkaW5nIGEgaGFzaCBvZiB0aGUgc3BlYyBpbiB0aGUgbG9naWNhbCBpZCAoaW50ZXJuYWxpc2VkIGluIHRoZVxuICAgIC8vIGFkZFRvTG9naWNhbElkIG1ldGhvZCBzaW5jZSB0aGlzIGlzIGhvdyBjaGFuZ2VzIG9mIGluZGl2aWR1YWwgcmVzb3VyY2VzL21ldGhvZHMgZXRjIHRyaWdnZXIgcmVkZXBsb3ltZW50cyBpbiBDREspXG4gICAgdGhpcy5hcGkubGF0ZXN0RGVwbG95bWVudD8uYWRkVG9Mb2dpY2FsSWQocHJlcGFyZWRTcGVjKTtcblxuICAgIC8vIEdyYW50IEFQSSBHYXRld2F5IHBlcm1pc3Npb24gdG8gaW52b2tlIHRoZSBpbnRlZ3JhdGlvbnNcbiAgICBPYmplY3Qua2V5cyhpbnRlZ3JhdGlvbnMpLmZvckVhY2goKG9wZXJhdGlvbklkKSA9PiB7XG4gICAgICBpbnRlZ3JhdGlvbnNbb3BlcmF0aW9uSWRdLmludGVncmF0aW9uLmdyYW50KHtcbiAgICAgICAgb3BlcmF0aW9uSWQsXG4gICAgICAgIHNjb3BlOiB0aGlzLFxuICAgICAgICBhcGk6IHRoaXMuYXBpLFxuICAgICAgICAuLi5vcGVyYXRpb25Mb29rdXBbb3BlcmF0aW9uSWRdLFxuICAgICAgfSk7XG4gICAgfSk7XG5cbiAgICAvLyBHcmFudCBBUEkgR2F0ZXdheSBwZXJtaXNzaW9uIHRvIGludm9rZSBlYWNoIGN1c3RvbSBhdXRob3JpemVyIGxhbWJkYSAoaWYgYW55KVxuICAgIGdldEF1dGhvcml6ZXJGdW5jdGlvbnMocHJvcHMpLmZvckVhY2goKHsgbGFiZWwsIGZ1bmN0aW9uOiBsYW1iZGEgfSkgPT4ge1xuICAgICAgbmV3IENmblBlcm1pc3Npb24odGhpcywgYExhbWJkYVBlcm1pc3Npb24tJHtsYWJlbH1gLCB7XG4gICAgICAgIGFjdGlvbjogXCJsYW1iZGE6SW52b2tlRnVuY3Rpb25cIixcbiAgICAgICAgcHJpbmNpcGFsOiBcImFwaWdhdGV3YXkuYW1hem9uYXdzLmNvbVwiLFxuICAgICAgICBmdW5jdGlvbk5hbWU6IGxhbWJkYS5mdW5jdGlvbkFybixcbiAgICAgICAgc291cmNlQXJuOiBzdGFjay5mb3JtYXRBcm4oe1xuICAgICAgICAgIHNlcnZpY2U6IFwiZXhlY3V0ZS1hcGlcIixcbiAgICAgICAgICByZXNvdXJjZTogdGhpcy5hcGkucmVzdEFwaUlkLFxuICAgICAgICAgIHJlc291cmNlTmFtZTogXCIqLypcIixcbiAgICAgICAgfSksXG4gICAgICB9KTtcbiAgICB9KTtcblxuICAgIC8vIENyZWF0ZSBhbmQgYXNzb2NpYXRlIHRoZSB3ZWIgYWNsIGlmIG5vdCBkaXNhYmxlZFxuICAgIGlmICghcHJvcHMud2ViQWNsT3B0aW9ucz8uZGlzYWJsZSkge1xuICAgICAgY29uc3QgYWNsID0gbmV3IE9wZW5BcGlHYXRld2F5V2ViQWNsKHRoaXMsIGAke2lkfS1BY2xgLCB7XG4gICAgICAgIC4uLnByb3BzLndlYkFjbE9wdGlvbnMsXG4gICAgICAgIGFwaURlcGxveW1lbnRTdGFnZUFybjogdGhpcy5hcGkuZGVwbG95bWVudFN0YWdlLnN0YWdlQXJuLFxuICAgICAgfSk7XG5cbiAgICAgIHRoaXMud2ViQWNsID0gYWNsLndlYkFjbDtcbiAgICAgIHRoaXMuaXBTZXQgPSBhY2wuaXBTZXQ7XG4gICAgICB0aGlzLndlYkFjbEFzc29jaWF0aW9uID0gYWNsLndlYkFjbEFzc29jaWF0aW9uO1xuICAgIH1cblxuICAgIFtcIkF3c1NvbHV0aW9ucy1JQU00XCIsIFwiQXdzUHJvdG90eXBpbmctSUFNTm9NYW5hZ2VkUG9saWNpZXNcIl0uZm9yRWFjaChcbiAgICAgIChSdWxlSWQpID0+IHtcbiAgICAgICAgTmFnU3VwcHJlc3Npb25zLmFkZFJlc291cmNlU3VwcHJlc3Npb25zKFxuICAgICAgICAgIHRoaXMsXG4gICAgICAgICAgW1xuICAgICAgICAgICAge1xuICAgICAgICAgICAgICBpZDogUnVsZUlkLFxuICAgICAgICAgICAgICByZWFzb246XG4gICAgICAgICAgICAgICAgXCJDbG91ZHdhdGNoIFJvbGUgcmVxdWlyZXMgYWNjZXNzIHRvIGNyZWF0ZS9yZWFkIGdyb3VwcyBhdCB0aGUgcm9vdCBsZXZlbC5cIixcbiAgICAgICAgICAgICAgYXBwbGllc1RvOiBbXG4gICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgcmVnZXg6IGAvXlBvbGljeTo6YXJuOiR7UERLTmFnLmdldFN0YWNrUGFydGl0aW9uUmVnZXgoXG4gICAgICAgICAgICAgICAgICAgIHN0YWNrXG4gICAgICAgICAgICAgICAgICApfTppYW06OmF3czpwb2xpY3kvc2VydmljZS1yb2xlL0FtYXpvbkFQSUdhdGV3YXlQdXNoVG9DbG91ZFdhdGNoTG9ncyQvZ2AsXG4gICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgXSxcbiAgICAgICAgICB0cnVlXG4gICAgICAgICk7XG4gICAgICB9XG4gICAgKTtcblxuICAgIFtcIkF3c1NvbHV0aW9ucy1BUElHMlwiLCBcIkF3c1Byb3RvdHlwaW5nLUFQSUdXUmVxdWVzdFZhbGlkYXRpb25cIl0uZm9yRWFjaChcbiAgICAgIChSdWxlSWQpID0+IHtcbiAgICAgICAgTmFnU3VwcHJlc3Npb25zLmFkZFJlc291cmNlU3VwcHJlc3Npb25zKFxuICAgICAgICAgIHRoaXMsXG4gICAgICAgICAgW1xuICAgICAgICAgICAge1xuICAgICAgICAgICAgICBpZDogUnVsZUlkLFxuICAgICAgICAgICAgICByZWFzb246XG4gICAgICAgICAgICAgICAgXCJUaGlzIGNvbnN0cnVjdCBpbXBsZW1lbnRzIGZpbmUgZ3JhaW5lZCB2YWxpZGF0aW9uIHZpYSBPcGVuQXBpLlwiLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICBdLFxuICAgICAgICAgIHRydWVcbiAgICAgICAgKTtcbiAgICAgIH1cbiAgICApO1xuICB9XG59XG4iXX0=