import asyncio

from asgiref.sync import sync_to_async
from django.db import models as django_models


def default_cache_func(
        django_model,
        redis_root,
        redis_model,
        save_related_models,
        get_or_create_redis_model_from_django_model,
):

    redis_dicts = redis_root.get(redis_model, return_dict=True)
    # cache_to_django(
    #     django_model,
    #     redis_dicts,
    #     redis_root,
    # )
    django_to_cache(
        django_model,
        redis_model,
        redis_root,
        save_related_models,
        get_or_create_redis_model_from_django_model,
    )


def cache_to_django(django_model, redis_dicts, redis_root):
    async def run_cache_to_django(django_model, redis_dicts, redis_root):
        async_chunk_size = redis_root.async_db_requests_limit
        async_tasks = []
        completed = 0
        to_complete = len(redis_dicts.keys())

        for redis_instance_id, redis_instance_data in redis_dicts.items():
            async_tasks.append(
                update_or_create_django_instance_from_redis_instance(
                    redis_instance_data,
                    django_model,
                )
            )
            completed += 1
            if len(async_tasks) == async_chunk_size or completed == to_complete:
                await asyncio.gather(*async_tasks)
                async_tasks = []
                print(f'Recorded {completed} {django_model.__name__} instances from cache to django')

    loop = asyncio.get_event_loop()
    loop.run_until_complete(run_cache_to_django(django_model, redis_dicts, redis_root))


async def update_or_create_django_instance_from_redis_instance(
        redis_instance,
        django_model,
):
    django_id = redis_instance['django_id']
    django_params, django_many_to_many_params = redis_dict_to_django_params(
        redis_instance,
        django_model
    )
    new_django_params = {}
    for k, v in django_params.items():
        if k != 'django_id':
            new_django_params[k] = v
    django_params = new_django_params
    django_instance = None
    if django_id != 'null':
        if django_model.objects.filter(id=django_id):
            django_instance = django_model.objects.filter(id=django_id).update(
                **django_params
            )
            update_django_many_to_many(
                django_model,
                django_id,
                django_many_to_many_params
            )
    if django_instance is None:
        django_instance = django_model.objects.create(
            **django_params
        )
        update_django_many_to_many(
            django_model,
            django_id,
            django_many_to_many_params
        )
    return django_instance


def redis_dict_to_django_params(
        redis_dict,
        django_model
):
    django_params = {}
    django_many_to_many_params = {}
    for django_field in django_model._meta.get_fields():
        django_field_name = django_field.name
        if django_field_name in redis_dict.keys():
            if django_field.__class__ == django_models.ForeignKey:
                foreign_key_model = django_field.remote_field.model
                redis_foreign_key_instance = redis_dict[django_field_name]
                django_foreign_key_instance = update_or_create_django_instance_from_redis_instance(
                    redis_foreign_key_instance,
                    foreign_key_model
                )
                django_params[django_field_name] = django_foreign_key_instance
            elif django_field.__class__ == django_models.ManyToManyField:
                many_to_many_model = django_field.remote_field.model
                django_many_to_many_instances = []
                redis_many_to_many_instances = redis_dict[django_field.name]
                for redis_many_to_many_instance in redis_many_to_many_instances:
                    django_many_to_many_instance = update_or_create_django_instance_from_redis_instance(
                        redis_many_to_many_instance,
                        many_to_many_model
                    )
                    django_many_to_many_instances.append(django_many_to_many_instance)
                django_many_to_many_params[django_field_name] = django_many_to_many_instances
            else:
                value = redis_dict[django_field.name]
                django_params[django_field_name] = value
    return django_params, django_many_to_many_params


def update_django_many_to_many(
        django_model,
        django_id,
        django_many_to_many_params
):
    django_instance = django_model.objects.get(id=django_id)
    for param_name, many_to_many_objects in django_many_to_many_params.items():
        param = getattr(django_instance, param_name, None)
        if param is not None:
            param.clear()
            for obj in many_to_many_objects:
                param.add(obj)


async def django_sync_to_async_list(
        django_func,
        params_to_use=None,
):
    if params_to_use is None:
        params_to_use = {}
    django_instances_qs = await sync_to_async(django_func)(**params_to_use)
    django_instances_iter = await sync_to_async(django_instances_qs.__iter__)()
    django_instances = list(django_instances_iter)
    return django_instances


@sync_to_async
def django_sync_to_async_get(
        django_instance,
        param_to_get,
):
    param = getattr(django_instance, param_to_get)
    return param


def django_to_cache(
        django_model,
        redis_model,
        redis_root,
        save_related_models,
        get_or_create_redis_model_from_django_model,
):
    async def run_django_to_cache(
            django_model,
            redis_model,
            redis_root,
            save_related_models,
            get_or_create_redis_model_from_django_model,
    ):
        django_instances = await django_sync_to_async_list(django_model.objects.all)
        async_chunk_size = redis_root.async_db_requests_limit
        async_tasks = []
        completed = 0
        to_complete = len(django_instances)

        for django_instance in django_instances:
            async_tasks.append(
                update_or_create_redis_instance_from_django_instance(
                    django_instance,
                    redis_model,
                    redis_root,
                    save_related_models,
                    get_or_create_redis_model_from_django_model,
                )
            )
            completed += 1
            if len(async_tasks) == async_chunk_size or completed == to_complete:
                await asyncio.gather(*async_tasks)
                async_tasks = []
                print(f'Recorded {completed} {django_model.__name__} instances from django to cache')

    loop = asyncio.get_event_loop()
    loop.run_until_complete(
        run_django_to_cache(
            django_model,
            redis_model,
            redis_root,
            save_related_models,
            get_or_create_redis_model_from_django_model,
        )
    )


async def update_or_create_redis_instance_from_django_instance(
        django_instance,
        redis_model,
        redis_root,
        get_or_create_redis_model_from_django_model,
        save_related_models,
):
    redis_instance = redis_root.get(redis_model, django_id=django_instance.id)
    redis_dict = await django_instance_to_redis_params(
        django_instance,
        redis_root,
        save_related_models,
        get_or_create_redis_model_from_django_model,
    )

    if not redis_instance:
        fields_to_create = redis_dict
        redis_instance = redis_model(
            redis_root=redis_root,
            django_id=django_instance.id,
            **fields_to_create
        ).save()
        result_redis_instance_id = redis_instance['id']
        print(f'{redis_model} with django_id={django_instance.id} created...')
    else:
        redis_instance = redis_instance[0]
        fields_to_update = check_fields_need_to_update(
            redis_instance,
            redis_dict,
        )
        redis_root.update(redis_model, redis_instance,
                          **fields_to_update
                          )
        result_redis_instance_id = redis_instance['id']
        print(f'{redis_model} with django_id={django_instance.id} existing...')

    return result_redis_instance_id


def check_fields_need_to_update(
        redis_instance,
        redis_dict,
):
    fields_to_update = {}
    for field_name, field_value in redis_dict.items():
        if field_name not in redis_instance.keys():
            fields_to_update[field_name] = field_value
        else:
            if redis_instance[field_name] != field_value:
                fields_to_update[field_name] = field_value
    return fields_to_update


async def django_instance_field_to_redis_value(
        django_instance,
        django_field,
        redis_root,
        save_related_models,
        get_or_create_redis_model_from_django_model,
):
    django_field_value = await django_sync_to_async_get(django_instance, django_field.name)
    redis_value = django_field_value
    if django_field.__class__ == django_models.ForeignKey:
        if django_field_value is None:
            redis_value = None
        else:
            django_foreign_key_instance = django_field_value
            if save_related_models:
                redis_foreign_key_instance = await get_or_create_redis_instance_from_django_instance(
                    django_instance,
                    redis_root,
                    save_related_models,
                    get_or_create_redis_model_from_django_model,
                )
                redis_value = redis_foreign_key_instance
            else:
                redis_value = django_foreign_key_instance.id
    elif django_field.__class__ == django_models.ManyToManyField:
        django_many_to_many_instances = await django_sync_to_async_list(django_field_value.all)
        if save_related_models:
            redis_value = [
                await get_or_create_redis_instance_from_django_instance(
                    django_instance,
                    redis_root,
                    save_related_models,
                    get_or_create_redis_model_from_django_model,
                )
                for django_many_to_many_instance in django_many_to_many_instances
            ]
        else:
            redis_value = [
                django_many_to_many_instance.id
                for django_many_to_many_instance in django_many_to_many_instances
            ]
    elif django_field.__class__.__name__.startswith('Image') or django_field.__class__.__name__.startswith('File'):
        try:
            redis_value = django_field_value.file.path
        except:
            redis_value = None
    return redis_value


async def django_instance_to_redis_params(
        django_instance,
        redis_root,
        save_related_models,
        get_or_create_redis_model_from_django_model,
):
    redis_params = {}
    for i, django_field in enumerate(django_instance.__class__._meta.get_fields()):
        if not django_field.__class__.__name__.endswith('Rel'):
            redis_param = await django_instance_field_to_redis_value(
                django_instance,
                django_field,
                redis_root,
                save_related_models,
                get_or_create_redis_model_from_django_model,
            )
            redis_params[django_field.name] = redis_param
    return redis_params


def get_or_create_redis_instance_from_django_instance(
        django_instance,
        redis_root,
        save_related_models,
        get_or_create_redis_model_from_django_model,
):
    django_instance_model = django_instance.__class__
    redis_model = get_or_create_redis_model_from_django_model(django_instance_model, redis_root)
    redis_instance = redis_root.get(redis_model, django_id=django_instance.id)
    if not redis_instance:
        redis_dict = django_instance_to_redis_params(
            django_instance,
            redis_root,
            save_related_models,
            get_or_create_redis_model_from_django_model,
        )
        redis_instance = redis_model(
            redis_root=redis_root,
            django_id=django_instance.id,
            **redis_dict
        ).save()
    else:
        redis_instance = redis_instance[0]

    return redis_instance
