"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const lib_1 = require("../lib");
const cdk = require("@aws-cdk/core");
const lambda = require("@aws-cdk/aws-lambda");
const api = require("@aws-cdk/aws-apigateway");
require("@aws-cdk/assert/jest");
function deployNewFunc(stack) {
    const lambdaFunctionProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_10_X,
        handler: 'index.handler'
    };
    return new lib_1.CognitoToApiGatewayToLambda(stack, 'test-cognito-apigateway-lambda', {
        lambdaFunctionProps
    });
}
test('override cognito properties', () => {
    const stack = new cdk.Stack();
    const lambdaFunctionProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_12_X,
        handler: 'index.handler'
    };
    const cognitoUserPoolProps = {
        userPoolName: 'test',
        userVerification: {}
    };
    new lib_1.CognitoToApiGatewayToLambda(stack, 'test-cognito-apigateway-lambda', {
        lambdaFunctionProps,
        cognitoUserPoolProps
    });
    expect(stack).toHaveResource('AWS::Cognito::UserPool', {
        AdminCreateUserConfig: {
            AllowAdminCreateUserOnly: true
        },
        EmailVerificationMessage: "The verification code to your new account is {####}",
        EmailVerificationSubject: "Verify your new account",
        SmsVerificationMessage: "The verification code to your new account is {####}",
        UserPoolAddOns: {
            AdvancedSecurityMode: "ENFORCED"
        },
        UserPoolName: "test",
        VerificationMessageTemplate: {
            DefaultEmailOption: "CONFIRM_WITH_CODE",
            EmailMessage: "The verification code to your new account is {####}",
            EmailSubject: "Verify your new account",
            SmsMessage: "The verification code to your new account is {####}"
        }
    });
});
test('check exception for Missing existingObj from props', () => {
    const stack = new cdk.Stack();
    const props = {};
    try {
        new lib_1.CognitoToApiGatewayToLambda(stack, 'test-cognito-apigateway-lambda', props);
    }
    catch (e) {
        expect(e).toBeInstanceOf(Error);
    }
});
test('check properties', () => {
    const stack = new cdk.Stack();
    const construct = deployNewFunc(stack);
    expect(construct.userPool !== null);
    expect(construct.userPoolClient !== null);
    expect(construct.apiGateway !== null);
    expect(construct.lambdaFunction !== null);
    expect(construct.apiGatewayCloudWatchRole !== null);
    expect(construct.apiGatewayLogGroup !== null);
    expect(construct.apiGatewayAuthorizer !== null);
});
test('override cognito cognitoUserPoolClientProps', () => {
    const stack = new cdk.Stack();
    const lambdaFunctionProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_12_X,
        handler: 'index.handler'
    };
    const cognitoUserPoolClientProps = {
        authFlows: {
            userSrp: true
        }
    };
    new lib_1.CognitoToApiGatewayToLambda(stack, 'test-cognito-apigateway-lambda', {
        lambdaFunctionProps,
        cognitoUserPoolClientProps
    });
    expect(stack).toHaveResource('AWS::Cognito::UserPoolClient', {
        ExplicitAuthFlows: [
            "ALLOW_USER_SRP_AUTH",
            "ALLOW_REFRESH_TOKEN_AUTH"
        ],
    });
});
test('Check for default Cognito Auth Type', () => {
    const stack = new cdk.Stack();
    const lambdaFunctionProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_12_X,
        handler: 'index.handler'
    };
    new lib_1.CognitoToApiGatewayToLambda(stack, 'test-cognito-apigateway-lambda', {
        lambdaFunctionProps
    });
    expect(stack).toHaveResource('AWS::ApiGateway::Method', {
        ResourceId: {
            "Fn::GetAtt": [
                "testcognitoapigatewaylambdaLambdaRestApi2E272431",
                "RootResourceId"
            ]
        },
        AuthorizationType: "COGNITO_USER_POOLS",
        AuthorizerId: {
            Ref: "testcognitoapigatewaylambdaCognitoAuthorizer170CACC9"
        },
    });
    expect(stack).toHaveResource('AWS::ApiGateway::Method', {
        ResourceId: {
            Ref: "testcognitoapigatewaylambdaLambdaRestApiproxy23E1DA20"
        },
        AuthorizationType: "COGNITO_USER_POOLS",
        AuthorizerId: {
            Ref: "testcognitoapigatewaylambdaCognitoAuthorizer170CACC9"
        },
    });
});
test('override Auth Type to COGNITO', () => {
    const stack = new cdk.Stack();
    const lambdaFunctionProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_12_X,
        handler: 'index.handler'
    };
    new lib_1.CognitoToApiGatewayToLambda(stack, 'test-cognito-apigateway-lambda', {
        lambdaFunctionProps,
        apiGatewayProps: {
            defaultMethodOptions: {
                authorizationType: api.AuthorizationType.COGNITO
            }
        }
    });
    expect(stack).toHaveResource('AWS::ApiGateway::Method', {
        ResourceId: {
            "Fn::GetAtt": [
                "testcognitoapigatewaylambdaLambdaRestApi2E272431",
                "RootResourceId"
            ]
        },
        AuthorizationType: "COGNITO_USER_POOLS",
        AuthorizerId: {
            Ref: "testcognitoapigatewaylambdaCognitoAuthorizer170CACC9"
        },
    });
    expect(stack).toHaveResource('AWS::ApiGateway::Method', {
        ResourceId: {
            Ref: "testcognitoapigatewaylambdaLambdaRestApiproxy23E1DA20"
        },
        AuthorizationType: "COGNITO_USER_POOLS",
        AuthorizerId: {
            Ref: "testcognitoapigatewaylambdaCognitoAuthorizer170CACC9"
        },
    });
});
test('Try to override Auth Type to NONE', () => {
    const stack = new cdk.Stack();
    const lambdaFunctionProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_12_X,
        handler: 'index.handler'
    };
    new lib_1.CognitoToApiGatewayToLambda(stack, 'test-cognito-apigateway-lambda', {
        lambdaFunctionProps,
        apiGatewayProps: {
            defaultMethodOptions: {
                authorizationType: api.AuthorizationType.NONE
            }
        }
    });
    expect(stack).toHaveResource('AWS::ApiGateway::Method', {
        ResourceId: {
            "Fn::GetAtt": [
                "testcognitoapigatewaylambdaLambdaRestApi2E272431",
                "RootResourceId"
            ]
        },
        AuthorizationType: "COGNITO_USER_POOLS",
        AuthorizerId: {
            Ref: "testcognitoapigatewaylambdaCognitoAuthorizer170CACC9"
        },
    });
    expect(stack).toHaveResource('AWS::ApiGateway::Method', {
        ResourceId: {
            Ref: "testcognitoapigatewaylambdaLambdaRestApiproxy23E1DA20"
        },
        AuthorizationType: "COGNITO_USER_POOLS",
        AuthorizerId: {
            Ref: "testcognitoapigatewaylambdaCognitoAuthorizer170CACC9"
        },
    });
});
test('Override apiGatewayProps', () => {
    const stack = new cdk.Stack();
    const lambdaFunctionProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_12_X,
        handler: 'index.handler'
    };
    new lib_1.CognitoToApiGatewayToLambda(stack, 'test-cognito-apigateway-lambda', {
        lambdaFunctionProps,
        apiGatewayProps: {
            defaultMethodOptions: {
                operationName: 'foo-bar'
            }
        }
    });
    expect(stack).toHaveResource('AWS::ApiGateway::Method', {
        OperationName: "foo-bar",
        ResourceId: {
            "Fn::GetAtt": [
                "testcognitoapigatewaylambdaLambdaRestApi2E272431",
                "RootResourceId"
            ]
        },
        AuthorizationType: "COGNITO_USER_POOLS",
        AuthorizerId: {
            Ref: "testcognitoapigatewaylambdaCognitoAuthorizer170CACC9"
        },
    });
    expect(stack).toHaveResource('AWS::ApiGateway::Method', {
        OperationName: "foo-bar",
        ResourceId: {
            Ref: "testcognitoapigatewaylambdaLambdaRestApiproxy23E1DA20"
        },
        AuthorizationType: "COGNITO_USER_POOLS",
        AuthorizerId: {
            Ref: "testcognitoapigatewaylambdaCognitoAuthorizer170CACC9"
        },
    });
});
test('Override apiGatewayProps to support CORS', () => {
    const stack = new cdk.Stack();
    const lambdaFunctionProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_12_X,
        handler: 'index.handler'
    };
    new lib_1.CognitoToApiGatewayToLambda(stack, 'test-cognito-apigateway-lambda', {
        lambdaFunctionProps,
        apiGatewayProps: {
            defaultCorsPreflightOptions: {
                allowOrigins: api.Cors.ALL_ORIGINS,
                allowMethods: api.Cors.ALL_METHODS
            }
        }
    });
    expect(stack).toHaveResource('AWS::ApiGateway::Method', {
        HttpMethod: "ANY",
        ResourceId: {
            "Fn::GetAtt": [
                "testcognitoapigatewaylambdaLambdaRestApi2E272431",
                "RootResourceId"
            ]
        },
        AuthorizationType: "COGNITO_USER_POOLS",
        AuthorizerId: {
            Ref: "testcognitoapigatewaylambdaCognitoAuthorizer170CACC9"
        },
    });
    expect(stack).toHaveResource('AWS::ApiGateway::Method', {
        HttpMethod: "ANY",
        ResourceId: {
            Ref: "testcognitoapigatewaylambdaLambdaRestApiproxy23E1DA20"
        },
        AuthorizationType: "COGNITO_USER_POOLS",
        AuthorizerId: {
            Ref: "testcognitoapigatewaylambdaCognitoAuthorizer170CACC9"
        },
    });
    expect(stack).toHaveResource('AWS::ApiGateway::Method', {
        HttpMethod: "OPTIONS",
        ResourceId: {
            "Fn::GetAtt": [
                "testcognitoapigatewaylambdaLambdaRestApi2E272431",
                "RootResourceId"
            ]
        },
        AuthorizationType: "NONE"
    });
    expect(stack).toHaveResource('AWS::ApiGateway::Method', {
        HttpMethod: "OPTIONS",
        ResourceId: {
            Ref: "testcognitoapigatewaylambdaLambdaRestApiproxy23E1DA20"
        },
        AuthorizationType: "NONE"
    });
});
test('Override apiGatewayProps with proxy = false and add POST method', () => {
    const stack = new cdk.Stack();
    const lambdaFunctionProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_12_X,
        handler: 'index.handler'
    };
    const c = new lib_1.CognitoToApiGatewayToLambda(stack, 'test-cognito-apigateway-lambda', {
        lambdaFunctionProps,
        apiGatewayProps: {
            proxy: false
        }
    });
    const r = c.apiGateway.root.addResource('foo');
    r.addMethod('POST');
    // Super imporant to call this method to Apply the Cognito Authorizers
    c.addAuthorizers();
    expect(stack).toHaveResource('AWS::ApiGateway::Method', {
        HttpMethod: "POST",
        ResourceId: {
            Ref: "testcognitoapigatewaylambdaLambdaRestApifoo89ACA437"
        },
        AuthorizationType: "COGNITO_USER_POOLS",
        AuthorizerId: {
            Ref: "testcognitoapigatewaylambdaCognitoAuthorizer170CACC9"
        },
    });
    expect(stack).toHaveResource('AWS::ApiGateway::Resource', {
        PathPart: "foo",
    });
});
test('Override apiGatewayProps with proxy = false and add OPTIONS method', () => {
    const stack = new cdk.Stack();
    const lambdaFunctionProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_12_X,
        handler: 'index.handler'
    };
    const c = new lib_1.CognitoToApiGatewayToLambda(stack, 'test-cognito-apigateway-lambda', {
        lambdaFunctionProps,
        apiGatewayProps: {
            proxy: false
        }
    });
    const r = c.apiGateway.root.addResource('foo');
    r.addMethod('OPTIONS');
    // Mandatory to call this method to Apply the Cognito Authorizers
    c.addAuthorizers();
    expect(stack).toHaveResource('AWS::ApiGateway::Method', {
        HttpMethod: "OPTIONS",
        ResourceId: {
            Ref: "testcognitoapigatewaylambdaLambdaRestApifoo89ACA437"
        },
        AuthorizationType: "NONE",
    });
    expect(stack).toHaveResource('AWS::ApiGateway::Resource', {
        PathPart: "foo",
    });
});
//# sourceMappingURL=data:application/json;base64,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