"""
Generates a molecule dataset from a CSV source dataset.

**CHANGELOG**

0.1.0 - 10.12.2022 - Initial version

0.2.0 - 22.01.2023 - (1) Added the optional feature to extract the 3D atomic coordinates for each atom of
each molecule as well and save that as additional property of the resulting graph representation.
(2) Added the possibility to extract known train test-splits from the CSV file as well.
(3) It is not possible to use a CSV file from the local system as well instead of downloading from the
remote file share provider

0.3.0 - 24.02.2023 - (1) Added additional convenience where the printed logs are more detailed and an
additional artifact is created which contains all the omitted elements and the reason why they were omitted
(2) Added support for classification datasets via an additional flag parameter

0.4.0 - 21.03.2023 - big preprocessing update: The processing of the smiles strings into valid VGD elements
is not longer hard coded in this experiment, but has been moved as generic functionality of the
MoleculeProcessing class which is now used here.
Most importantly: This class also enables the dynamic generation of standalone python module which will be
shipped together with the dataset and which allows the streamlined conversion of any SMILES string into
the exact format of the VGD.
"""
import inspect
import os
import sys
import time
import json
import csv
import yaml
import datetime
import typing as t
from itertools import accumulate, chain
from pprint import pprint

import rdkit.Chem.Descriptors
import rdkit.Chem.Crippen
import rdkit.Chem.AllChem
from rdkit.rdBase import BlockLogs
import numpy as np
import numpy.linalg as la
import matplotlib.pyplot as plt
from matplotlib.backends.backend_pdf import PdfPages
from pycomex.experiment import Experiment
from pycomex.util import Skippable
from rdkit import Chem

import visual_graph_datasets.typing as tc
from visual_graph_datasets.config import Config
from visual_graph_datasets.web import AbstractFileShare, get_file_share
from visual_graph_datasets.processing.base import identity, list_identity
from visual_graph_datasets.processing.base import create_processing_module
from visual_graph_datasets.processing.molecules import chem_prop, chem_descriptor
from visual_graph_datasets.processing.molecules import apply_atom_callbacks, apply_bond_callbacks
from visual_graph_datasets.processing.molecules import mol_from_smiles
from visual_graph_datasets.processing.molecules import OneHotEncoder
from visual_graph_datasets.processing.molecules import MoleculeProcessing
from visual_graph_datasets.visualization.base import create_frameless_figure
from visual_graph_datasets.visualization.molecules import visualize_molecular_graph_from_mol
from visual_graph_datasets.data import NumericJsonEncoder
from visual_graph_datasets.data import load_visual_graph_dataset
from visual_graph_datasets.data import generate_visual_graph_dataset_metadata

# This will block the annoying warning generated by RDKit
block = BlockLogs()

# == SOURCE PARAMETERS ==
# These parameters determine how to handle the source CSV file of the dataset. There is the possibility
# to define a file from the local system or to download a file from the VGD remote location

# The vgd file share provider from which to download the CSV file to be used as the source for the VGD
# conversion.
FILE_SHARE_PROVIDER: str = 'main'
# This may be one of the following two things:
# 1. A valid absolute file path on the local system pointing to a CSV file to be used as the source for
#    the VGD conversion
# 2. A valid relative path to a CSV file stashed on the given vgd file share provider which will be
#    downloaded first and then processed.
CSV_FILE_NAME: str = 'source/benzene_solubility.csv'
# Optionally, this may define the string name of the CSV column which contains the integer index
# associated with each dataset element. If this is not given, then integer indices will be randomly
# generated for each element in the final VGD
INDEX_COLUMN_NAME: t.Optional[str] = None
# This has to be the string name of the CSV column which contains the SMILES string representation of
# the molecule.
SMILES_COLUMN_NAME: str = 'SMILES'
# This has to be the string name of the CSV column which contains the target value
TARGET_TYPE: str = 'regression'  # 'classification'
TARGET_COLUMN_NAMES: t.List[str] = ['LogS']
# The keys of this dictionary are integers which represent the indices of various train test splits. The
# values are the string names of the columns which define those corresponding splits. It is expected that
# these CSV columns contain a "1" if that corresponding element is considered as part of the training set
# of that split and "0" if it is part of the test set.
# This dictionary may be empty and then no information about splits will be added to the dataset at all.
SPLIT_COLUMN_NAMES: t.Dict[int, str] = {
}

# == PROCESSING PARAMETERS ==
# These parameters control the processing of the raw SMILES into the molecule representations with RDKit
# and then finally the conversion into the graph dict representation.


class VgdMoleculeProcessing(MoleculeProcessing):

    node_attribute_map = {
        'symbol': {
            'callback': chem_prop('GetSymbol', OneHotEncoder(
                ['B', 'C', 'N', 'O', 'F', 'Si', 'P', 'S', 'Cl', 'As', 'Se', 'Br', 'Te', 'I', 'At'],
                add_unknown=True,
                dtype=str
            )),
            'description': 'one-hot encoding of atom type',
        },
        'hybridization': {
            'callback': chem_prop('GetHybridization', OneHotEncoder(
                [2, 3, 4, 5, 6],
                add_unknown=True,
                dtype=int,
            )),
            'description': 'one-hot encoding of atom hybridization',
        },
        'total_degree': {
            'callback': chem_prop('GetTotalDegree', OneHotEncoder(
                [0, 1, 2, 3, 4, 5],
                add_unknown=False,
                dtype=int
            )),
            'description': 'one-hot encoding of the degree of the atom'
        },
        'num_hydrogen_atoms': {
            'callback': chem_prop('GetTotalNumHs', OneHotEncoder(
                [0, 1, 2, 3, 4],
                add_unknown=False,
                dtype=int
            )),
            'description': 'one-hot encoding of the total number of attached hydrogen atoms'
        },
        'mass': {
            'callback': chem_prop('GetMass', list_identity),
            'description': 'The mass of the atom'
        },
        'charge': {
            'callback': chem_prop('GetFormalCharge', list_identity),
            'description': 'The charge of the atom',
        },
        'is_aromatic': {
            'callback': chem_prop('GetIsAromatic', list_identity),
            'description': 'Boolean flag of whether the atom is aromatic',
        },
        'is_in_ring': {
            'callback': chem_prop('IsInRing', list_identity),
            'description': 'Boolean flag of whether atom is part of a ring'
        }
    }

    edge_attribute_map = {
        'bond_type': {
            'callback': chem_prop('GetBondType', OneHotEncoder(
                [1, 2, 3, 12],
                add_unknown=False,
                dtype=int,
            )),
            'description': 'one-hot encoding of the bond type',
        },
        'stereo': {
            'callback': chem_prop('GetStereo', OneHotEncoder(
                [0, 1, 2, 3],
                add_unknown=False,
                dtype=int,
            )),
            'description': 'one-hot encoding of the stereo property'
        },
        'is_aromatic': {
            'callback': chem_prop('GetIsAromatic', list_identity),
            'description': 'boolean flag of whether bond is aromatic',
        },
        'is_in_ring': {
            'callback': chem_prop('IsInRing', list_identity),
            'description': 'boolean flag of whether bond is part of ring',
        },
        'is_conjugated': {
            'callback': chem_prop('GetIsConjugated', list_identity),
            'description': 'boolean flag of whether bond is conjugated'
        }
    }

    graph_attribute_map = {
        'molecular_weight': {
            'callback': chem_descriptor(Chem.Descriptors.ExactMolWt, list_identity),
            'description': 'the exact molecular weight of the molecule',
        },
        'num_radical_electrons': {
            'callback': chem_descriptor(Chem.Descriptors.NumRadicalElectrons, list_identity),
            'description': 'the total number of radical electrons in the molecule',
        },
        'num_valence_electrons': {
            'callback': chem_descriptor(Chem.Descriptors.NumValenceElectrons, list_identity),
            'description': 'the total number of valence electrons in the molecule'
        }
    }


PROCESSING = VgdMoleculeProcessing()
# If this flag is True, the undirected edges which make up molecular graph will be converted into two
# opposing directed edges. Depends on the downstream ML framework to be used.
UNDIRECTED_EDGES_AS_TWO: bool = True
# If this flag is True, the coordinates of each atom will be calculated for each molecule and the resulting
# 3D coordinate vector will be added as a separate property of the resulting graph dict.
USE_NODE_COORDINATES: bool = True
GRAPH_METADATA_CALLBACKS = {
    'name': lambda mol, data: data['smiles'],
    'smiles': lambda mol, data: data['smiles'],
}


# Filters: This list is supposed to contain callback functions which take the molecule object and the data
# dictionary belonging to that function and if even one of those callbacks returns true, the corresponding
# element will be skipped and NOT included in the final VGD. In that sense these callbacks define the
# rules by which to filter the original dataset.
def no_bonds(mol, data):
    # All molecules which don't have any bonds do not qualify as graphs and will be skipped
    return len(mol.GetBonds()) == 0 or len(mol.GetAtoms()) == 1


FILTER_CALLBACKS = [
    no_bonds
]

# == DATASET PARAMETERS ==
# These parameters control aspects of the visual graph dataset creation process

# The name given to the visual graph dataset folder which will be created.
DATASET_NAME: str = 'benzene'
# The width and height of the molecule visualization PNGs in pixels.
IMAGE_WIDTH: int = 1000
IMAGE_HEIGHT: int = 1000
# This dict will be converted into the .meta.yml file which will be added to the final visual graph dataset
# folder. This is an optional file, which can add additional meta information about the entire dataset
# itself. Such as documentation in the form of a description of the dataset etc.
DATASET_META: t.Optional[dict] = {
    'version': '0.1.0',
    # A list of strings where each element is a description about the changes introduced in a newer
    # version of the dataset.
    'changelog': [
        '0.1.0 - 29.01.2023 - initial version'
    ],
    # A general description about the dataset, which gives a general overview about where the data was
    # sampled from, what the input features look like, what the prediction target is etc...
    'description': (
        'Small dataset consisting of molecular graphs, where the target is the measured logS value of '
        'the molecules solubility in Benzene.'
    ),
    # A list of informative strings (best case containing URLS) which are used as references for the
    # dataset. This could for example be a reference to a paper where the dataset was first introduced
    # or a link to site where the raw data can be downloaded etc.
    'references': [
        'Library used for the processing and visualization of molecules. https://www.rdkit.org/',
    ],
    # A small description about how to interpret the visualizations which were created by this dataset.
    'visualization_description': (
        'Molecular graphs generated by RDKit based on the SMILES representation of the molecule.'
    ),
    # A dictionary, where the keys should be the integer indices of the target value vector for the dataset
    # and the values should be string descriptions of what the corresponding target value is about.
    'target_descriptions': {
        0: 'measured logS values of the molecules solubility in Benzene. (unprocessed)'
    }
}

# == EVALUATION PARAMETERS ==
EVAL_LOG_STEP = 100
NUM_BINS = 10
PLOT_COLOR = 'gray'

# == EXPERIMENT PARAMETERS ==
DEBUG = True
BASE_PATH = os.getcwd()
NAMESPACE = 'results/generate_molecule_dataset_from_csv'
with Skippable(), (e := Experiment(base_path=BASE_PATH, namespace=NAMESPACE, glob=globals())):

    # Here we provide the possibility for sub experiments to add more specific filters for their purposes
    # by using a hook.
    FILTER_CALLBACKS = e.apply_hook(
        'modify_filter_callbacks',
        filter_callbacks=FILTER_CALLBACKS,
        default=FILTER_CALLBACKS
    )

    e.info('generating a molecule visual graph dataset from CSV source file...')
    config = Config()
    config.load()

    @e.hook('load_data', default=True)
    def load_data(_e, config):
        # -- get source dataset --
        # If the given path is not a valid file on the local system, we will interpret it as a relative
        # path to download from the file share provider.
        if os.path.exists(_e.parameters['CSV_FILE_NAME']):
            file_path = _e.parameters['CSV_FILE_NAME']
            _e.info('CSV found on local system')
        else:
            _e.info('downloading from remote file share...')
            file_share: AbstractFileShare = get_file_share(config, _e.parameters['FILE_SHARE_PROVIDER'])
            file_path = file_share.download_file(_e.parameters['CSV_FILE_NAME'], _e.path)
            _e.info('CSV downloaded from file share')

        # -- Load that data into the required format --
        raw_data_list = []
        with open(file_path) as file:
            dict_reader = csv.DictReader(file)
            for c, row in enumerate(dict_reader):
                smiles = row[_e.p['SMILES_COLUMN_NAME']]

                # 23.02.2023
                # To extend the experiment to also work with classification datasets, I had to change
                # the parameter with the column names to be a list instead of just a single value.
                targets = []
                for column_name in _e.p['TARGET_COLUMN_NAMES']:
                    value = row[column_name]
                    if _e.p['TARGET_TYPE'] == 'classification':
                        # For a classification dataset, we expect each of the given columns to represent one
                        # class that is present in the dataset. The value of that column for every element
                        # of the dataset should be some sort of boolean indication of whether that element
                        # belongs to that class.
                        value = int(value)

                    targets.append(value)

                data = {
                    'smiles': smiles,
                    'target': np.array(targets),
                    'data': row,
                }
                raw_data_list.append(data)

        return raw_data_list

    # The end result of the "data loading" process should be a list of dictionaries, where each dictionary
    # represents one item in the dataset. The content of these dicts may be arbitrary to some point,
    # depending on how sub-experiments may extend the functionality, but these should at least contain they
    # should at least contain the following elements:
    # - smiles: the SMILES string representation
    # - target: the target value for that element
    # - data: a dictionary containing additional data loaded from the CSV.
    e.info(f'loading data...')
    raw_data_list: t.List[dict] = e.apply_hook(
        'load_data',
        config=config
    )
    dataset_length = len(raw_data_list)
    e.info(f'loaded data with {dataset_length} elements')

    # -- Processing the dataset into visual graph dataset --
    e.info('creating the dataset folder...')
    dataset_path = os.path.join(e.path, DATASET_NAME)
    os.mkdir(dataset_path)
    e['dataset_path'] = dataset_path

    e.info('creating visual graph dataset...')
    # 24.02.2023
    # In this dictionary we want to use the smiles identifiers of the elements as the keys and the values
    # should be short reasons why these elements were omitted from the final dataset. In the end we then
    # want to save this information as a json file into the archive folder as an artifact.
    omitted_elements: t.Dict[str, str] = {}

    start_time = time.time()
    index: int = 0
    for d in raw_data_list:
        smiles = d['smiles']
        # ~ Convert the smiles string into a molecule
        try:
            mol = mol_from_smiles(smiles)
            d['mol'] = mol
        except Exception as exc:
            print(f' ! Error converting smiles "{smiles}" to mol: {exc}, skipping...')
            continue

        # ~ Convert the Mol object into a GraphDict
        data: dict = {**d['data'], **d}

        # As the first thing we are going to apply the filters to check if the current element is even
        # a valid element according to the rules defined by those callbacks
        skip = False
        for cb in FILTER_CALLBACKS:
            if cb(mol, data):
                message = f' ! skipping "{data[SMILES_COLUMN_NAME]}" due to filter "{cb.__name__}"'
                omitted_elements[smiles] = message
                e.info(message)

                skip = True
                break

        if skip:
            continue

        target = [float(v) for v in d['target']]

        # 23.03.2023 - This is an instance of a ProcessingBase subclass. This class is specifically designed
        # to wrap all the functionality which is needed to create a valid VGD element representation given
        # only the SMILES string representation of a molecule.
        # This method will already create the two required files: The visualization PNG and the metadata
        # JSON file.
        PROCESSING.create(
            smiles=smiles,
            index=str(index),
            name=smiles,
            double_edges_undirected=UNDIRECTED_EDGES_AS_TWO,
            use_node_coordinates=USE_NODE_COORDINATES,
            width=IMAGE_WIDTH,
            height=IMAGE_HEIGHT,
            create_svg=False,
            output_path=dataset_path,
            graph_labels=target,
        )

        metadata_path = os.path.join(dataset_path, f'{index}.json')
        with open(metadata_path) as file:
            content = file.read()
            metadata = json.loads(content)

        # Optionally, if defined, we also add the information about the train test splits to the metadata
        # of the element.
        if len(SPLIT_COLUMN_NAMES) != 0:
            # If the element defined in the column of the given name is a 1 then this indicates that it is
            # supposed to be a training element. If it is 0 then it is supposed to be a test element.
            metadata['train_indices'] = [index
                                         for index, name in SPLIT_COLUMN_NAMES.items()
                                         if name in data and int(data[name]) == 1]

            metadata['test_indices'] = [index
                                        for index, name in SPLIT_COLUMN_NAMES.items()
                                        if name in data and int(data[name]) == 0]

        # But there can also be custom entries which are defined as callbacks in this dictionary. These
        # values will be associated with the same string keys, which are also used in the callbacks dict
        for name, callback in GRAPH_METADATA_CALLBACKS.items():
            metadata[name] = callback(mol, data)

        # Saving the metadata
        with open(metadata_path, mode='w') as file:
            content = json.dumps(metadata, cls=NumericJsonEncoder)
            file.write(content)

        # In regular intervals we will print how it's currently going aka how many elements have already
        # been processed in what time and how much more time is going to be approximately needed.
        if index % EVAL_LOG_STEP == 0:
            elapsed_time = time.time() - start_time
            time_per_element = elapsed_time / (index+1)
            remaining_elements = dataset_length - index
            remaining_time = time_per_element * remaining_elements
            eta = datetime.datetime.now() + datetime.timedelta(seconds=remaining_time)
            e.info(f' * {index}/{dataset_length} elements created'
                   f' - elapsed time: {elapsed_time:.2f}s'
                   f' - remaining time: {remaining_time:.2f}s'
                   f' - eta: {eta:%a %H:%M}')

        index += 1

    e.commit_json('omitted_elements.json', omitted_elements)
    e.info(f'created {index} out of {dataset_length} original elements. '
           f'The rest was skipped either due to errors or filter exclusions.')

    e.info(f'loading dataset...')
    metadata_map, index_data_map = load_visual_graph_dataset(
        e['dataset_path'],
        logger=e.logger,
        log_step=EVAL_LOG_STEP,
        metadata_contains_index=True
    )

    # 21.03.2023 - This was essentially the whole point of the preprocessing update: This function will
    # automatically generate the code for a python standalone python module which contains all the
    # functionality to convert any given SMILES (domain-specific representation of a molecule) into a
    # valid visual graph dataset element representation (graph representation + visualization).
    e.info(f'generating pre-processing python module for {PROCESSING.__class__}...')
    module_code = create_processing_module(PROCESSING)
    module_path = os.path.join(dataset_path, 'process.py')
    with open(module_path, mode='w') as file:
        file.write(module_code)

    e.info('generating dataset metadata...')
    metadata_map.update(generate_visual_graph_dataset_metadata(index_data_map))
    metadata_map.update(DATASET_META)

    # 21.03.2023 - Another cool feature of the Processing class: This method will automatically create a
    # dictionary that contains natural language descriptions for all the elements of the node, edge and
    # graph attribute vectors.
    metadata_map.update(PROCESSING.get_description_map())

    yaml_path = os.path.join(dataset_path, '.meta.yml')
    with open(yaml_path, mode='w') as file:
        yaml.dump(metadata_map, file)
        e.info(f'metadata written to: {yaml_path}')

    e.apply_hook('after_experiment')


with Skippable(), e.analysis:
    e.info('attempting to load visual graph dataset...')
    metadata_map, index_data_map = load_visual_graph_dataset(
        e['dataset_path'],
        logger=e.logger,
        log_step=EVAL_LOG_STEP,
        metadata_contains_index=True
    )
    e.info(f'loaded visual graph dataset with {len(index_data_map)} elements')

    # -- Plotting information about the dataset --

    @e.hook('dataset_info', default=True)
    def dataset_info(_e, index_data_map):
        pdf_path = os.path.join(e.path, 'dataset_info.pdf')
        with PdfPages(pdf_path) as pdf:
            _e.info(f'target value distribution...')
            fig, ax = plt.subplots(ncols=1, nrows=1, figsize=(12, 12))
            ax.set_title('Target Value Distribution')
            targets = [d['metadata']['target'][0] for i, d in index_data_map.items()]
            _e.info(f'number of targets: {len(targets)}')
            _e.info(f' * min: {np.min(targets):.2f} - mean: {np.mean(targets)} - max: {np.max(targets):.2f}')
            n, bins, edges = ax.hist(
                targets,
                bins=_e.p['NUM_BINS'],
                color=_e.p['PLOT_COLOR'],
            )
            ax.set_xticks(bins)
            ax.set_xticklabels([round(v, 2) for v in bins])
            pdf.savefig(fig)

            _e.info('graph size distribution...')
            fig, ax = plt.subplots(ncols=1, nrows=1, figsize=(12, 12))
            ax.set_title('Graph Size Distribution')
            sizes = [len(d['metadata']['graph']['node_indices']) for d in index_data_map.values()]
            n, bins, edges = ax.hist(
                sizes,
                bins=_e.p['NUM_BINS'],
                color=_e.p['PLOT_COLOR'],
            )
            ax.set_xticks(bins)
            ax.set_xticklabels([int(v) for v in bins])
            pdf.savefig(fig)


    e.info(f'plotting dataset analyses...')
    e.apply_hook(
        'dataset_info',
        index_data_map=index_data_map,
    )

