"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.LambdaInsightsVersion = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("@aws-cdk/core");
const region_info_1 = require("@aws-cdk/region-info");
// This is the name of the mapping that will be added to the CloudFormation template, if a stack is region agnostic
const DEFAULT_MAPPING_PREFIX = 'LambdaInsightsVersions';
// To add new versions, update fact-tables.ts `CLOUDWATCH_LAMBDA_INSIGHTS_ARNS` and create a new `public static readonly VERSION_A_B_C_D`
/**
 * Version of CloudWatch Lambda Insights.
 *
 * @stability stable
 */
class LambdaInsightsVersion {
    constructor() {
        /**
         * The arn of the Lambda Insights extension.
         *
         * @stability stable
         */
        this.layerVersionArn = '';
    }
    /**
     * Use the insights extension associated with the provided ARN.
     *
     * Make sure the ARN is associated
     * with same region as your function
     *
     * @see https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/Lambda-Insights-extension-versions.html
     * @stability stable
     */
    static fromInsightVersionArn(arn) {
        class InsightsArn extends LambdaInsightsVersion {
            constructor() {
                super(...arguments);
                this.layerVersionArn = arn;
            }
        }
        return new InsightsArn();
    }
    // Use the verison to build the object. Not meant to be called by the user -- user should use e.g. VERSION_1_0_54_0
    static fromInsightsVersion(insightsVersion) {
        // Check if insights version is valid. This should only happen if one of the public static readonly versions are set incorrectly
        const versionExists = region_info_1.RegionInfo.regions.some(regionInfo => regionInfo.cloudwatchLambdaInsightsArn(insightsVersion));
        if (!versionExists) {
            throw new Error(`Insights version ${insightsVersion} does not exist.`);
        }
        class InsightsVersion extends LambdaInsightsVersion {
            constructor() {
                super(...arguments);
                this.layerVersionArn = core_1.Lazy.uncachedString({
                    produce: (context) => getVersionArn(context, insightsVersion),
                });
            }
        }
        return new InsightsVersion();
    }
}
exports.LambdaInsightsVersion = LambdaInsightsVersion;
_a = JSII_RTTI_SYMBOL_1;
LambdaInsightsVersion[_a] = { fqn: "@aws-cdk/aws-lambda.LambdaInsightsVersion", version: "1.117.0" };
/**
 * Version 1.0.54.0.
 *
 * @stability stable
 */
LambdaInsightsVersion.VERSION_1_0_54_0 = LambdaInsightsVersion.fromInsightsVersion('1.0.54.0');
/**
 * Version 1.0.86.0.
 *
 * @stability stable
 */
LambdaInsightsVersion.VERSION_1_0_86_0 = LambdaInsightsVersion.fromInsightsVersion('1.0.86.0');
/**
 * Version 1.0.89.0.
 *
 * @stability stable
 */
LambdaInsightsVersion.VERSION_1_0_89_0 = LambdaInsightsVersion.fromInsightsVersion('1.0.89.0');
/**
 * Version 1.0.98.0.
 *
 * @stability stable
 */
LambdaInsightsVersion.VERSION_1_0_98_0 = LambdaInsightsVersion.fromInsightsVersion('1.0.98.0');
/**
 * Function to retrieve the correct Lambda Insights ARN from RegionInfo,
 * or create a mapping to look it up at stack deployment time.
 *
 * This function is run on CDK synthesis.
 */
function getVersionArn(context, insightsVersion) {
    const scopeStack = core_1.Stack.of(context.scope);
    const region = scopeStack.region;
    // Region is defined, look up the arn, or throw an error if the version isn't supported by a region
    if (region !== undefined && !core_1.Token.isUnresolved(region)) {
        const arn = region_info_1.RegionInfo.get(region).cloudwatchLambdaInsightsArn(insightsVersion);
        if (arn === undefined) {
            throw new Error(`Insights version ${insightsVersion} is not supported in region ${region}`);
        }
        return arn;
    }
    // Otherwise, need to add a mapping to be looked up at deployment time
    /**
     * See this for the context as to why the mappings are the way they are
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/mappings-section-structure.html
     *
     * Mappings have to have a structure like this, and no functions can be used inside them:
     * <Alphanumeric only>
     * - <Can be non-alphanumeric>
     * -- { <alphanumeric>: "value1"},
     * -- { <alphanumeric>: "value2"}
     *
     * So we cannot have an otherwise ideal mapping like this, because '1.0.98.0' is non-alphanumeric:
     * LambdaInsightsVersions
     * - us-east-1
     * -- {'1.0.98.0': 'arn1'},
     * -- {'1.0.89.0': 'arn2'}
     *
     * To get around this limitation, this is the mapping structure:
     * LambdaInsightsVersions10980 // for version 1.0.98.0
     * - us-east-1
     * -- {'arn': 'arn1'},
     * - us-east-2
     * -- {'arn': 'arn2'}
     * LambdaInsightsVersions10890 // a separate mapping version 1.0.89.0
     * - us-east-1
     * -- {'arn': 'arn3'},
     * - us-east-2
     * -- {'arn': 'arn4'}
     */
    const mapName = DEFAULT_MAPPING_PREFIX + insightsVersion.split('.').join('');
    const mapping = {};
    const region2arns = region_info_1.RegionInfo.regionMap(region_info_1.FactName.cloudwatchLambdaInsightsVersion(insightsVersion));
    for (const [reg, arn] of Object.entries(region2arns)) {
        mapping[reg] = { arn };
    }
    // Only create a given mapping once. If another version of insights is used elsewhere, that mapping will also exist
    if (!scopeStack.node.tryFindChild(mapName)) {
        new core_1.CfnMapping(scopeStack, mapName, { mapping });
    }
    // The ARN will be looked up at deployment time from the mapping we created
    return core_1.Fn.findInMap(mapName, core_1.Aws.REGION, 'arn');
}
//# sourceMappingURL=data:application/json;base64,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