"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SmithyBuildProject = void 0;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const path = require("path");
const projen_1 = require("projen");
const smithy_build_1 = require("projen/lib/smithy/smithy-build");
const sample_executable_1 = require("./components/sample-executable");
const smithy_generated_output_1 = require("./components/smithy-generated-output");
/**
 * Creates a project which transforms a Smithy model to OpenAPI
 */
class SmithyBuildProject extends projen_1.Project {
    constructor(options) {
        super(options);
        // Store whether we've synthesized the project
        this.synthed = false;
        // HACK: remove all components but the ones we are registering - removes .gitignore, tasks, etc since these are
        // unused and a distraction for end-users!
        // @ts-ignore
        this._components = [];
        const samplePath = path.join(__dirname, "..", "..", "..", "samples", "smithy");
        // Add all the smithy gradle files, which the user is free to edit
        [
            "build.gradle",
            "gradle/wrapper/gradle-wrapper.jar",
            "gradle/wrapper/gradle-wrapper.properties",
        ].forEach((file) => {
            new projen_1.SampleFile(this, file, {
                sourcePath: path.join(samplePath, file),
            });
        });
        ["gradlew", "gradlew.bat"].forEach((executable) => {
            new sample_executable_1.SampleExecutable(this, executable, {
                sourcePath: path.join(samplePath, executable),
            });
        });
        new projen_1.SampleFile(this, "settings.gradle", {
            contents: `rootProject.name = '${this.name.replace(/[\/\\:<>"?\*|]/g, "-")}'`,
        });
        const { namespace: serviceNamespace, serviceName } = options.serviceName;
        const modelDir = "src/main/smithy";
        // Create the default smithy model
        new projen_1.SampleFile(this, path.join(modelDir, "main.smithy"), {
            contents: `$version: "2"
namespace ${serviceNamespace}

use aws.protocols#restJson1

/// A sample smithy api
@restJson1
service ${serviceName} {
    version: "1.0"
    operations: [SayHello]
}

@readonly
@http(method: "GET", uri: "/hello")
operation SayHello {
    input: SayHelloInput
    output: SayHelloOutput
    errors: [ApiError]
}

string Name
string Message

@input
structure SayHelloInput {
    @httpQuery("name")
    @required
    name: Name
}

@output
structure SayHelloOutput {
    @required
    message: Message
}

@error("client")
structure ApiError {
    @required
    errorMessage: Message
}
`,
        });
        // Create the smithy build json file
        new smithy_build_1.SmithyBuild(this, {
            version: "2.0",
            ...options.smithyBuildOptions,
            projections: {
                ...options.smithyBuildOptions?.projections,
                openapi: {
                    plugins: {
                        openapi: {
                            service: `${serviceNamespace}#${serviceName}`,
                            // By default, preserve tags in the generated spec, but allow users to explicitly overwrite this
                            tags: true,
                            ...options.smithyBuildOptions?.projections?.openapi?.plugins
                                ?.openapi,
                        },
                    },
                },
            },
        });
        // SmithyBuild component above always writes to smithy-build.json
        this.smithyBuildConfigPath = path.join(this.outdir, "smithy-build.json");
        this.smithyBuildOutputPath = path.join(this.outdir, options.buildOutputDir);
        new smithy_generated_output_1.SmithyGeneratedOutput(this, {
            modelPath: path.join(this.outdir, modelDir),
            gradleProjectPath: this.outdir,
            smithyBuildConfigPath: this.smithyBuildConfigPath,
            outputPath: this.smithyBuildOutputPath,
        });
    }
    /**
     * @inheritDoc
     */
    synth() {
        // Save some time by only synthesizing once. We synthesize this project early so that it's available for the parent
        // project's install phase (pre-synth). Projen will call this method again at the usual time to synthesize this,
        // project, at which point we're already done so can skip.
        if (this.synthed) {
            return;
        }
        super.synth();
        this.synthed = true;
    }
}
exports.SmithyBuildProject = SmithyBuildProject;
//# sourceMappingURL=data:application/json;base64,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