"use strict";
/**
 *  Copyright 2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.LambdaToDynamoDB = void 0;
const dynamodb = require("@aws-cdk/aws-dynamodb");
const defaults = require("@aws-solutions-constructs/core");
const core_1 = require("@aws-cdk/core");
class LambdaToDynamoDB extends core_1.Construct {
    /**
     * @summary Constructs a new instance of the LambdaToDynamoDB class.
     * @param {cdk.App} scope - represents the scope for all the resources.
     * @param {string} id - this is a a scope-unique id.
     * @param {LambdaToDynamoDBProps} props - user provided props for the construct
     * @since 0.8.0
     * @access public
     */
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        this.lambdaFunction = defaults.buildLambdaFunction(this, {
            existingLambdaObj: props.existingLambdaObj,
            lambdaFunctionProps: props.lambdaFunctionProps
        });
        this.dynamoTable = defaults.buildDynamoDBTable(this, {
            dynamoTableProps: props.dynamoTableProps,
            existingTableObj: props.existingTableObj
        });
        this.lambdaFunction.addEnvironment('DDB_TABLE_NAME', this.dynamoTable.tableName);
        // Add the requested or default table permissions
        if (props.tablePermissions) {
            const _permissions = props.tablePermissions.toUpperCase();
            if (_permissions === 'ALL') {
                this.dynamoTable.grantFullAccess(this.lambdaFunction.grantPrincipal);
            }
            else if (_permissions === 'READ') {
                this.dynamoTable.grantReadData(this.lambdaFunction.grantPrincipal);
            }
            else if (_permissions === 'READWRITE') {
                this.dynamoTable.grantReadWriteData(this.lambdaFunction.grantPrincipal);
            }
            else if (_permissions === 'WRITE') {
                this.dynamoTable.grantWriteData(this.lambdaFunction.grantPrincipal);
            }
        }
        else {
            this.dynamoTable.grantReadWriteData(this.lambdaFunction.grantPrincipal);
        }
        // Conditional metadata for cfn_nag
        if (((_a = props.dynamoTableProps) === null || _a === void 0 ? void 0 : _a.billingMode) === dynamodb.BillingMode.PROVISIONED) {
            const cfnTable = this.dynamoTable.node.findChild('Resource');
            cfnTable.cfnOptions.metadata = {
                cfn_nag: {
                    rules_to_suppress: [{
                            id: 'W73',
                            reason: `PROVISIONED billing mode is a default and is not explicitly applied as a setting.`
                        }]
                }
            };
        }
    }
}
exports.LambdaToDynamoDB = LambdaToDynamoDB;
//# sourceMappingURL=data:application/json;base64,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