"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ServerDeploymentGroup = exports.InstanceTagSet = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const s3 = require("@aws-cdk/aws-s3");
const cdk = require("@aws-cdk/core");
const core_1 = require("@aws-cdk/core");
const codedeploy_generated_1 = require("../codedeploy.generated");
const utils_1 = require("../utils");
const application_1 = require("./application");
const deployment_config_1 = require("./deployment-config");
const load_balancer_1 = require("./load-balancer");
/**
 * Represents a reference to a CodeDeploy EC2/on-premise Deployment Group.
 *
 * If you're managing the Deployment Group alongside the rest of your CDK resources,
 * use the {@link ServerDeploymentGroup} class.
 *
 * If you want to reference an already existing Deployment Group,
 * or one defined in a different CDK Stack,
 * use the {@link #import} method.
 */
class ServerDeploymentGroupBase extends cdk.Resource {
    constructor(scope, id, deploymentConfig, props) {
        super(scope, id, props);
        this.deploymentConfig = deploymentConfig || deployment_config_1.ServerDeploymentConfig.ONE_AT_A_TIME;
    }
}
class ImportedServerDeploymentGroup extends ServerDeploymentGroupBase {
    constructor(scope, id, props) {
        super(scope, id, props.deploymentConfig);
        this.role = undefined;
        this.autoScalingGroups = undefined;
        this.application = props.application;
        this.deploymentGroupName = props.deploymentGroupName;
        this.deploymentGroupArn = utils_1.arnForDeploymentGroup(props.application.applicationName, props.deploymentGroupName);
    }
}
/**
 * Represents a set of instance tag groups.
 * An instance will match a set if it matches all of the groups in the set -
 * in other words, sets follow 'and' semantics.
 * You can have a maximum of 3 tag groups inside a set.
 */
class InstanceTagSet {
    constructor(...instanceTagGroups) {
        if (instanceTagGroups.length > 3) {
            throw new Error('An instance tag set can have a maximum of 3 instance tag groups, ' +
                `but ${instanceTagGroups.length} were provided`);
        }
        this._instanceTagGroups = instanceTagGroups;
    }
    get instanceTagGroups() {
        return this._instanceTagGroups.slice();
    }
}
exports.InstanceTagSet = InstanceTagSet;
_a = JSII_RTTI_SYMBOL_1;
InstanceTagSet[_a] = { fqn: "@aws-cdk/aws-codedeploy.InstanceTagSet", version: "1.157.0" };
/**
 * A CodeDeploy Deployment Group that deploys to EC2/on-premise instances.
 * @resource AWS::CodeDeploy::DeploymentGroup
 */
class ServerDeploymentGroup extends ServerDeploymentGroupBase {
    constructor(scope, id, props = {}) {
        var _c;
        super(scope, id, props.deploymentConfig, {
            physicalName: props.deploymentGroupName,
        });
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_codedeploy_ServerDeploymentGroupProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.constructor);
            }
            throw error;
        }
        this.application = props.application || new application_1.ServerApplication(this, 'Application', {
            applicationName: props.deploymentGroupName === cdk.PhysicalName.GENERATE_IF_NEEDED ? cdk.PhysicalName.GENERATE_IF_NEEDED : undefined,
        });
        this.role = props.role || new iam.Role(this, 'Role', {
            assumedBy: new iam.ServicePrincipal('codedeploy.amazonaws.com'),
            managedPolicies: [iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSCodeDeployRole')],
        });
        this._autoScalingGroups = props.autoScalingGroups || [];
        this.installAgent = (_c = props.installAgent) !== null && _c !== void 0 ? _c : true;
        this.codeDeployBucket = s3.Bucket.fromBucketName(this, 'Bucket', `aws-codedeploy-${cdk.Stack.of(this).region}`);
        for (const asg of this._autoScalingGroups) {
            this.addCodeDeployAgentInstallUserData(asg);
        }
        this.alarms = props.alarms || [];
        const resource = new codedeploy_generated_1.CfnDeploymentGroup(this, 'Resource', {
            applicationName: this.application.applicationName,
            deploymentGroupName: this.physicalName,
            serviceRoleArn: this.role.roleArn,
            deploymentConfigName: props.deploymentConfig &&
                props.deploymentConfig.deploymentConfigName,
            autoScalingGroups: cdk.Lazy.list({ produce: () => this._autoScalingGroups.map(asg => asg.autoScalingGroupName) }, { omitEmpty: true }),
            loadBalancerInfo: this.loadBalancerInfo(props.loadBalancer),
            deploymentStyle: props.loadBalancer === undefined
                ? undefined
                : {
                    deploymentOption: 'WITH_TRAFFIC_CONTROL',
                },
            ec2TagSet: this.ec2TagSet(props.ec2InstanceTags),
            onPremisesTagSet: this.onPremiseTagSet(props.onPremiseInstanceTags),
            alarmConfiguration: cdk.Lazy.any({ produce: () => utils_1.renderAlarmConfiguration(this.alarms, props.ignorePollAlarmsFailure) }),
            autoRollbackConfiguration: cdk.Lazy.any({ produce: () => utils_1.renderAutoRollbackConfiguration(this.alarms, props.autoRollback) }),
        });
        this.deploymentGroupName = this.getResourceNameAttribute(resource.ref);
        this.deploymentGroupArn = this.getResourceArnAttribute(utils_1.arnForDeploymentGroup(this.application.applicationName, resource.ref), {
            service: 'codedeploy',
            resource: 'deploymentgroup',
            resourceName: `${this.application.applicationName}/${this.physicalName}`,
            arnFormat: core_1.ArnFormat.COLON_RESOURCE_NAME,
        });
    }
    /**
     * Import an EC2/on-premise Deployment Group defined either outside the CDK app,
     * or in a different region.
     *
     * @param scope the parent Construct for this new Construct
     * @param id the logical ID of this new Construct
     * @param attrs the properties of the referenced Deployment Group
     * @returns a Construct representing a reference to an existing Deployment Group
     */
    static fromServerDeploymentGroupAttributes(scope, id, attrs) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_codedeploy_ServerDeploymentGroupAttributes(attrs);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromServerDeploymentGroupAttributes);
            }
            throw error;
        }
        return new ImportedServerDeploymentGroup(scope, id, attrs);
    }
    /**
     * Adds an additional auto-scaling group to this Deployment Group.
     *
     * @param asg the auto-scaling group to add to this Deployment Group.
     * [disable-awslint:ref-via-interface] is needed in order to install the code
     * deploy agent by updating the ASGs user data.
     */
    addAutoScalingGroup(asg) {
        this._autoScalingGroups.push(asg);
        this.addCodeDeployAgentInstallUserData(asg);
    }
    /**
     * Associates an additional alarm with this Deployment Group.
     *
     * @param alarm the alarm to associate with this Deployment Group
     */
    addAlarm(alarm) {
        this.alarms.push(alarm);
    }
    get autoScalingGroups() {
        return this._autoScalingGroups.slice();
    }
    validate() {
        return utils_1.validateName('Deployment group', this.physicalName);
    }
    addCodeDeployAgentInstallUserData(asg) {
        if (!this.installAgent) {
            return;
        }
        this.codeDeployBucket.grantRead(asg, 'latest/*');
        switch (asg.osType) {
            case ec2.OperatingSystemType.LINUX:
                asg.addUserData('set +e', // make sure we don't exit on the `which` failing
                'PKG_CMD=`which yum 2>/dev/null`', 'set -e', // continue with failing on error
                'if [ -z "$PKG_CMD" ]; then', 'PKG_CMD=apt-get', 'else', 'PKG_CMD=yum', 'fi', '$PKG_CMD update -y', 'set +e', // make sure we don't exit on the next command failing (we check its exit code below)
                '$PKG_CMD install -y ruby2.0', 'RUBY2_INSTALL=$?', 'set -e', // continue with failing on error
                'if [ $RUBY2_INSTALL -ne 0 ]; then', '$PKG_CMD install -y ruby', 'fi', 'AWS_CLI_PACKAGE_NAME=awscli', 'if [ "$PKG_CMD" = "yum" ]; then', 'AWS_CLI_PACKAGE_NAME=aws-cli', 'fi', '$PKG_CMD install -y $AWS_CLI_PACKAGE_NAME', 'TMP_DIR=`mktemp -d`', 'cd $TMP_DIR', `aws s3 cp s3://aws-codedeploy-${cdk.Stack.of(this).region}/latest/install . --region ${cdk.Stack.of(this).region}`, 'chmod +x ./install', './install auto', 'rm -fr $TMP_DIR');
                break;
            case ec2.OperatingSystemType.WINDOWS:
                asg.addUserData('Set-Variable -Name TEMPDIR -Value (New-TemporaryFile).DirectoryName', `aws s3 cp s3://aws-codedeploy-${cdk.Stack.of(this).region}/latest/codedeploy-agent.msi $TEMPDIR\\codedeploy-agent.msi`, 'cd $TEMPDIR', '.\\codedeploy-agent.msi /quiet /l c:\\temp\\host-agent-install-log.txt');
                break;
        }
    }
    loadBalancerInfo(loadBalancer) {
        if (!loadBalancer) {
            return undefined;
        }
        switch (loadBalancer.generation) {
            case load_balancer_1.LoadBalancerGeneration.FIRST:
                return {
                    elbInfoList: [
                        { name: loadBalancer.name },
                    ],
                };
            case load_balancer_1.LoadBalancerGeneration.SECOND:
                return {
                    targetGroupInfoList: [
                        { name: loadBalancer.name },
                    ],
                };
        }
    }
    ec2TagSet(tagSet) {
        if (!tagSet || tagSet.instanceTagGroups.length === 0) {
            return undefined;
        }
        return {
            ec2TagSetList: tagSet.instanceTagGroups.map(tagGroup => {
                return {
                    ec2TagGroup: this.tagGroup2TagsArray(tagGroup),
                };
            }),
        };
    }
    onPremiseTagSet(tagSet) {
        if (!tagSet || tagSet.instanceTagGroups.length === 0) {
            return undefined;
        }
        return {
            onPremisesTagSetList: tagSet.instanceTagGroups.map(tagGroup => {
                return {
                    onPremisesTagGroup: this.tagGroup2TagsArray(tagGroup),
                };
            }),
        };
    }
    tagGroup2TagsArray(tagGroup) {
        const tagsInGroup = new Array();
        for (const tagKey in tagGroup) {
            if (tagGroup.hasOwnProperty(tagKey)) {
                const tagValues = tagGroup[tagKey];
                if (tagKey.length > 0) {
                    if (tagValues.length > 0) {
                        for (const tagValue of tagValues) {
                            tagsInGroup.push({
                                key: tagKey,
                                value: tagValue,
                                type: 'KEY_AND_VALUE',
                            });
                        }
                    }
                    else {
                        tagsInGroup.push({
                            key: tagKey,
                            type: 'KEY_ONLY',
                        });
                    }
                }
                else {
                    if (tagValues.length > 0) {
                        for (const tagValue of tagValues) {
                            tagsInGroup.push({
                                value: tagValue,
                                type: 'VALUE_ONLY',
                            });
                        }
                    }
                    else {
                        throw new Error('Cannot specify both an empty key and no values for an instance tag filter');
                    }
                }
            }
        }
        return tagsInGroup;
    }
}
exports.ServerDeploymentGroup = ServerDeploymentGroup;
_b = JSII_RTTI_SYMBOL_1;
ServerDeploymentGroup[_b] = { fqn: "@aws-cdk/aws-codedeploy.ServerDeploymentGroup", version: "1.157.0" };
//# sourceMappingURL=data:application/json;base64,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