"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.PipelineWithChangeControl = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_codebuild_1 = require("aws-cdk-lib/aws-codebuild");
const aws_codepipeline_1 = require("aws-cdk-lib/aws-codepipeline");
const aws_codepipeline_actions_1 = require("aws-cdk-lib/aws-codepipeline-actions");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const constructs_1 = require("constructs");
const change_controller_1 = require("../time-windows/change-controller/change-controller");
/**
 * A pipeline with a change controller.
 */
class PipelineWithChangeControl extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const sourceOutput = new aws_codepipeline_1.Artifact('SourceOutput');
        const buildOutput = new aws_codepipeline_1.Artifact('BuildOutput');
        const pipeline = new aws_codepipeline_1.Pipeline(this, 'Pipeline', {
            pipelineName: props.pipelineName,
            role: props.pipelineRole,
        });
        // Source Stage
        const sourceStage = pipeline.addStage({
            stageName: 'Source',
        });
        const sourceAction = new aws_codepipeline_actions_1.CodeCommitSourceAction({
            actionName: 'CodeCommitSource',
            branch: 'main',
            trigger: aws_codepipeline_actions_1.CodeCommitTrigger.EVENTS,
            repository: props.sourceRepository,
            output: sourceOutput,
        });
        sourceStage.addAction(sourceAction);
        // Build stage
        const buildStage = pipeline.addStage({
            stageName: 'Build',
        });
        const buildProject = new aws_codebuild_1.PipelineProject(this, 'BuildStageProject', {
            buildSpec: aws_codebuild_1.BuildSpec.fromObject({
                version: '0.2',
                phases: {
                    install: {
                        'runtime-versions': {
                            nodejs: 16,
                        },
                        // eslint-disable-next-line quote-props
                        commands: ['yarn install'],
                    },
                    build: {
                        commands: [
                            'echo Build started on `date`',
                            'yarn build',
                            'yarn cdk synth',
                        ],
                    },
                    post_build: {
                        commands: ['echo Build completed on `date`'],
                    },
                },
                artifacts: {
                    files: ['cdk.out/**/*', 'node_modules/**/*', 'lib/*'],
                },
            }),
            environment: {
                buildImage: aws_codebuild_1.LinuxBuildImage.AMAZON_LINUX_2_4,
                privileged: true,
                computeType: aws_codebuild_1.ComputeType.SMALL,
            },
        });
        const buildAction = new aws_codepipeline_actions_1.CodeBuildAction({
            actionName: 'BuildAction',
            project: buildProject,
            input: sourceOutput,
            outputs: [buildOutput],
        });
        buildStage.addAction(buildAction);
        // Deploy Stage
        const deployStage = pipeline.addStage({
            stageName: 'Deploy',
        });
        // CodeBuild Deploy Project
        const deployProject = new aws_codebuild_1.PipelineProject(this, 'DeployStageProject', {
            buildSpec: aws_codebuild_1.BuildSpec.fromObject({
                version: '0.2',
                phases: {
                    install: {
                        'runtime-versions': {
                            nodejs: 16,
                        },
                        // eslint-disable-next-line quote-props
                        commands: ['npm install -g aws-cdk'],
                    },
                    build: {
                        commands: [
                            'echo Deploy started on `date`',
                            'cdk deploy --app cdk.out',
                        ],
                    },
                    post_build: {
                        commands: ['echo Deploy completed on `date`'],
                    },
                },
            }),
            environment: {
                buildImage: aws_codebuild_1.LinuxBuildImage.AMAZON_LINUX_2_4,
                privileged: true,
                computeType: aws_codebuild_1.ComputeType.SMALL,
            },
        });
        const policyStatement = new aws_iam_1.PolicyStatement({
            actions: ['cloudformation:DescribeStacks', 'ssm:GetParameter'],
            resources: ['*'],
        });
        const cdkDeployRoleArn = aws_cdk_lib_1.Arn.format({
            service: 'iam',
            resource: 'role',
            resourceName: 'cdk-hnb659fds-deploy-role-*',
            region: '',
            arnFormat: aws_cdk_lib_1.ArnFormat.SLASH_RESOURCE_NAME,
        }, aws_cdk_lib_1.Stack.of(this));
        const cdkLookupRoleArn = aws_cdk_lib_1.Arn.format({
            service: 'iam',
            resource: 'role',
            resourceName: 'cdk-hnb659fds-lookup-role-*',
            region: '',
            arnFormat: aws_cdk_lib_1.ArnFormat.SLASH_RESOURCE_NAME,
        }, aws_cdk_lib_1.Stack.of(this));
        const cdkPublishingRoleArn = aws_cdk_lib_1.Arn.format({
            service: 'iam',
            resource: 'role',
            resourceName: 'cdk-hnb659fds-file-publishing-role-*',
            region: '',
            arnFormat: aws_cdk_lib_1.ArnFormat.SLASH_RESOURCE_NAME,
        }, aws_cdk_lib_1.Stack.of(this));
        const assumeRole = new aws_iam_1.PolicyStatement({
            actions: ['sts:AssumeRole'],
            resources: [cdkDeployRoleArn, cdkLookupRoleArn, cdkPublishingRoleArn],
        });
        deployProject.addToRolePolicy(policyStatement);
        deployProject.addToRolePolicy(assumeRole);
        // CodeBuild Deploy Action
        const deployAction = new aws_codepipeline_actions_1.CodeBuildAction({
            actionName: 'DeployAction',
            project: deployProject,
            input: buildOutput,
        });
        // Add Deploy Action to Stage
        deployStage.addAction(deployAction);
        // Add Change Controller
        props.changeControlCalendar._bind(this);
        pipeline.stages.forEach((stage) => {
            new change_controller_1.ChangeController(this, `change-controller-${stage.stageName}`, {
                calendar: props.changeControlCalendar,
                stage,
                schedule: props.changeControlCheckSchedule,
                searchTerms: props.searchTerms,
            });
        });
    }
}
exports.PipelineWithChangeControl = PipelineWithChangeControl;
_a = JSII_RTTI_SYMBOL_1;
PipelineWithChangeControl[_a] = { fqn: "@cdklabs/cdk-codepipeline-extensions.PipelineWithChangeControl", version: "0.0.117" };
//# sourceMappingURL=data:application/json;base64,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