"""
Events.py

Defines events such as tap, swipe, etc. to be sent to devices. Each event has a get_cmd_str method
which returns the necessary arguments to send the event in an `adb shell` context.
"""

from typing import List, Type, Union, Tuple

import re

from .Event import Event

# User-defined types:
# any instantiated object that derives from Event
EventInstance = Union["Tap", "LongTap", "Swipe", "Text", "EnterKey", "PowerButton"]
# any uninstantied class type that derives from Event
EventClass = Union[Type["Tap"], Type["LongTap"], Type["Swipe"], Type["Text"], Type["EnterKey"],
        Type["PowerButton"]]

# TODO: modify events so they use the 'sendevent' adb shell command; this makes it possible to send
# events asynchronously in an efficient manner

def get_recorded_events(filepath: str = "events.txt") -> List[Tuple[int, EventInstance]]:
    """
    Given a filepath which points to a file generated by DeviceConnector.record_events(),
    returns a list of tuples formatted like (timestamp, event). The filepath defaults to
    "events.txt". Note that the list of tuples will be ordered according to the ordering in the
    file, not necessarily by the order of the events' timestamps.
    """
    
    regexes = {
        Tap:         r"^\[(\d+)\] TAP (\d+) (\d+)$",
        LongTap:     r"^\[(\d+)\] LONG_TAP (\d+) (\d+) (\d+)$",
        Swipe:       r"^\[(\d+)\] SWIPE (\d+) (\d+) (-?\d+) (-?\d+) (\d+)$",
        Text:        r'^\[(\d+)\] TEXT "(.+)"$',
        EnterKey:    r"^\[(\d+)\] ENTER_KEY$",
        PowerButton: r"^\[(\d+)\] POWER_BUTTON$"
    }

    event_infos = []

    with open(filepath, "r") as file:

        for line in file:

            line = line.strip()

            for event_class in regexes:

                captured_groups = re.findall(regexes[event_class], line)

                if len(captured_groups) > 0:

                    if event_class in (PowerButton, EnterKey):
                        event_infos.append((int(captured_groups[0]), event_class()))
                    
                    elif event_class == Text:
                        event_infos.append((int(captured_groups[0][0]),
                                event_class(captured_groups[0][1])))
                    
                    else:
                        event_infos.append((int(captured_groups[0][0]),
                                event_class(*(int(arg) for arg in captured_groups[0][1:]))))
    
    return event_infos

class Tap(Event):

    x: int = None
    y: int = None

    def __init__(self, x: int, y: int) -> None:
        """
        A short tap at pixel location (x, y)
        """

        self.x = x
        self.y = y

        self.cmd_str = "input tap " + str(self.x) + " " + str(self.y)
    
    def __str__(self) -> str:
        return "TAP " + str(self.x) + " " + str(self.y)

class LongTap(Event):

    x: int = None
    y: int = None
    duration: int = None

    def __init__(self, x: int, y: int, duration: int) -> None:
        """
        A long tap at pixel location (x, y) lasting the given duration milliseconds
        """

        self.x = x
        self.y = y
        self.duration = duration

        x = str(self.x)
        y = str(self.y)

        self.cmd_str = "input swipe " + x + " " + y + " " + x + " " + y + " " + str(self.duration)
    
    def __str__(self) -> str:
        return "LONG_TAP " + str(self.x) + " " + str(self.y) + " " + str(self.duration)

class Swipe(Event):

    # TODO: make it so that a swipe is a series of coordinates rather than just a start and end

    x: int = None
    y: int = None
    dx: int = None
    dy: int = None
    duration: int = None

    def __init__(self, x: int, y: int, dx: int, dy: int, duration: int) -> None:
        """
        A swipe starting from pixel location (x, y) and moving (dx, dy) pixels, lasting for the
        given duration in milliseconds
        """

        self.x = x
        self.y = y
        self.dx = dx
        self.dy = dy
        self.duration = duration

        self.cmd_str = ("input swipe " + str(self.x) + " " + str(self.y) + " " +
                str(self.x + self.dx) + " " + str(self.y + self.dy) + " " + str(self.duration))
    
    def __str__(self) -> str:
        return ("SWIPE " + str(self.x) + " " + str(self.y) + " " + str(self.dx) + " " +
                str(self.dy) + " " + str(self.duration))

class Text(Event):

    text: str = None

    def __init__(self, text: str) -> None:
        """
        Types the given text to the device. No whitespace allowed except for a literal space.
        """

        # only whitespace that's allowed is a space
        if re.search(r"\s(?:(?<! ))", text):
            raise ValueError('Invalid text: "' + repr(text)[1:-1] + '"')

        self.text = text

        # build the cmd_str so that it's compatible with bash shell
        self.cmd_str = "input text '"
        for char in self.text:
            if char == "'":
                self.cmd_str += "'\\'"
            self.cmd_str += char
        self.cmd_str += "'"
    
    def __str__(self) -> str:
        return 'TEXT "' + repr(self.text)[1:-1] + '"'

class EnterKey(Event):

    def __init__(self) -> None:
        self.cmd_str = "input keyevent 66"
    
    def __str__(self) -> str:
        return "ENTER_KEY"

class PowerButton(Event):

    def __init__(self) -> None:
        self.cmd_str = "input keyevent 26"
    
    def __str__(self) -> str:
        return "POWER_BUTTON"
