"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.GeneratedPythonClientProject = void 0;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const path = require("path");
const projen_1 = require("projen");
const python_1 = require("projen/lib/python");
const generated_python_client_source_code_1 = require("./components/generated-python-client-source-code");
const open_api_generator_ignore_file_1 = require("./components/open-api-generator-ignore-file");
/**
 * Python project containing a python client (and lambda handler wrappers) generated using OpenAPI Generator CLI
 */
class GeneratedPythonClientProject extends python_1.PythonProject {
    constructor(options) {
        super({
            sample: false,
            pytest: false,
            ...options,
        });
        /**
         * The directory in which the built layer is output
         */
        this.layerDistDir = "dist/layer";
        // Store whether we've synthesized the project
        this.synthed = false;
        // With pip and venv (default), it's useful to install our package into the shared venv to make
        // it easier for other packages in the monorepo to take dependencies on this package.
        if ((options.venv ?? true) && (options.pip ?? true)) {
            this.depsManager.installTask.exec("pip install --editable .");
        }
        // Package into a directory that can be used as a lambda layer. This is done as part of install since the end user
        // must control build order in the monorepo via explicit dependencies, and adding here means we can run as part of
        // initial project synthesis which ensures this is created regardless of whether the user has remembered to
        // configure build order.
        if (options.generateLayer) {
            const relativeLayerDir = path.join(".", this.layerDistDir, "python");
            this.depsManager.installTask.exec(`rm -rf ${relativeLayerDir}`);
            this.depsManager.installTask.exec(`pip install . --target ${relativeLayerDir}`);
        }
        // The NX monorepo will rewrite install tasks to install-py to ensure installs can be run sequentially in dependency
        // order. This runs as part of monorepo synthesis which is too late for this project since we synthesize early to
        // ensure the generated client code is available for the install phase of the api project itself. Thus, we rewrite
        // the install tasks ourselves instead.
        if (this.parent && "addImplicitDependency" in this.parent) {
            const installPyTask = this.addTask("install-py");
            this.depsManager.installTask.steps.forEach((step) => this.copyStepIntoTask(step, installPyTask));
            this.depsManager.installTask.reset();
        }
        // Use a package.json to ensure the client is discoverable by nx
        new projen_1.JsonFile(this, "package.json", {
            obj: {
                name: this.name,
                __pdk__: true,
                version: options.version,
                scripts: Object.fromEntries(this.tasks.all.map((task) => [task.name, `npx projen ${task.name}`])),
            },
            readonly: true,
        });
        new open_api_generator_ignore_file_1.OpenApiGeneratorIgnoreFile(this);
        new generated_python_client_source_code_1.GeneratedPythonClientSourceCode(this, {
            specPath: options.specPath,
            invokeGenerator: options.generateClient,
        });
    }
    /**
     * @inheritDoc
     */
    synth() {
        // Save some time by only synthesizing once. We synthesize this project early so that it's available for the parent
        // project's install phase (pre-synth). Projen will call this method again at the usual time to synthesize this,
        // project, at which point we're already done so can skip.
        if (this.synthed) {
            return;
        }
        super.synth();
        this.synthed = true;
    }
    /**
     * Copies the given step into the given task within this project
     * @private
     */
    copyStepIntoTask(step, task) {
        if (step.exec) {
            task.exec(step.exec, { name: step.name, cwd: step.cwd });
        }
        else if (step.say) {
            task.say(step.say, { name: step.name, cwd: step.cwd });
        }
        else if (step.spawn) {
            const stepToSpawn = this.tasks.tryFind(step.spawn);
            if (stepToSpawn) {
                task.spawn(stepToSpawn, { name: step.name, cwd: step.cwd });
            }
        }
        else if (step.builtin) {
            task.builtin(step.builtin);
        }
    }
}
exports.GeneratedPythonClientProject = GeneratedPythonClientProject;
//# sourceMappingURL=data:application/json;base64,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