import * as iam from '@aws-cdk/aws-iam';
import * as kms from '@aws-cdk/aws-kms';
import * as s3 from '@aws-cdk/aws-s3';
import * as sfn from '@aws-cdk/aws-stepfunctions';
import { Construct } from 'constructs';
/**
 * Properties for starting a Query Execution.
 *
 * @stability stable
 */
export interface AthenaStartQueryExecutionProps extends sfn.TaskStateBaseProps {
    /**
     * Query that will be started.
     *
     * @stability stable
     */
    readonly queryString: string;
    /**
     * Unique string string to ensure idempotence.
     *
     * @default - No client request token
     * @stability stable
     */
    readonly clientRequestToken?: string;
    /**
     * Database within which query executes.
     *
     * @default - No query execution context
     * @stability stable
     */
    readonly queryExecutionContext?: QueryExecutionContext;
    /**
     * Configuration on how and where to save query.
     *
     * @default - No result configuration
     * @stability stable
     */
    readonly resultConfiguration?: ResultConfiguration;
    /**
     * Configuration on how and where to save query.
     *
     * @default - No work group
     * @stability stable
     */
    readonly workGroup?: string;
}
/**
 * Start an Athena Query as a Task.
 *
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/connect-athena.html
 * @stability stable
 */
export declare class AthenaStartQueryExecution extends sfn.TaskStateBase {
    private readonly props;
    private static readonly SUPPORTED_INTEGRATION_PATTERNS;
    /**
     * @stability stable
     */
    protected readonly taskMetrics?: sfn.TaskMetricsConfig;
    /**
     * @stability stable
     */
    protected readonly taskPolicies?: iam.PolicyStatement[];
    private readonly integrationPattern;
    /**
     * @stability stable
     */
    constructor(scope: Construct, id: string, props: AthenaStartQueryExecutionProps);
    private createPolicyStatements;
    private renderEncryption;
    /**
     * Provides the Athena start query execution service integration task configuration
     */
    /**
     * @internal
     */
    protected _renderTask(): any;
}
/**
 * Location of query result along with S3 bucket configuration.
 *
 * @see https://docs.aws.amazon.com/athena/latest/APIReference/API_ResultConfiguration.html
 * @stability stable
 */
export interface ResultConfiguration {
    /**
     * S3 path of query results.
     *
     * @default - Query Result Location set in Athena settings for this workgroup
     * @stability stable
     * @example
     *
     * s3://query-results-bucket/folder/
     */
    readonly outputLocation?: s3.Location;
    /**
     * Encryption option used if enabled in S3.
     *
     * @default - SSE_S3 encrpytion is enabled with default encryption key
     * @stability stable
     */
    readonly encryptionConfiguration?: EncryptionConfiguration;
}
/**
 * Encryption Configuration of the S3 bucket.
 *
 * @see https://docs.aws.amazon.com/athena/latest/APIReference/API_EncryptionConfiguration.html
 * @stability stable
 */
export interface EncryptionConfiguration {
    /**
     * Type of S3 server-side encryption enabled.
     *
     * @default EncryptionOption.S3_MANAGED
     * @stability stable
     */
    readonly encryptionOption: EncryptionOption;
    /**
     * KMS key ARN or ID.
     *
     * @default - No KMS key for Encryption Option SSE_S3 and default master key for Encryption Option SSE_KMS and CSE_KMS
     * @stability stable
     */
    readonly encryptionKey?: kms.IKey;
}
/**
 * Encryption Options of the S3 bucket.
 *
 * @see https://docs.aws.amazon.com/athena/latest/APIReference/API_EncryptionConfiguration.html#athena-Type-EncryptionConfiguration-EncryptionOption
 * @stability stable
 */
export declare enum EncryptionOption {
    /**
     * Server side encryption (SSE) with an Amazon S3-managed key.
     *
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingServerSideEncryption.html
     * @stability stable
     */
    S3_MANAGED = "SSE_S3",
    /**
     * Server-side encryption (SSE) with an AWS KMS key managed by the account owner.
     *
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingKMSEncryption.html
     * @stability stable
     */
    KMS = "SSE_KMS",
    /**
     * Client-side encryption (CSE) with an AWS KMS key managed by the account owner.
     *
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingClientSideEncryption.html
     * @stability stable
     */
    CLIENT_SIDE_KMS = "CSE_KMS"
}
/**
 * Database and data catalog context in which the query execution occurs.
 *
 * @see https://docs.aws.amazon.com/athena/latest/APIReference/API_QueryExecutionContext.html
 * @stability stable
 */
export interface QueryExecutionContext {
    /**
     * Name of catalog used in query execution.
     *
     * @default - No catalog
     * @stability stable
     */
    readonly catalogName?: string;
    /**
     * Name of database used in query execution.
     *
     * @default - No database
     * @stability stable
     */
    readonly databaseName?: string;
}
