# -*- coding: utf-8 -*-
from setuptools import setup

modules = \
['pygame_input']
install_requires = \
['pygame>=1.9.0,<2.0.0']

setup_kwargs = {
    'name': 'pygame-input',
    'version': '1.0.0',
    'description': 'pygame-input is a tool to simplify input handling with pygame',
    'long_description': '\n[![Code style: black](https://img.shields.io/badge/code%20style-black-000000.svg)](https://github.com/psf/black)\n[![](https://img.shields.io/badge/build%20system-poetry-ff69b5)](https://python-poetry.org/)\n[![Pypi Version](https://img.shields.io/pypi/v/pygame-input.svg)](https://pypi.python.org/pypi/PygameGUILib)\n[![Python Version](https://img.shields.io/pypi/pyversions/pygame-input.svg)](https://pypi.python.org/pypi/PygameGUILib)\n[![Pypi Downloads](https://img.shields.io/pypi/dm/pygame-input.svg)](https://pypi.python.org/pypi/pygame-input)\n\npygame-input\n========\n\n`pygame-input` will simplify your input handling with pygame\nby providing a simple way to assign callbacks to given key press\nor joystick events.\n\nLook how easy it is to use:\n\n```python\nimport pygame\nfrom pygame_input import Inputs, Button, JoyButton\n\ninputs = Inputs()\ninputs["fire"] = Button(pygame.K_SPACE, JoyButton(1))\ninputs["fire"].on_press_repeated(player.fire, delay=0.1)\n```\n\nThis will call your `player.fire()` function every 0.1 seconds while \nany button, whether it is the space bar or the button one on your\njoystick is pressed.\n    \nFeatures\n--------\n\n- Joystick\n- Boolean input values (ie. *is key pressed ?*) \n    and scalar input values (ie. *how much is the stick on the left ?*)\n- Register callbacks on:\n    - press,\n    - release,\n    - double-press \n    - or all the time\n- Every function and class has a detailed docstring\n    \nWhat `pygame-input` is not (yet ?):\n - handling key modifiers\n - handling mouse input\n - recognising mouse gestures\n - doing gamepad configuration \n    (ie. you need to know which id is the "A" button)\n    \nThough the first two will probably be implemented quite\nsoon, depending on when I need them.\n\nInstallation\n------------\n\nInstall `pygame-input` by running::\n\n    pip install pygame-input\n\nAlternativelly you can just copy `pygame_input.py` into your\ngame folder as the whole code is just one file. Feel free to\nmodify it as much as you need.\n\nUsage\n-----\n\n#### Defining the inputs\nThe first step to be able to to register callbacks \nis to create an `Inputs` object, which is bacically a \npython\'s `dict` with a `trigger` method. We will \ncome back on `trigger` later. So:\n\n```python\nfrom pygame_input import *\ninputs = Inputs()\n```\n    \nThen you can add all the input types in dictionary.\nThere are two input types: `Button` and `Axis`.\n\nA `Button` represents anything that is either\npressed or released, like a key on your keyboard or \na button of your joystick.\n\nAn `Axis` is for anything that takes a value between -1 and 1,\nfor instance, the position of a joystick, but a pair of \nkeys can also be seen as an axis: if we consider the pair of \nleft and right arrows, the axis would have value -1 if the left key\nis pressed, 1 if it is the right and 0 if none or both are pressed. \nThis is very convenient for moving a player.\n\nEnough talking, let\'s add a `Button` to our `inputs`.\n\n```python\ninputs["fire"] = Button(pygame.K_SPACE, pygame.K_RETURN)\n```\n\nThe `Button` takes every posible way to press it as parameters.\nHere we want to trigger the fire input when we press either \nthe space bar or return. Any pygame key code is a valid argument\nbut if we also want joystick support \n\n```python\ninputs["fire"] = Button(pygame.K_SPACE, pygame.K_RETURN, JoyButton(1))\n```\nIt would also trigger if the button 1 on joystick 0 is pressed. \nNote that you need to [initialise the joysticks](https://www.pygame.org/docs/ref/joystick.html) \nyourself. If you want to fire only when the button 1 on the 3rd joystick\nis pressed, you would pass `JoyButton(1, 3)` instead.\n \nOther parameters for buttons can be:\n - `QuitEvent()` that matches the `pygame.QUIT` event.\n - `JoyAxisTrigger(axis, threshold=0.5, above=True, joy_id=0)`\n  that matches when an axis has a value above `threshold` \n  (or below if `above=False`)\n  \nWe will also add an `Axis` so our player can move horizontally.\n\n```python\nimport pygame\nfrom pygame_input import Axis\ninputs["hmove"] = Axis(\n    (pygame.K_LEFT, pygame.K_a), \n    (pygame.K_RIGHT, pygame.K_d),\n    JoyAxis(1),\n)\n```\n\nWhat did you do here ? The two first arguments of `Axis`\nare the negative and positive keys. Negative keys corresponds to a value\nof -1 (think: left/down) and positive to a value of +1 (think: right/up).\nIt accepts a single key code or a list of key codes.\n\nAfter that, can follow any number of `JoyAxis` that correspond \nto an axis on the joystick. The signature of `JoyAxis` is more complex\nbut quite explicit :\n\n```python\n@dataclass(frozen=True)\nclass JoyAxis:\n    axis: int\n    reversed: bool = False\n    """Whether the positive and negative should be reversed."""\n    threshold: float = 0.2\n    """Any value of smaller magnitude will be considered as zero."""\n    sensibility: float = 1.0\n    """Multiply the value by this amount. Useful if a joystick doesn\'t go all the way to +/-1"""\n    joy_id: int = 0\n    """The id used to initialise the joystick."""\n```\n\nIf you don\'t know about dataclasses, don\'t worry and consider that \nthe attributes defined here are the parameters of `JoyAxis()`. Though\nfeel free to check [what dataclasses are](https://realpython.com/python-data-classes/),\nbecause they\'re nice :)\n\n#### Adding callbacks\n\nNow that you know everything about defining the inputs, \nwe can add callbacks to them. Those two steps are different, \nbecause they can happen at different places in the code. \nFor instance inputs definition can be in the main in on a settings screen.\nYou may also want to add callbacks during the player creation\nand also register a callback on the camera when the player moves.\n\nEnough theory. There are 5 ways to add a callback `f` to a `Button`:\n - `always_call(f)`:\n - `on_press(f)`\n - `on_release(f)`\n - `on_double_press(f)`\n - `on_press_repeated(f, delay)`\n \nTheir names are self explanatory. `on_press_repeated` \naccepts a `delay` argument: the callback will be called every `delay` seconds.\n\n**Each callback is always called with one argument: \nthe `Button` or `Axis` that triggered it.** \nThis way you can access the axis `value` or the button\'s `press_time`.\n\nFor an `Axis` there only `always_call(f)` is available, since the others\ndon\'t make sense. Example:\n\n```python\ninputs["fire"].on_press_repeated(player.fire, delay=0.5)\ninputs["hmove"].allways_call(player.horizontal_move)\n```\n\n#### Triggering callbacks\n\nTo trigger the callbacks the only thing that is needed is \nto call `inputs.trigger` with a list of the events \nthat happened since last frame.\n\n```python\nwhile True:\n    # Event handling\n    inputs.trigger(pygame.events.get())\n\n    # Game logic\n    ...\n\n    # Draw everything\n    ...\n\n    pygame.display.update()\n```\n\n**Note**: if you also need to process the events in a different way,\nyou need to convert `pygame.events.get()` to a list first and use the list:\n\n```python\nevents = list(pygame.events.get())\ninputs.trigger(events)\nfor event in events:\n    if event.type == ...:\n        ...\n```\n\nFor more, see the [examples](examples).\n\nContribute\n----------\n\n- Issue Tracker: https://gitlab.com/ddorn/pygame-input/issues\n- Source Code: https://gitlab.com/ddorn/pygame-input\n\nSupport\n-------\n\nIf you are having issues, please let me know.\nYou can open an issue or send me a mail, \nmy email address is on my gitlab profile.\n \nLicense\n-------\n\nThe project is licensed under the MIT license.',
    'author': 'Diego Dorn',
    'author_email': 'pygame-input@lama-corp.space',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://gitlab.com/ddorn/pygame-input',
    'py_modules': modules,
    'install_requires': install_requires,
    'python_requires': '>=3.6,<4.0',
}


setup(**setup_kwargs)
