"use strict";
const logs = require("@aws-cdk/aws-logs");
const s3 = require("@aws-cdk/aws-s3");
const sqs = require("@aws-cdk/aws-sqs");
const cdk = require("@aws-cdk/core");
const _ = require("lodash");
const nodeunit_1 = require("nodeunit");
const lambda = require("../lib");
module.exports = nodeunit_1.testCase({
    'add incompatible layer'(test) {
        // GIVEN
        const stack = new cdk.Stack(undefined, 'TestStack');
        const bucket = new s3.Bucket(stack, 'Bucket');
        const code = new lambda.S3Code(bucket, 'ObjectKey');
        const func = new lambda.Function(stack, 'myFunc', {
            runtime: lambda.Runtime.PYTHON_3_7,
            handler: 'index.handler',
            code,
        });
        const layer = new lambda.LayerVersion(stack, 'myLayer', {
            code,
            compatibleRuntimes: [lambda.Runtime.NODEJS]
        });
        // THEN
        test.throws(() => func.addLayers(layer), /This lambda function uses a runtime that is incompatible with this layer/);
        test.done();
    },
    'add compatible layer'(test) {
        // GIVEN
        const stack = new cdk.Stack(undefined, 'TestStack');
        const bucket = new s3.Bucket(stack, 'Bucket');
        const code = new lambda.S3Code(bucket, 'ObjectKey');
        const func = new lambda.Function(stack, 'myFunc', {
            runtime: lambda.Runtime.PYTHON_3_7,
            handler: 'index.handler',
            code,
        });
        const layer = new lambda.LayerVersion(stack, 'myLayer', {
            code,
            compatibleRuntimes: [lambda.Runtime.PYTHON_3_7]
        });
        // THEN
        // should not throw
        func.addLayers(layer);
        test.done();
    },
    'add compatible layer for deep clone'(test) {
        // GIVEN
        const stack = new cdk.Stack(undefined, 'TestStack');
        const bucket = new s3.Bucket(stack, 'Bucket');
        const code = new lambda.S3Code(bucket, 'ObjectKey');
        const runtime = lambda.Runtime.PYTHON_3_7;
        const func = new lambda.Function(stack, 'myFunc', {
            runtime,
            handler: 'index.handler',
            code,
        });
        const clone = _.cloneDeep(runtime);
        const layer = new lambda.LayerVersion(stack, 'myLayer', {
            code,
            compatibleRuntimes: [clone]
        });
        // THEN
        // should not throw
        func.addLayers(layer);
        test.done();
    },
    'empty inline code is not allowed'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN/THEN
        test.throws(() => new lambda.Function(stack, 'fn', {
            handler: 'foo',
            runtime: lambda.Runtime.NODEJS_10_X,
            code: lambda.Code.fromInline('')
        }), /Lambda inline code cannot be empty/);
        test.done();
    },
    'logGroup is correctly returned'(test) {
        const stack = new cdk.Stack();
        const fn = new lambda.Function(stack, 'fn', {
            handler: 'foo',
            runtime: lambda.Runtime.NODEJS_10_X,
            code: lambda.Code.fromInline('foo'),
        });
        const logGroup = fn.logGroup;
        test.ok(logGroup.logGroupName);
        test.ok(logGroup.logGroupArn);
        test.done();
    },
    'dlq is returned when provided by user'(test) {
        var _a, _b, _c;
        const stack = new cdk.Stack();
        const dlQueue = new sqs.Queue(stack, 'DeadLetterQueue', {
            queueName: 'MyLambda_DLQ',
            retentionPeriod: cdk.Duration.days(14)
        });
        const fn = new lambda.Function(stack, 'fn', {
            handler: 'foo',
            runtime: lambda.Runtime.NODEJS_10_X,
            code: lambda.Code.fromInline('foo'),
            deadLetterQueue: dlQueue,
        });
        const deadLetterQueue = fn.deadLetterQueue;
        test.ok((_a = deadLetterQueue) === null || _a === void 0 ? void 0 : _a.queueArn);
        test.ok((_b = deadLetterQueue) === null || _b === void 0 ? void 0 : _b.queueName);
        test.ok((_c = deadLetterQueue) === null || _c === void 0 ? void 0 : _c.queueUrl);
        test.done();
    },
    'dlq is returned when setup by cdk'(test) {
        var _a, _b, _c;
        const stack = new cdk.Stack();
        const fn = new lambda.Function(stack, 'fn', {
            handler: 'foo',
            runtime: lambda.Runtime.NODEJS_10_X,
            code: lambda.Code.fromInline('foo'),
            deadLetterQueueEnabled: true,
        });
        const deadLetterQueue = fn.deadLetterQueue;
        test.ok((_a = deadLetterQueue) === null || _a === void 0 ? void 0 : _a.queueArn);
        test.ok((_b = deadLetterQueue) === null || _b === void 0 ? void 0 : _b.queueName);
        test.ok((_c = deadLetterQueue) === null || _c === void 0 ? void 0 : _c.queueUrl);
        test.done();
    },
    'dlq is undefined when not setup'(test) {
        const stack = new cdk.Stack();
        const fn = new lambda.Function(stack, 'fn', {
            handler: 'foo',
            runtime: lambda.Runtime.NODEJS_10_X,
            code: lambda.Code.fromInline('foo'),
        });
        const deadLetterQueue = fn.deadLetterQueue;
        test.ok(deadLetterQueue === undefined);
        test.done();
    },
    'one and only one child LogRetention construct will be created'(test) {
        const stack = new cdk.Stack();
        const fn = new lambda.Function(stack, 'fn', {
            handler: 'foo',
            runtime: lambda.Runtime.NODEJS_10_X,
            code: lambda.Code.fromInline('foo'),
            logRetention: logs.RetentionDays.FIVE_DAYS,
        });
        // tslint:disable:no-unused-expression
        // Call logGroup a few times. If more than one instance of LogRetention was created,
        // the second call will fail on duplicate constructs.
        fn.logGroup;
        fn.logGroup;
        fn.logGroup;
        // tslint:enable:no-unused-expression
        test.done();
    },
    'fails when inline code is specified on an incompatible runtime'(test) {
        const stack = new cdk.Stack();
        test.throws(() => new lambda.Function(stack, 'fn', {
            handler: 'foo',
            runtime: lambda.Runtime.PROVIDED,
            code: lambda.Code.fromInline('foo')
        }), /Inline source not allowed for/);
        test.done();
    }
});
//# sourceMappingURL=data:application/json;base64,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