"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.GeneratedTypescriptClientProject = void 0;
/*********************************************************************************************************************
 Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.

 Licensed under the Apache License, Version 2.0 (the "License").
 You may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 ******************************************************************************************************************** */
const javascript_1 = require("projen/lib/javascript");
const typescript_1 = require("projen/lib/typescript");
const generated_typescript_client_source_code_1 = require("./components/generated-typescript-client-source-code");
const open_api_generator_ignore_file_1 = require("./components/open-api-generator-ignore-file");
/**
 * Typescript project containing a typescript client (and lambda handler wrappers) generated using OpenAPI Generator CLI
 */
class GeneratedTypescriptClientProject extends typescript_1.TypeScriptProject {
    constructor(options) {
        super({
            ...options,
            sampleCode: false,
            tsconfig: {
                ...options.tsconfig,
                compilerOptions: {
                    lib: ["dom", "es2019"],
                    // Generated code isn't very strict!
                    strict: false,
                    alwaysStrict: false,
                    noImplicitAny: false,
                    noImplicitReturns: false,
                    noImplicitThis: false,
                    noUnusedLocals: false,
                    noUnusedParameters: false,
                    strictNullChecks: false,
                    strictPropertyInitialization: false,
                    ...options?.tsconfig?.compilerOptions,
                },
            },
            eslint: false,
        });
        // Store whether we've synthesized the project
        this.synthed = false;
        // Disable strict peer dependencies for pnpm as the default typescript project dependencies have type mismatches
        // (ts-jest@27 and @types/jest@28)
        if (this.package.packageManager === javascript_1.NodePackageManager.PNPM) {
            this.npmConfig = new javascript_1.NpmConfig(this);
            this.npmConfig.addConfig("strict-peer-dependencies", "false");
        }
        // For event and context types
        this.addDeps("@types/aws-lambda");
        // Tell OpenAPI Generator CLI not to generate files that we will generate via this project, or don't need.
        const openapiGeneratorIgnore = new open_api_generator_ignore_file_1.OpenApiGeneratorIgnoreFile(this);
        openapiGeneratorIgnore.addPatterns("package.json", "tsconfig.json", "tsconfig.esm.json", "src/index.ts");
        new generated_typescript_client_source_code_1.GeneratedTypescriptClientSourceCode(this, {
            specPath: options.specPath,
            invokeGenerator: options.generateClient,
        });
    }
    /**
     * @inheritDoc
     */
    synth() {
        // Save some time by only synthesizing once. We synthesize this project early so that it's available for the parent
        // project's install phase (pre-synth). Projen will call this method again at the usual time to synthesize this,
        // project, at which point we're already done so can skip.
        if (this.synthed) {
            return;
        }
        super.synth();
        this.synthed = true;
    }
}
exports.GeneratedTypescriptClientProject = GeneratedTypescriptClientProject;
//# sourceMappingURL=data:application/json;base64,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