"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.invokeCustomDocsGenerator = exports.invokeOpenApiGenerator = exports.NonClientGeneratorDirectory = void 0;
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const fs = require("fs");
const path = require("path");
const util_1 = require("projen/lib/util");
/**
 * Enum for generator directories for non-client generators
 */
var NonClientGeneratorDirectory;
(function (NonClientGeneratorDirectory) {
    NonClientGeneratorDirectory["DOCS"] = "docs";
})(NonClientGeneratorDirectory = exports.NonClientGeneratorDirectory || (exports.NonClientGeneratorDirectory = {}));
const serializeProperties = (properties) => Object.entries(properties)
    .map(([key, value]) => `${key}=${value}`)
    .join(",");
/**
 * Clean up any files that have already been generated by the openapi generator
 * @param generatedOutputPath output path for the generator
 */
const cleanPreviouslyGeneratedFiles = (generatedOutputPath) => {
    // OpenAPI generator writes a manifest called FILES which lists the files it generated.
    const openApiGeneratedFilesManifestPath = path.join(generatedOutputPath, ".openapi-generator", "FILES");
    // If the manifest exists, delete the files it lists
    if (fs.existsSync(openApiGeneratedFilesManifestPath)) {
        const previouslyGeneratedFiles = new Set(fs
            .readFileSync(openApiGeneratedFilesManifestPath, { encoding: "utf-8" })
            .split("\n")
            .filter((x) => x));
        previouslyGeneratedFiles.forEach((previouslyGeneratedFile) => {
            fs.unlinkSync(path.join(generatedOutputPath, previouslyGeneratedFile));
        });
    }
};
/**
 * Generate client code or docs by invoking the root generate script
 */
exports.invokeOpenApiGenerator = (options) => {
    // Prior to generating, clean up any previously generated files so that we don't have stale generated code from
    // previous executions.
    cleanPreviouslyGeneratedFiles(options.outputPath);
    const additionalProperties = options.additionalProperties
        ? ` --additional-properties "${serializeProperties(options.additionalProperties)}"`
        : "";
    util_1.exec(`./generate --generator ${options.generator} --spec-path ${options.specPath} --output-path ${options.outputPath} --generator-dir ${options.generatorDirectory}${additionalProperties}`, {
        cwd: path.resolve(__dirname, "..", "..", "..", "..", "scripts", "generators"),
    });
};
/**
 * Invoke a custom documentation generator script
 */
exports.invokeCustomDocsGenerator = (options) => {
    util_1.exec(`./${options.generator}${options.args ? ` ${options.args}` : ""}`, {
        cwd: path.resolve(__dirname, "..", "..", "..", "..", "scripts", "custom", "docs"),
    });
};
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidXRpbHMuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi8uLi8uLi8uLi9zcmMvcHJvamVjdC9jb2RlZ2VuL2NvbXBvbmVudHMvdXRpbHMudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBQUE7c0NBQ3NDO0FBQ3RDLHlCQUF5QjtBQUN6Qiw2QkFBNkI7QUFDN0IsMENBQXVDO0FBR3ZDOztHQUVHO0FBQ0gsSUFBWSwyQkFFWDtBQUZELFdBQVksMkJBQTJCO0lBQ3JDLDRDQUFhLENBQUE7QUFDZixDQUFDLEVBRlcsMkJBQTJCLEdBQTNCLG1DQUEyQixLQUEzQixtQ0FBMkIsUUFFdEM7QUFtQ0QsTUFBTSxtQkFBbUIsR0FBRyxDQUFDLFVBQXFDLEVBQUUsRUFBRSxDQUNwRSxNQUFNLENBQUMsT0FBTyxDQUFDLFVBQVUsQ0FBQztLQUN2QixHQUFHLENBQUMsQ0FBQyxDQUFDLEdBQUcsRUFBRSxLQUFLLENBQUMsRUFBRSxFQUFFLENBQUMsR0FBRyxHQUFHLElBQUksS0FBSyxFQUFFLENBQUM7S0FDeEMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxDQUFDO0FBRWY7OztHQUdHO0FBQ0gsTUFBTSw2QkFBNkIsR0FBRyxDQUFDLG1CQUEyQixFQUFFLEVBQUU7SUFDcEUsdUZBQXVGO0lBQ3ZGLE1BQU0saUNBQWlDLEdBQUcsSUFBSSxDQUFDLElBQUksQ0FDakQsbUJBQW1CLEVBQ25CLG9CQUFvQixFQUNwQixPQUFPLENBQ1IsQ0FBQztJQUVGLG9EQUFvRDtJQUNwRCxJQUFJLEVBQUUsQ0FBQyxVQUFVLENBQUMsaUNBQWlDLENBQUMsRUFBRTtRQUNwRCxNQUFNLHdCQUF3QixHQUFHLElBQUksR0FBRyxDQUN0QyxFQUFFO2FBQ0MsWUFBWSxDQUFDLGlDQUFpQyxFQUFFLEVBQUUsUUFBUSxFQUFFLE9BQU8sRUFBRSxDQUFDO2FBQ3RFLEtBQUssQ0FBQyxJQUFJLENBQUM7YUFDWCxNQUFNLENBQUMsQ0FBQyxDQUFDLEVBQUUsRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUNwQixDQUFDO1FBQ0Ysd0JBQXdCLENBQUMsT0FBTyxDQUFDLENBQUMsdUJBQXVCLEVBQUUsRUFBRTtZQUMzRCxFQUFFLENBQUMsVUFBVSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsbUJBQW1CLEVBQUUsdUJBQXVCLENBQUMsQ0FBQyxDQUFDO1FBQ3pFLENBQUMsQ0FBQyxDQUFDO0tBQ0o7QUFDSCxDQUFDLENBQUM7QUFFRjs7R0FFRztBQUNVLFFBQUEsc0JBQXNCLEdBQUcsQ0FBQyxPQUEwQixFQUFFLEVBQUU7SUFDbkUsK0dBQStHO0lBQy9HLHVCQUF1QjtJQUN2Qiw2QkFBNkIsQ0FBQyxPQUFPLENBQUMsVUFBVSxDQUFDLENBQUM7SUFFbEQsTUFBTSxvQkFBb0IsR0FBRyxPQUFPLENBQUMsb0JBQW9CO1FBQ3ZELENBQUMsQ0FBQyw2QkFBNkIsbUJBQW1CLENBQzlDLE9BQU8sQ0FBQyxvQkFBb0IsQ0FDN0IsR0FBRztRQUNOLENBQUMsQ0FBQyxFQUFFLENBQUM7SUFDUCxXQUFJLENBQ0YsMEJBQTBCLE9BQU8sQ0FBQyxTQUFTLGdCQUFnQixPQUFPLENBQUMsUUFBUSxrQkFBa0IsT0FBTyxDQUFDLFVBQVUsb0JBQW9CLE9BQU8sQ0FBQyxrQkFBa0IsR0FBRyxvQkFBb0IsRUFBRSxFQUN0TDtRQUNFLEdBQUcsRUFBRSxJQUFJLENBQUMsT0FBTyxDQUNmLFNBQVMsRUFDVCxJQUFJLEVBQ0osSUFBSSxFQUNKLElBQUksRUFDSixJQUFJLEVBQ0osU0FBUyxFQUNULFlBQVksQ0FDYjtLQUNGLENBQ0YsQ0FBQztBQUNKLENBQUMsQ0FBQztBQWdCRjs7R0FFRztBQUNVLFFBQUEseUJBQXlCLEdBQUcsQ0FDdkMsT0FBb0MsRUFDcEMsRUFBRTtJQUNGLFdBQUksQ0FBQyxLQUFLLE9BQU8sQ0FBQyxTQUFTLEdBQUcsT0FBTyxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsSUFBSSxPQUFPLENBQUMsSUFBSSxFQUFFLENBQUMsQ0FBQyxDQUFDLEVBQUUsRUFBRSxFQUFFO1FBQ3RFLEdBQUcsRUFBRSxJQUFJLENBQUMsT0FBTyxDQUNmLFNBQVMsRUFDVCxJQUFJLEVBQ0osSUFBSSxFQUNKLElBQUksRUFDSixJQUFJLEVBQ0osU0FBUyxFQUNULFFBQVEsRUFDUixNQUFNLENBQ1A7S0FDRixDQUFDLENBQUM7QUFDTCxDQUFDLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyIvKiEgQ29weXJpZ2h0IFtBbWF6b24uY29tXShodHRwOi8vYW1hem9uLmNvbS8pLCBJbmMuIG9yIGl0cyBhZmZpbGlhdGVzLiBBbGwgUmlnaHRzIFJlc2VydmVkLlxuU1BEWC1MaWNlbnNlLUlkZW50aWZpZXI6IEFwYWNoZS0yLjAgKi9cbmltcG9ydCAqIGFzIGZzIGZyb20gXCJmc1wiO1xuaW1wb3J0ICogYXMgcGF0aCBmcm9tIFwicGF0aFwiO1xuaW1wb3J0IHsgZXhlYyB9IGZyb20gXCJwcm9qZW4vbGliL3V0aWxcIjtcbmltcG9ydCB7IENsaWVudExhbmd1YWdlIH0gZnJvbSBcIi4uLy4uL2xhbmd1YWdlc1wiO1xuXG4vKipcbiAqIEVudW0gZm9yIGdlbmVyYXRvciBkaXJlY3RvcmllcyBmb3Igbm9uLWNsaWVudCBnZW5lcmF0b3JzXG4gKi9cbmV4cG9ydCBlbnVtIE5vbkNsaWVudEdlbmVyYXRvckRpcmVjdG9yeSB7XG4gIERPQ1MgPSBcImRvY3NcIixcbn1cblxuLyoqXG4gKiBHZW5lcmF0b3IgZGlyZWN0b3J5IGZvciBvcGVuYXBpIGdlbmVyYXRpb24gY29udGFpbmluZyB0ZW1wbGF0ZXMsIGNvbmZpZyBldGMuXG4gKi9cbmV4cG9ydCB0eXBlIEdlbmVyYXRvckRpcmVjdG9yeSA9IENsaWVudExhbmd1YWdlIHwgTm9uQ2xpZW50R2VuZXJhdG9yRGlyZWN0b3J5O1xuXG4vKipcbiAqIE9wdGlvbnMgZm9yIGdlbmVyYXRpbmcgY2xpZW50IGNvZGUgb3IgZG9jcyB1c2luZyBPcGVuQVBJIEdlbmVyYXRvciBDTElcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBHZW5lcmF0aW9uT3B0aW9ucyB7XG4gIC8qKlxuICAgKiBUaGUgT3BlbkFQSSBnZW5lcmF0b3IgdG8gdXNlIHRvIGdlbmVyYXRlIHRoZSBjb2RlL2RvY3NcbiAgICovXG4gIHJlYWRvbmx5IGdlbmVyYXRvcjogc3RyaW5nO1xuICAvKipcbiAgICogVGhlIGRpcmVjdG9yeSB0byB1c2UgZm9yIE9wZW5BUEkgZ2VuZXJhdGlvblxuICAgKi9cbiAgcmVhZG9ubHkgZ2VuZXJhdG9yRGlyZWN0b3J5OiBHZW5lcmF0b3JEaXJlY3Rvcnk7XG4gIC8qKlxuICAgKiBUaGUgcGF0aCBvZiB0aGUgT3BlbkFQSSBzcGVjIHRvIGdlbmVyYXRlIHRoZSBjbGllbnQgZm9yXG4gICAqL1xuICByZWFkb25seSBzcGVjUGF0aDogc3RyaW5nO1xuICAvKipcbiAgICogVGhlIGRpcmVjdG9yeSBpbiB3aGljaCB0aGUgZ2VuZXJhdGVkIGNvZGUgc2hvdWxkIGJlIG91dHB1dFxuICAgKi9cbiAgcmVhZG9ubHkgb3V0cHV0UGF0aDogc3RyaW5nO1xuICAvKipcbiAgICogQWRkaXRpb25hbCBwcm9wZXJ0aWVzIHRvIHBhc3MgdG8gdGhlIGdlbmVyYXRlIGNsaVxuICAgKi9cbiAgcmVhZG9ubHkgYWRkaXRpb25hbFByb3BlcnRpZXM/OiB7XG4gICAgW2tleTogc3RyaW5nXTogc3RyaW5nO1xuICB9O1xufVxuXG5jb25zdCBzZXJpYWxpemVQcm9wZXJ0aWVzID0gKHByb3BlcnRpZXM6IHsgW2tleTogc3RyaW5nXTogc3RyaW5nIH0pID0+XG4gIE9iamVjdC5lbnRyaWVzKHByb3BlcnRpZXMpXG4gICAgLm1hcCgoW2tleSwgdmFsdWVdKSA9PiBgJHtrZXl9PSR7dmFsdWV9YClcbiAgICAuam9pbihcIixcIik7XG5cbi8qKlxuICogQ2xlYW4gdXAgYW55IGZpbGVzIHRoYXQgaGF2ZSBhbHJlYWR5IGJlZW4gZ2VuZXJhdGVkIGJ5IHRoZSBvcGVuYXBpIGdlbmVyYXRvclxuICogQHBhcmFtIGdlbmVyYXRlZE91dHB1dFBhdGggb3V0cHV0IHBhdGggZm9yIHRoZSBnZW5lcmF0b3JcbiAqL1xuY29uc3QgY2xlYW5QcmV2aW91c2x5R2VuZXJhdGVkRmlsZXMgPSAoZ2VuZXJhdGVkT3V0cHV0UGF0aDogc3RyaW5nKSA9PiB7XG4gIC8vIE9wZW5BUEkgZ2VuZXJhdG9yIHdyaXRlcyBhIG1hbmlmZXN0IGNhbGxlZCBGSUxFUyB3aGljaCBsaXN0cyB0aGUgZmlsZXMgaXQgZ2VuZXJhdGVkLlxuICBjb25zdCBvcGVuQXBpR2VuZXJhdGVkRmlsZXNNYW5pZmVzdFBhdGggPSBwYXRoLmpvaW4oXG4gICAgZ2VuZXJhdGVkT3V0cHV0UGF0aCxcbiAgICBcIi5vcGVuYXBpLWdlbmVyYXRvclwiLFxuICAgIFwiRklMRVNcIlxuICApO1xuXG4gIC8vIElmIHRoZSBtYW5pZmVzdCBleGlzdHMsIGRlbGV0ZSB0aGUgZmlsZXMgaXQgbGlzdHNcbiAgaWYgKGZzLmV4aXN0c1N5bmMob3BlbkFwaUdlbmVyYXRlZEZpbGVzTWFuaWZlc3RQYXRoKSkge1xuICAgIGNvbnN0IHByZXZpb3VzbHlHZW5lcmF0ZWRGaWxlcyA9IG5ldyBTZXQoXG4gICAgICBmc1xuICAgICAgICAucmVhZEZpbGVTeW5jKG9wZW5BcGlHZW5lcmF0ZWRGaWxlc01hbmlmZXN0UGF0aCwgeyBlbmNvZGluZzogXCJ1dGYtOFwiIH0pXG4gICAgICAgIC5zcGxpdChcIlxcblwiKVxuICAgICAgICAuZmlsdGVyKCh4KSA9PiB4KVxuICAgICk7XG4gICAgcHJldmlvdXNseUdlbmVyYXRlZEZpbGVzLmZvckVhY2goKHByZXZpb3VzbHlHZW5lcmF0ZWRGaWxlKSA9PiB7XG4gICAgICBmcy51bmxpbmtTeW5jKHBhdGguam9pbihnZW5lcmF0ZWRPdXRwdXRQYXRoLCBwcmV2aW91c2x5R2VuZXJhdGVkRmlsZSkpO1xuICAgIH0pO1xuICB9XG59O1xuXG4vKipcbiAqIEdlbmVyYXRlIGNsaWVudCBjb2RlIG9yIGRvY3MgYnkgaW52b2tpbmcgdGhlIHJvb3QgZ2VuZXJhdGUgc2NyaXB0XG4gKi9cbmV4cG9ydCBjb25zdCBpbnZva2VPcGVuQXBpR2VuZXJhdG9yID0gKG9wdGlvbnM6IEdlbmVyYXRpb25PcHRpb25zKSA9PiB7XG4gIC8vIFByaW9yIHRvIGdlbmVyYXRpbmcsIGNsZWFuIHVwIGFueSBwcmV2aW91c2x5IGdlbmVyYXRlZCBmaWxlcyBzbyB0aGF0IHdlIGRvbid0IGhhdmUgc3RhbGUgZ2VuZXJhdGVkIGNvZGUgZnJvbVxuICAvLyBwcmV2aW91cyBleGVjdXRpb25zLlxuICBjbGVhblByZXZpb3VzbHlHZW5lcmF0ZWRGaWxlcyhvcHRpb25zLm91dHB1dFBhdGgpO1xuXG4gIGNvbnN0IGFkZGl0aW9uYWxQcm9wZXJ0aWVzID0gb3B0aW9ucy5hZGRpdGlvbmFsUHJvcGVydGllc1xuICAgID8gYCAtLWFkZGl0aW9uYWwtcHJvcGVydGllcyBcIiR7c2VyaWFsaXplUHJvcGVydGllcyhcbiAgICAgICAgb3B0aW9ucy5hZGRpdGlvbmFsUHJvcGVydGllc1xuICAgICAgKX1cImBcbiAgICA6IFwiXCI7XG4gIGV4ZWMoXG4gICAgYC4vZ2VuZXJhdGUgLS1nZW5lcmF0b3IgJHtvcHRpb25zLmdlbmVyYXRvcn0gLS1zcGVjLXBhdGggJHtvcHRpb25zLnNwZWNQYXRofSAtLW91dHB1dC1wYXRoICR7b3B0aW9ucy5vdXRwdXRQYXRofSAtLWdlbmVyYXRvci1kaXIgJHtvcHRpb25zLmdlbmVyYXRvckRpcmVjdG9yeX0ke2FkZGl0aW9uYWxQcm9wZXJ0aWVzfWAsXG4gICAge1xuICAgICAgY3dkOiBwYXRoLnJlc29sdmUoXG4gICAgICAgIF9fZGlybmFtZSxcbiAgICAgICAgXCIuLlwiLFxuICAgICAgICBcIi4uXCIsXG4gICAgICAgIFwiLi5cIixcbiAgICAgICAgXCIuLlwiLFxuICAgICAgICBcInNjcmlwdHNcIixcbiAgICAgICAgXCJnZW5lcmF0b3JzXCJcbiAgICAgICksXG4gICAgfVxuICApO1xufTtcblxuLyoqXG4gKiBPcHRpb25zIGZvciBnZW5lcmF0aW5nIGRvY3VtZW50YXRpb24gdmlhIGEgY3VzdG9tIGdlbmVyYXRvciBzY3JpcHRcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBDdXN0b21Eb2NzR2VuZXJhdGlvbk9wdGlvbnMge1xuICAvKipcbiAgICogTmFtZSBvZiB0aGUgZ2VuZXJhdG9yIHNjcmlwdCB3aGljaCBleGlzdHMgaW4gc2NyaXB0cy9jdXN0b20vZG9jc1xuICAgKi9cbiAgcmVhZG9ubHkgZ2VuZXJhdG9yOiBzdHJpbmc7XG4gIC8qKlxuICAgKiBBbnkgYXJndW1lbnRzIHRvIHBhc3MgdG8gdGhlIHNjcmlwdFxuICAgKi9cbiAgcmVhZG9ubHkgYXJncz86IHN0cmluZztcbn1cblxuLyoqXG4gKiBJbnZva2UgYSBjdXN0b20gZG9jdW1lbnRhdGlvbiBnZW5lcmF0b3Igc2NyaXB0XG4gKi9cbmV4cG9ydCBjb25zdCBpbnZva2VDdXN0b21Eb2NzR2VuZXJhdG9yID0gKFxuICBvcHRpb25zOiBDdXN0b21Eb2NzR2VuZXJhdGlvbk9wdGlvbnNcbikgPT4ge1xuICBleGVjKGAuLyR7b3B0aW9ucy5nZW5lcmF0b3J9JHtvcHRpb25zLmFyZ3MgPyBgICR7b3B0aW9ucy5hcmdzfWAgOiBcIlwifWAsIHtcbiAgICBjd2Q6IHBhdGgucmVzb2x2ZShcbiAgICAgIF9fZGlybmFtZSxcbiAgICAgIFwiLi5cIixcbiAgICAgIFwiLi5cIixcbiAgICAgIFwiLi5cIixcbiAgICAgIFwiLi5cIixcbiAgICAgIFwic2NyaXB0c1wiLFxuICAgICAgXCJjdXN0b21cIixcbiAgICAgIFwiZG9jc1wiXG4gICAgKSxcbiAgfSk7XG59O1xuIl19