"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.OpenApiGatewayPythonProject = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*********************************************************************************************************************
 Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.

 Licensed under the Apache License, Version 2.0 (the "License").
 You may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 ******************************************************************************************************************** */
const path = require("path");
const projen_1 = require("projen");
const python_1 = require("projen/lib/python");
const docs_project_1 = require("./codegen/docs-project");
const generate_1 = require("./codegen/generate");
const languages_1 = require("./languages");
const python_2 = require("./samples/python");
const open_api_spec_project_1 = require("./spec/open-api-spec-project");
const OPENAPI_GATEWAY_PDK_PACKAGE_NAME = "aws_prototyping_sdk.open_api_gateway";
/**
 * Synthesizes a Python Project with an OpenAPI spec, generated clients, a CDK construct for deploying the API
 * with API Gateway, and generated lambda handler wrappers for type-safe handling of requests.
 *
 * @pjid open-api-gateway-py
 */
class OpenApiGatewayPythonProject extends python_1.PythonProject {
    constructor(options) {
        super({
            ...options,
            sample: false,
            venv: true,
            venvOptions: {
                envdir: ".env",
                ...options?.venvOptions,
            },
            pip: true,
            poetry: false,
            pytest: false,
            setuptools: true,
        });
        if (options.specFile) {
            this.specDir = path.dirname(options.specFile);
            this.specFileName = path.basename(options.specFile);
        }
        else {
            this.specDir = "spec";
            this.specFileName = "spec.yaml";
        }
        this.generatedCodeDir = options.generatedCodeDir ?? "generated";
        this.apiSrcDir = options.apiSrcDir ?? "api";
        // Generated project should have a dependency on this project, in order to run the generation scripts
        [OPENAPI_GATEWAY_PDK_PACKAGE_NAME, "constructs", "aws-cdk-lib"].forEach((dep) => this.addDependency(dep));
        // Synthesize the openapi spec early since it's used by the generated python client, which is also synth'd early
        const spec = new open_api_spec_project_1.OpenApiSpecProject({
            name: `${this.name}-spec`,
            parent: this,
            outdir: path.join(this.moduleName, this.specDir),
            specFileName: this.specFileName,
            parsedSpecFileName: options.parsedSpecFileName,
        });
        spec.synth();
        // Parent the generated code with this project's parent for better integration with monorepos
        this.hasParent = !!options.parent;
        const generatedCodeDirRelativeToParent = this.hasParent
            ? path.join(options.outdir, this.generatedCodeDir)
            : this.generatedCodeDir;
        // Always generate the python client since this project will take a dependency on it in order to produce the
        // type-safe cdk construct wrapper.
        const clientLanguages = new Set(options.clientLanguages);
        clientLanguages.add(languages_1.ClientLanguage.PYTHON);
        // Share the same env between this project and the generated client. Accept a custom venv if part of a monorepo
        const envDir = options.venvOptions?.envdir || ".env";
        // env directory relative to the generated python client
        const clientEnvDir = path.join("..", ...this.generatedCodeDir.split("/").map(() => ".."), envDir);
        this.generatedClients = generate_1.generateClientProjects(clientLanguages, {
            parent: this.hasParent ? options.parent : this,
            parentPackageName: this.name,
            generatedCodeDir: generatedCodeDirRelativeToParent,
            parsedSpecPath: spec.parsedSpecPath,
            typescriptOptions: {
                defaultReleaseBranch: "main",
                ...options.typescriptClientOptions,
            },
            pythonOptions: {
                authorName: options.authorName ?? "APJ Cope",
                authorEmail: options.authorEmail ?? "apj-cope@amazon.com",
                version: "0.0.0",
                ...options.pythonClientOptions,
                // We are more prescriptive about the generated client since we must set up a dependency in the shared env
                pip: true,
                poetry: false,
                venv: true,
                venvOptions: {
                    envdir: clientEnvDir,
                },
                generateLayer: true,
            },
            javaOptions: {
                version: "0.0.0",
                ...options.javaClientOptions,
            },
        });
        this.generatedPythonClient = this.generatedClients[languages_1.ClientLanguage.PYTHON];
        // Synth early so that the generated code is available prior to this project's install phase
        this.generatedPythonClient.synth();
        // Add a dependency on the generated python client, which should be available since we share the virtual env
        this.addDependency(this.generatedPythonClient.moduleName);
        if (this.hasParent) {
            // Since the generated python client project is parented by this project's parent rather than this project,
            // projen will clean up the generated client when synthesizing this project unless we add an explicit exclude.
            this.addExcludeFromCleanup(`${this.generatedCodeDir}/**/*`);
        }
        // Get the lambda layer dir relative to the root of this project
        const pythonLayerDistDir = path.join(this.generatedCodeDir, languages_1.ClientLanguage.PYTHON, this.generatedPythonClient.layerDistDir);
        // Ensure it's included in the package
        new projen_1.TextFile(this, "MANIFEST.in", {
            lines: [`recursive-include ${pythonLayerDistDir} *`],
        });
        // Generate the sample source and test code
        const sampleOptions = {
            openApiGatewayPackageName: OPENAPI_GATEWAY_PDK_PACKAGE_NAME,
            pythonClientPackageName: this.generatedPythonClient.moduleName,
            sampleCode: options.sample,
            specDir: this.specDir,
            parsedSpecFileName: spec.parsedSpecFileName,
            moduleName: this.moduleName,
        };
        // Define some helpers for resolving resource paths in __init__.py
        new projen_1.SampleFile(this, path.join(this.moduleName, "__init__.py"), {
            contents: `import pkgutil, json
from os import path
from pathlib import Path

SPEC = json.loads(pkgutil.get_data(__name__, "${this.specDir}/${spec.parsedSpecFileName}"))

def get_project_root():
    return Path(__file__).absolute().parent.parent

def get_generated_client_layer_directory():
    return path.join(str(get_project_root()), "${pythonLayerDistDir}")
`,
        });
        new projen_1.SampleDir(this, path.join(this.moduleName, this.apiSrcDir), {
            files: python_2.getPythonSampleSource(sampleOptions),
        });
        // Set up pytest manually since the default pytest generates tests for sample code which doesn't exist
        const pytestVersion = options.pytestOptions?.version || "6.2.1";
        this.addDevDependency(`pytest@${pytestVersion}`);
        this.testTask.exec("pytest");
        // Generate documentation if requested
        if ((options.documentationFormats ?? []).length > 0) {
            new docs_project_1.DocsProject({
                parent: this,
                outdir: path.join(this.generatedCodeDir, "documentation"),
                name: "docs",
                formats: [...new Set(options.documentationFormats)],
                specPath: spec.parsedSpecPath,
            });
        }
    }
}
exports.OpenApiGatewayPythonProject = OpenApiGatewayPythonProject;
_a = JSII_RTTI_SYMBOL_1;
OpenApiGatewayPythonProject[_a] = { fqn: "@aws-prototyping-sdk/open-api-gateway.OpenApiGatewayPythonProject", version: "0.3.9" };
//# sourceMappingURL=data:application/json;base64,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