"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getAggregateAlarmState = exports.getAlarms = exports.getCloudwatchAlarms = exports.AlarmState = void 0;
const aws_sdk_1 = require("aws-sdk");
var AlarmState;
(function (AlarmState) {
    AlarmState["OK"] = "OK";
    AlarmState["ALARM"] = "ALARM";
    AlarmState["INSUFFICIENT_DATA"] = "INSUFFICIENT_DATA";
})(AlarmState = exports.AlarmState || (exports.AlarmState = {}));
;
;
;
exports.getCloudwatchAlarms = async (params) => {
    const cloudwatch = new aws_sdk_1.CloudWatch();
    let nextToken;
    let results = [];
    do {
        const result = await cloudwatch
            .describeAlarms({ AlarmTypes: ['CompositeAlarm', 'MetricAlarm'], ...params, NextToken: nextToken })
            .promise();
        results = result.CompositeAlarms ? results.concat(result.CompositeAlarms?.map(alarm => {
            return {
                alarmArn: alarm.AlarmArn,
                state: alarm.StateValue,
                reason: alarm.StateReason,
                alarmDescription: alarm.AlarmDescription,
            };
        })) : results;
        results = result.MetricAlarms ? results.concat(result.MetricAlarms?.map(alarm => {
            return {
                alarmArn: alarm.AlarmArn,
                state: alarm.StateValue,
                reason: alarm.StateReason,
                alarmDescription: alarm.AlarmDescription,
            };
        })) : results;
        nextToken = result.NextToken;
    } while (nextToken);
    return results;
};
/**
 * Returns all alarms starting with the provided prefix in the state of ALARM. In addition,
 * any exceptions caught will be logged and converted into an alarm state of options.defaultState
 * with an associated reason.
 */
exports.getAlarms = async (searchTerms) => {
    const states = [];
    try {
        const alarmsForAccountAndRegion = await exports.getCloudwatchAlarms();
        const alarms = hasAlarmMatching(alarmsForAccountAndRegion, searchTerms).map((alarm) => {
            alarm.reason = `${alarm.alarmArn} in ${alarm.state} due to ${alarm.reason}`;
            return alarm;
        });
        if (alarms.length === 0) {
            alarms.push({
                state: AlarmState.ALARM,
                reason: `No alarms were found for the provided search terms: ${searchTerms.join(', ')}.`,
            });
        }
    }
    catch (e) {
        const message = 'Unable to retrieve alarms.';
        states.push({ reason: message, state: AlarmState.ALARM });
        console.error(message, e);
    }
    return states;
};
/**
 * Calculate and return an aggregate alarm state across alarms returned from all
 * CloudWatch clients provided. If any client errors are thrown, the returned state
 * will be ALARM.
 *
 */
exports.getAggregateAlarmState = async (searchTerms) => {
    const alarmDetails = await exports.getAlarms(searchTerms);
    return {
        alarmDetails,
        state: alarmDetails.find((a) => a.state === AlarmState.ALARM)?.state ?? AlarmState.OK,
        summary: `${alarmDetails
            .filter((a) => a.state === AlarmState.ALARM)
            .map((a) => a.reason)
            .join(', ')}`,
    };
};
const hasAlarmMatching = (alarms, searchTerms) => {
    return alarms.flatMap((alarm) => searchTerms.map((searchTerm) => {
        return alarm.alarmDescription?.includes(searchTerm) ? alarm : undefined;
    }).filter((sameAlarm) => sameAlarm)).filter((sameAlarm, index) => alarms.indexOf(sameAlarm) === index);
};
//# sourceMappingURL=data:application/json;base64,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