# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['statemachine', 'statemachine.contrib']

package_data = \
{'': ['*'],
 'statemachine': ['locale/en/LC_MESSAGES/*', 'locale/pt_BR/LC_MESSAGES/*']}

setup_kwargs = {
    'name': 'python-statemachine',
    'version': '2.0.0',
    'description': 'Python Finite State Machines made easy.',
    'long_description': '# Python StateMachine\n\n[![pypi](https://img.shields.io/pypi/v/python-statemachine.svg)](https://pypi.python.org/pypi/python-statemachine)\n[![downloads](https://img.shields.io/pypi/dm/python-statemachine.svg)](https://pypi.python.org/pypi/python-statemachine)\n[![build status](https://github.com/fgmacedo/python-statemachine/actions/workflows/python-package.yml/badge.svg?branch=develop)](https://github.com/fgmacedo/python-statemachine/actions/workflows/python-package.yml?query=branch%3Adevelop)\n[![Coverage report](https://codecov.io/gh/fgmacedo/python-statemachine/branch/develop/graph/badge.svg)](https://codecov.io/gh/fgmacedo/python-statemachine)\n[![Documentation Status](https://readthedocs.org/projects/python-statemachine/badge/?version=latest)](https://python-statemachine.readthedocs.io/en/latest/?badge=latest)\n[![GitHub commits since last release (main)](https://img.shields.io/github/commits-since/fgmacedo/python-statemachine/main/develop)](https://github.com/fgmacedo/python-statemachine/compare/main...develop)\n\n\nPython [finite-state machines](https://en.wikipedia.org/wiki/Finite-state_machine) made easy.\n\n\n* Free software: MIT license\n* Documentation: https://python-statemachine.readthedocs.io.\n\n\nWelcome to python-statemachine, an intuitive and powerful state machine framework designed for a\ngreat developer experience.\n\n🚀 With StateMachine, you can easily create complex, dynamic systems with clean, readable code.\n\n💡 Our framework makes it easy to understand and reason about the different states, events and\ntransitions in your system, so you can focus on building great products.\n\n🔒 python-statemachine also provides robust error handling and ensures that your system stays\nin a valid state at all times.\n\n\nA few reasons why you may consider using it:\n\n* 📈 python-statemachine is designed to help you build scalable,\n  maintainable systems that can handle any complexity.\n* 💪 You can easily create and manage multiple state machines within a single application.\n* 🚫 Prevents common mistakes and ensures that your system stays in a valid state at all times.\n\n\n## Getting started\n\n\nTo install Python State Machine, run this command in your terminal:\n\n    pip install python-statemachine\n\nTo generate diagrams from your machines, you\'ll also need `pydot` and `Graphviz`. You can\ninstall this library already with `pydot` dependency using the `extras` install option. See\nour docs for more details.\n\n    pip install python-statemachine[diagrams]\n\nDefine your state machine:\n\n```py\n>>> from statemachine import StateMachine, State\n\n>>> class TrafficLightMachine(StateMachine):\n...     "A traffic light machine"\n...     green = State(initial=True)\n...     yellow = State()\n...     red = State()\n...\n...     cycle = green.to(yellow) | yellow.to(red) | red.to(green)\n...\n...     slowdown = green.to(yellow)\n...     stop = yellow.to(red)\n...     go = red.to(green)\n...\n...     def before_cycle(self, event: str, source: State, target: State, message: str = ""):\n...         message = ". " + message if message else ""\n...         return f"Running {event} from {source.id} to {target.id}{message}"\n...\n...     def on_enter_red(self):\n...         print("Don\'t move.")\n...\n...     def on_exit_red(self):\n...         print("Go ahead!")\n\n```\n\nYou can now create an instance:\n\n```py\n>>> traffic_light = TrafficLightMachine()\n\n```\n\nThen start sending events:\n\n```py\n>>> traffic_light.cycle()\n\'Running cycle from green to yellow\'\n\n```\n\nYou can inspect the current state:\n\n```py\n>>> traffic_light.current_state.id\n\'yellow\'\n\n```\n\nA `State` human-readable name is automatically derived from the `State.id`:\n\n```py\n>>> traffic_light.current_state.name\n\'Yellow\'\n\n```\n\nOr get a complete state representation for debugging purposes:\n\n```py\n>>> traffic_light.current_state\nState(\'Yellow\', id=\'yellow\', value=\'yellow\', initial=False, final=False)\n\n```\n\nThe ``State`` instance can also be checked by equality:\n\n```py\n>>> traffic_light.current_state == TrafficLightMachine.yellow\nTrue\n\n>>> traffic_light.current_state == traffic_light.yellow\nTrue\n\n```\n\nBut for your convenience, can easily ask if a state is active at any time:\n\n```py\n>>> traffic_light.green.is_active\nFalse\n\n>>> traffic_light.yellow.is_active\nTrue\n\n>>> traffic_light.red.is_active\nFalse\n\n```\n\nEasily iterate over all states:\n\n```py\n>>> [s.id for s in traffic_light.states]\n[\'green\', \'red\', \'yellow\']\n\n```\n\nOr over events:\n\n```py\n>>> [t.name for t in traffic_light.events]\n[\'cycle\', \'go\', \'slowdown\', \'stop\']\n\n```\n\nCall an event by its name:\n\n```py\n>>> traffic_light.cycle()\nDon\'t move.\n\'Running cycle from yellow to red\'\n\n```\nOr send an event with the event name:\n\n```py\n>>> traffic_light.send(\'cycle\')\nGo ahead!\n\'Running cycle from red to green\'\n\n>>> traffic_light.green.is_active\nTrue\n\n```\nYou can\'t run a transition from an invalid state:\n\n```py\n>>> traffic_light.go()\nTraceback (most recent call last):\nstatemachine.exceptions.TransitionNotAllowed: Can\'t go when in Green.\n\n```\nKeeping the same state as expected:\n\n```py\n>>> traffic_light.green.is_active\nTrue\n\n```\n\nAnd you can pass arbitrary positional or keyword arguments to the event, and\nthey will be propagated to all actions and callbacks:\n\n```py\n>>> traffic_light.cycle(message="Please, now slowdown.")\n\'Running cycle from green to yellow. Please, now slowdown.\'\n\n```\n\n## A more useful example\n\nA simple didactic state machine for controlling an `Order`:\n\n```py\n>>> class OrderControl(StateMachine):\n...     waiting_for_payment = State(initial=True)\n...     processing = State()\n...     shipping = State()\n...     completed = State(final=True)\n...\n...     add_to_order = waiting_for_payment.to(waiting_for_payment)\n...     receive_payment = (\n...         waiting_for_payment.to(processing, cond="payments_enough")\n...         | waiting_for_payment.to(waiting_for_payment, unless="payments_enough")\n...     )\n...     process_order = processing.to(shipping, cond="payment_received")\n...     ship_order = shipping.to(completed)\n...\n...     def __init__(self):\n...         self.order_total = 0\n...         self.payments = []\n...         self.payment_received = False\n...         super(OrderControl, self).__init__()\n...\n...     def payments_enough(self, amount):\n...         return sum(self.payments) + amount >= self.order_total\n...\n...     def before_add_to_order(self, amount):\n...         self.order_total += amount\n...         return self.order_total\n...\n...     def before_receive_payment(self, amount):\n...         self.payments.append(amount)\n...         return self.payments\n...\n...     def after_receive_payment(self):\n...         self.payment_received = True\n...\n...     def on_enter_waiting_for_payment(self):\n...         self.payment_received = False\n\n```\n\nYou can use this machine as follows.\n\n```py\n>>> control = OrderControl()\n\n>>> control.add_to_order(3)\n3\n\n>>> control.add_to_order(7)\n10\n\n>>> control.receive_payment(4)\n[4]\n\n>>> control.current_state.id\n\'waiting_for_payment\'\n\n>>> control.current_state.name\n\'Waiting for payment\'\n\n>>> control.process_order()\nTraceback (most recent call last):\n...\nstatemachine.exceptions.TransitionNotAllowed: Can\'t process_order when in Waiting for payment.\n\n>>> control.receive_payment(6)\n[4, 6]\n\n>>> control.current_state.id\n\'processing\'\n\n>>> control.process_order()\n\n>>> control.ship_order()\n\n>>> control.payment_received\nTrue\n\n>>> control.order_total\n10\n\n>>> control.payments\n[4, 6]\n\n>>> control.completed.is_active\nTrue\n\n```\n\nThere\'s a lot more to cover, please take a look at our docs:\nhttps://python-statemachine.readthedocs.io.\n\n\n## Contributing to the project\n\n* <a class="github-button" href="https://github.com/fgmacedo/python-statemachine" data-icon="octicon-star" aria-label="Star fgmacedo/python-statemachine on GitHub">Star this project</a>\n* <a class="github-button" href="https://github.com/fgmacedo/python-statemachine/issues" data-icon="octicon-issue-opened" aria-label="Issue fgmacedo/python-statemachine on GitHub">Open an Issue</a>\n* <a class="github-button" href="https://github.com/fgmacedo/python-statemachine/fork" data-icon="octicon-repo-forked" aria-label="Fork fgmacedo/python-statemachine on GitHub">Fork</a>\n\n- If you found this project helpful, please consider giving it a star on GitHub.\n\n- **Contribute code**: If you would like to contribute code to this project, please submit a pull\nrequest. For more information on how to contribute, please see our [contributing.md]contributing.md) file.\n\n- **Report bugs**: If you find any bugs in this project, please report them by opening an issue\n  on our GitHub issue tracker.\n\n- **Suggest features**: If you have a great idea for a new feature, please let us know by opening\n  an issue on our GitHub issue tracker.\n\n- **Documentation**: Help improve this project\'s documentation by submitting pull requests.\n\n- **Promote the project**: Help spread the word about this project by sharing it on social media,\n  writing a blog post, or giving a talk about it. Tag me on Twitter\n  [@fgmacedo](https://twitter.com/fgmacedo) so I can share it too!\n',
    'author': 'Fernando Macedo',
    'author_email': 'fgmacedo@gmail.com',
    'maintainer': 'Fernando Macedo',
    'maintainer_email': 'fgmacedo@gmail.com',
    'url': 'https://github.com/fgmacedo/python-statemachine',
    'packages': packages,
    'package_data': package_data,
    'python_requires': '>=3.7,<3.12',
}


setup(**setup_kwargs)
