"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const lambda = require("@aws-cdk/aws-lambda");
const crypto = require("crypto");
const fs = require("fs");
const path = require("path");
const builder_1 = require("./builder");
const util_1 = require("./util");
/**
 * A Node.js Lambda function bundled using Parcel
 */
class GolangFunction extends lambda.Function {
    constructor(scope, id, props = {}) {
        const entry = findEntry(id, props.entry);
        const handler = props.handler || 'main';
        const buildCmd = props.buildCmd || 'go build -ldflags="-s -w"';
        const buildDir = props.buildDir || path.join(path.dirname(entry), '.build');
        const handlerDir = path.join(buildDir, crypto.createHash('sha256').update(entry).digest('hex'));
        const runtime = lambda.Runtime.GO_1_X;
        // Build with go
        const builder = new builder_1.Builder({
            entry,
            outDir: handlerDir,
            handler,
            buildCmd,
            extraEnv: props.extraEnv || {},
        });
        builder.build();
        super(scope, id, {
            ...props,
            runtime,
            code: lambda.Code.fromAsset(handlerDir),
            handler: handler,
        });
    }
}
exports.GolangFunction = GolangFunction;
/**
 * Searches for an entry file. Preference order is the following:
 * 1. Given entry file
 * 2. A .go file named as the defining file with id as suffix (defining-file.id.go)
 */
function findEntry(id, entry) {
    if (entry) {
        if (!/\.(go)$/.test(entry)) {
            throw new Error('Only Golang entry files are supported.');
        }
        if (!fs.existsSync(entry)) {
            throw new Error(`Cannot find entry file at ${entry}`);
        }
        return entry;
    }
    const definingFile = findDefiningFile();
    const libDir = path.join(definingFile, '..');
    const goHandlerFile = path.join(libDir, `/${id}/main.go`);
    if (fs.existsSync(goHandlerFile)) {
        return goHandlerFile;
    }
    throw new Error('Cannot find entry file.');
}
/**
 * Finds the name of the file where the `GolangFunction` is defined
 */
function findDefiningFile() {
    const stackTrace = util_1.parseStackTrace();
    const functionIndex = stackTrace.findIndex((s) => /GolangFunction/.test(s.methodName || ''));
    if (functionIndex === -1 || !stackTrace[functionIndex + 1]) {
        throw new Error('Cannot find defining file.');
    }
    return stackTrace[functionIndex + 1].file;
}
//# sourceMappingURL=data:application/json;base64,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