from collections import defaultdict
from parse import to_int, to_float, to_decimal, to_boolean, IMPLEMENTATIONS
import tracemalloc
import csv


def run_all_int(impl):
    """Run all the int test cases once each"""
    cases = [
        "2022",
        "+1024",
        "-40",
        "0",
        "+0",
        "-0",
        "2.99e8",
        "3.14e-3",
        "2718e-3",
        "1000e-2",
        "-5e5",
        "-2.74e-3",
        "1e0",
        "1e+0",
        "1e-0",
        "1.0e0",
        "1.0e+0",
        "1.0e-0",
        "  \t  128",
        "128       ",
        "\t128\t \t ",
        "3.1415",
        "512.00",
        "20_48",
        "-1148130695274254524232833201177681984022317702088695200477642736825766261392370313856659486316506269918445964"
        "63898746277344711896086305533142593135616665318539129989145312280000688779148240044871428926990063486244781615"
        "46364638836394731702604046635397090499655816239880894462960562331164953616422197033268134416890898445850560237"
        "94848079140589009347765004290027167066258305220081322362812917612678833172065989953964181270217798584040421598"
        "53183251540889433902091920554957783589672039160081957216630582755380425583726015528348786419432054508915275783"
        "882625175435528800822842770817965453762184851149029376",
        "-1148130695274254524232833201177681984022317702088695200477642736825766261392370313856659486316506269918445964"
        "63898746277344711896086305533142593135616665318539129989145312280000688779148240044871428926990063486244781615"
        "46364638836394731702604046635397090499655816239880894462960562331164953616422197033268134416890898445850560237"
        "94848079140589009347765004290027167066258305220081322362812917612678833172065989953964181270217798584040421598"
        "53183251540889433902091920554957783589672039160081957216630582755380425583726015528348786419432054508915275783"
        "882625175435528800822842770817965453762184851149029376",
        "nan",
        "NaN",
        "NAN",
        "inf",
        "-inf",
        "two",
        "12.4.5",
        "1/2",
        "4 + 2i",
        "5 + 3j",
        "2,500",
        "0xbadbeef",
        "0o775",
        "0b100001",
        "infinity",
        "INFINITY",
        "-infinity",
        "-INFINITY",
    ]

    for case in cases:
        # we don't care about the output, that's tested in unit tests
        # we just want to run each case
        to_int(case, impl=impl)


def run_all_float(impl):
    """Run all the float cases once each"""
    cases = [
        "2022",
        "32.23",
        "+1024",
        "+99.99",
        "-40",
        "-67.1322",
        "12.340000000",
        "-56.78000",
        "0",
        "+0",
        "-0",
        "0.0000000",
        "2.99e8",
        "3.14e-3",
        "-5e5",
        "-2.74e-3",
        "  \t  128.01",
        "128.01       ",
        "\t128.01\t \t ",
        "1000e-2",
        "512.00",
        "20_48",
        "1_499_999.9_9",
        "nan",
        "NaN",
        "NAN",
        "inf",
        "-inf",
        "two",
        "thirty-five point seven",
        "12.4.5",
        "1/2",
        "4 + 2i",
        "5.2 - 3.7j",
        "2,500",
        "0xbadbeef",
        "0o775",
        "0b100001",
    ]
    for case in cases:
        # we don't care about the output, that's tested in unit tests
        # we just want to run each case
        to_float(case, impl=impl)


def run_all_decimal(impl):
    cases = [
        "2022",
        "32.5",
        "+1024",
        "+99.99",
        "-40",
        "-67.1322",
        "12.3400",
        "-56.78000",
        "0",
        "+0",
        "-0",
        "0.0000000",
        "2.99e8",
        "3.14e-3",
        "-5e5",
        "-2.74e-3",
        "1000e-2",
        "  \t  128.01",
        "128.01       ",
        "\t128.01\t \t ",
        "512.00",
        "-512.00",
        "20_48",
        "1_499_999.9_9",
        "11481306952742545242328332011776819840223177020886952004776427368257662613923703138566594863165062699184459646"
        "38987462773447118960863055331425931356166653185391299891453122800006887791482400448714289269900634862447816154"
        "63646388363947317026040466353970904996558162398808944629605623311649536164221970332681344168908984458505602379"
        "48480791405890093477650042900271670662583052200813223628129176126788331720659899539641812702177985840404215985"
        "31832515408894339020919205549577835896720391600819572166305827553804255837260155283487864194320545089152757838"
        "82625175435528800822842770817965453762184851149029376.0",
        "-1148130695274254524232833201177681984022317702088695200477642736825766261392370313856659486316506269918445964"
        "63898746277344711896086305533142593135616665318539129989145312280000688779148240044871428926990063486244781615"
        "46364638836394731702604046635397090499655816239880894462960562331164953616422197033268134416890898445850560237"
        "94848079140589009347765004290027167066258305220081322362812917612678833172065989953964181270217798584040421598"
        "53183251540889433902091920554957783589672039160081957216630582755380425583726015528348786419432054508915275783"
        "882625175435528800822842770817965453762184851149029376.0",
        "0.000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "0000000000001",
        "-0.00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"
        "00000000000001",
        "nan",
        "NaN",
        "NAN",
        "inf",
        "-inf",
        "two",
        "thirty-five point seven",
        "12.4.5",
        "1/2",
        "4 + 2i",
        "5.2 - 3.7j",
        "2,500",
        "0xbadbeef",
        "0o775",
        "0b100001",
        "infinity",
        "INFINITY",
        "-infinity",
        "-INFINITY",
    ]
    for case in cases:
        # we don't care about the output, that's tested in unit tests
        # we just want to run each case
        to_decimal(case, impl=impl)


if __name__ == "__main__":
    import time

    functions = [
        run_all_int,
        run_all_float,
    ]
    skip_functions = [
        # "pandas",
        # "built-ins",
    ]
    results = defaultdict(lambda: {})
    csv_output = []
    loops = 10_000
    tracemalloc.start()
    run_all_int(0)
    run_all_float(0)
    for func in functions:
        for impl, name in IMPLEMENTATIONS.items():
            if name in skip_functions:
                print(f"-- Skipping {name}")
                continue
            print(f"Running {func.__name__}: {name}")
            tracemalloc.clear_traces()

            start = time.time()
            errors = 0
            for _ in range(loops):
                try:
                    func(impl=impl)
                except Exception as e:
                    print(e)
                    errors += 1

            size, peak = tracemalloc.get_traced_memory()
            results[func.__name__][name] = {
                "size": size,
                "peak": peak,
                "time": round(time.time() - start, 3),
                "errors": errors,
            }
            csv_output.append(
                {
                    "function": func.__name__,
                    "implementation": name,
                    "size": size,
                    "peak": peak,
                    "time": round(time.time() - start, 3),
                    "errors": errors,
                }
            )
    tracemalloc.stop()
    from pprint import pprint as pp

    pp(results)
    pp(csv_output)
    with open("out.csv", "w") as csv_outfile:
        writer = csv.DictWriter(
            csv_outfile,
            fieldnames=["function", "implementation", "time", "size", "peak"],
            extrasaction="ignore",
        )
        writer.writeheader()
        writer.writerows(csv_output)
