"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ChangeController = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cloudwatch_1 = require("aws-cdk-lib/aws-cloudwatch");
const aws_events_1 = require("aws-cdk-lib/aws-events");
const aws_events_targets_1 = require("aws-cdk-lib/aws-events-targets");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const constructs_1 = require("constructs");
const change_controller_function_1 = require("./change-controller-function");
/**
 * A change controller. When added to a stage in a pipeline, this will check against
 * a calendar and enable or disable the stage transition based off that calendar,
 * defaulting to closed when the calendar cannot be found or when
 * the check against it fails. It also checks to against alarms.
 */
class ChangeController extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const fn = new change_controller_function_1.ChangeControllerFunction(this, 'ChangeControllerLambda');
        // Grant permission for stage transitions
        fn.addToRolePolicy(new aws_iam_1.PolicyStatement({
            resources: [`${props.stage.pipeline.pipelineArn}*`],
            actions: [
                'codepipeline:EnableStageTransition',
                'codepipeline:DisableStageTransition',
                'codepipeline:GetPipelineState',
            ],
            effect: aws_iam_1.Effect.ALLOW,
        }));
        // Grant permission to retrieve calendars
        fn.addToRolePolicy(new aws_iam_1.PolicyStatement({
            resources: [props.calendar.calendarArn],
            actions: ['ssm:GetCalendarState'],
            effect: aws_iam_1.Effect.ALLOW,
        }));
        // Grant permisssion to check alarm states
        fn.addToRolePolicy(new aws_iam_1.PolicyStatement({
            resources: ['*'],
            actions: ['cloudwatch:DescribeAlarms'],
            effect: aws_iam_1.Effect.ALLOW,
        }));
        // Any error in the lambda function will close the time window
        fn.metricErrors()
            .with({ statistic: 'sum' })
            .createAlarm(this, 'change-controller-alarm', {
            alarmName: `ChangeController-${props.stage.pipeline.pipelineName}${props.stage.stageName}`,
            evaluationPeriods: 1,
            threshold: 1,
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
            treatMissingData: aws_cloudwatch_1.TreatMissingData.BREACHING,
        });
        // Create a rule to run the lambda on a schedule defined by the user
        new aws_events_1.Rule(this, 'Scheduler', {
            ruleName: fn.functionName,
            schedule: props.schedule,
            targets: [
                new aws_events_targets_1.LambdaFunction(fn, {
                    event: aws_events_1.RuleTargetInput.fromObject({
                        calendar: props.calendar,
                        searchTerms: props.searchTerms,
                        stageName: props.stage.stageName,
                        pipelineName: props.stage.pipeline.pipelineName,
                    }),
                }),
            ],
        });
    }
}
exports.ChangeController = ChangeController;
_a = JSII_RTTI_SYMBOL_1;
ChangeController[_a] = { fqn: "@cdklabs/cdk-codepipeline-extensions.ChangeController", version: "0.0.7" };
//# sourceMappingURL=data:application/json;base64,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