import { ISecurityGroup, IVpc, SubnetSelection } from '@aws-cdk/aws-ec2';
import * as lambda from '@aws-cdk/aws-lambda';
import * as secretsmanager from '@aws-cdk/aws-secretsmanager';
import { StreamEventSource, BaseStreamEventSourceProps } from './stream';
/**
 * Properties for a Kafka event source
 */
export interface KafkaEventSourceProps extends BaseStreamEventSourceProps {
    /**
     * The Kafka topic to subscribe to
     */
    readonly topic: string;
    /**
     * The secret with the Kafka credentials, see https://docs.aws.amazon.com/msk/latest/developerguide/msk-password.html for details
     * This field is required if your Kafka brokers are accessed over the Internet
     *
     * @default none
     */
    readonly secret?: secretsmanager.ISecret;
}
/**
 * Properties for a MSK event source
 */
export interface ManagedKafkaEventSourceProps extends KafkaEventSourceProps {
    /**
     * An MSK cluster construct
     */
    readonly clusterArn: string;
}
/**
 * The authentication method to use with SelfManagedKafkaEventSource
 */
export declare enum AuthenticationMethod {
    /**
     * SASL_SCRAM_512_AUTH authentication method for your Kafka cluster
     */
    SASL_SCRAM_512_AUTH = "SASL_SCRAM_512_AUTH",
    /**
     * SASL_SCRAM_256_AUTH authentication method for your Kafka cluster
     */
    SASL_SCRAM_256_AUTH = "SASL_SCRAM_256_AUTH",
    /**
     * BASIC_AUTH (SASL/PLAIN) authentication method for your Kafka cluster
     */
    BASIC_AUTH = "BASIC_AUTH",
    /**
     * CLIENT_CERTIFICATE_TLS_AUTH (mTLS) authentication method for your Kafka cluster
     */
    CLIENT_CERTIFICATE_TLS_AUTH = "CLIENT_CERTIFICATE_TLS_AUTH"
}
/**
 * Properties for a self managed Kafka cluster event source.
 * If your Kafka cluster is only reachable via VPC make sure to configure it.
 */
export interface SelfManagedKafkaEventSourceProps extends KafkaEventSourceProps {
    /**
     * The list of host and port pairs that are the addresses of the Kafka brokers in a "bootstrap" Kafka cluster that
     * a Kafka client connects to initially to bootstrap itself. They are in the format `abc.xyz.com:xxxx`.
     */
    readonly bootstrapServers: string[];
    /**
     * If your Kafka brokers are only reachable via VPC provide the VPC here
     *
     * @default none
     */
    readonly vpc?: IVpc;
    /**
     * If your Kafka brokers are only reachable via VPC, provide the subnets selection here
     *
     * @default - none, required if setting vpc
     */
    readonly vpcSubnets?: SubnetSelection;
    /**
     * If your Kafka brokers are only reachable via VPC, provide the security group here
     *
     * @default - none, required if setting vpc
     */
    readonly securityGroup?: ISecurityGroup;
    /**
     * The authentication method for your Kafka cluster
     *
     * @default AuthenticationMethod.SASL_SCRAM_512_AUTH
     */
    readonly authenticationMethod?: AuthenticationMethod;
}
/**
 * Use a MSK cluster as a streaming source for AWS Lambda
 */
export declare class ManagedKafkaEventSource extends StreamEventSource {
    private innerProps;
    private _eventSourceMappingId?;
    constructor(props: ManagedKafkaEventSourceProps);
    bind(target: lambda.IFunction): void;
    private sourceAccessConfigurations;
    /**
    * The identifier for this EventSourceMapping
    */
    get eventSourceMappingId(): string;
}
/**
 * Use a self hosted Kafka installation as a streaming source for AWS Lambda.
 */
export declare class SelfManagedKafkaEventSource extends StreamEventSource {
    private innerProps;
    constructor(props: SelfManagedKafkaEventSourceProps);
    bind(target: lambda.IFunction): void;
    private mappingId;
    private sourceAccessConfigurations;
}
