#!/usr/bin/env python
# -*- coding: utf-8 -*-

#
#  This file is part of the `pypath` python module
#
#  Copyright
#  2014-2022
#  EMBL, EMBL-EBI, Uniklinik RWTH Aachen, Heidelberg University
#
#  Authors: Dénes Türei (turei.denes@gmail.com)
#           Nicolàs Palacio
#           Sebastian Lobentanzer
#           Erva Ulusoy
#           Olga Ivanova
#           Ahmet Rifaioglu
#
#  Distributed under the GPLv3 License.
#  See accompanying file LICENSE.txt or copy at
#      http://www.gnu.org/licenses/gpl-3.0.html
#
#  Website: http://pypath.omnipathdb.org/
#

#
# this module makes possible a
# dynamic data integration, download
# files from various resources, in standard
# or non-standard text based and xml formats,
# process them, sometimes parse html
#

import bs4
import os

import pypath.share.common as common

_fonts = open('fonts.css', 'rb').read().decode('utf-8') \
    if os.path.exists('fonts.css') else ''

_ebi_logo = u'''data:image/png;base64,
    iVBORw0KGgoAAAANSUhEUgAAAL4AAAA7CAYAAADFCTD2AAAAAXNSR0IArs4c6QAAAAZiS0dEAP8A
    /wD/oL2nkwAAAAlwSFlzAAALEwAACxMBAJqcGAAAAAd0SU1FB+ADCg4dOsFzszIAAAAdaVRYdENv
    bW1lbnQAAAAAAENyZWF0ZWQgd2l0aCBHSU1QZC5lBwAAIABJREFUeNrtvXecXdV19/3dp9x+7/Re
    NEVlNKMuVEAFdapEBxtjg3mMYydx8qY5tp/ksZ/ESRwnb5wnTvIkccEJwZjYBoMpoggQQgWhrlGd
    ojK9lzu3nrLfP869d+6MhCyC3/9m8RHT9zlnn7XX/q21fmttIaWUzMiMfByRgJCAREoBQoCUCAQI
    kFKCEAhsJIrzqwDC+dA3epFXm/+DQx1vU1/YxI7Fn6O+eIkzbHp4AGxAybpm+nticrDMD2Xm+3La
    T5kcZUZm5L8nNmALiUQACkIA0sootqN0KbWTChLLWSSpbyWNKG+2/JS9l15FCoWzg8384uQPmIiN
    ZJRd2M5HiZIZO712pBRZy2NyRUmpkDbp4ir3PaP4M/KxLL0iQUmptgMeBEIqCCEySidFtsKpU1Rx
    PDbMyEQ/FiYuXUMKm5HoAH1jlydVVHE+CgChAiZSWGkVn2Ld0zcmhI0QMnVP9hW3rs28vRn5b4uj
    7SkoM6nsKMJRNqmAAGFLUAQWFgdbX+Jc72Eq8mZzY/2d5AdKKfAVY1uSsVgYRRrkeXIpCc3CtA0+
    aN/Jma4DVOfPYU3DvXi0IEJqiDTiEUpm0WWuT1rZ1Q+99RnFn5GPZ/SFQCCQGcurOKBEKhlLLxTn
    k3dOPcVLp39Gwo5gXtrNWDzMrU2fYuP8B5BYnOn5gPK82Wya9yBul4d9rS/zzOG/R0iV93v3MxQf
    5N4bfg9dcZTdxkZIFYRECAkSB3KJ9B4kHOh1FbAzo/gz8jG9WgdnCyFACqRIwQ8hpqibaSY41Pke
    MXscTdVImHHO9+5nedVN1BYv5e4lv8mtCyLoqougp4ChiW6OXNxJ3Izhd/mwpc2RjvfYuuCz5HsK
    QKRgk0hjehBCpnzslIN9DZnB+DPy8bCOBJkJ08iUnVXoH7/MB22v0NZ3DABVdaEpOknLRNgSy04i
    FBeK4gIgGh/l8uAphicGnd8XGm49SMJMOgvHMnEpCi7hBgFdg+c4fPFVeobOY9kWQohURCkdXWKa
    wztj8Wfk16n66YAKEoTEtm0uDZ3mv47+I0ORfnTdy+1z7+amufeztelh4scidI23UR6oZk39Diry
    5nCqey+vnXmartHL5PuKWDf7TtbU3826hnvpG++gP9pDga+I2xofwe8OcPzyLl458ywjkQGCnhDb
    mz5DY+UaXKon4+dmudZwlYDmjOLPyPWjmqvGzUUWqhDEjHE+aN9Jc88h3JoXC5PXzvyUxvKbaSy7
    kaArl7H4EB49SE3+fJJGnA8uvkVz9yE0xcVorB9FUZlfegPzy1by6Rv/mHAyjEfzUFfQiCUN3j7/
    HO1DzShSpSt8kcILr1CWV09JqOYqEEdm7UWTUf0ZxZ+RX6Hrqf8LiZACmTbxEoQQWLbBWHQAt+bD
    78nFsg2GIv3Y2PjdPpJmkqHoAAkrQZ6iUVu0kEh8FL87CEJlZKKX8fgIlmWR4wkSNsYYjY8TS0RQ
    gyr1xUuJxofxuEMoQiNpGYzEBlAUF17FTdSOMxIdJGFEAUgYUcKxEYKeXNwuf+YJpru4M4o/Ix8O
    Y2Q6Bp+CC2kILRyMMzzRzQcX3+T8YDP5viJW1WyktmARCypWc7L3ANFkGKRg1awthHz5DE/08G7r
    C3SPX6TAV8jq2jspy61ldtFCWvuPMxIbwOvy01C0kMJgNcPRPvacfZ7O8Qvk+ovZOGcHZTlzWFyx
    loGWFxlPjOIVGg2lqygKVjIwdpH32l6lc6yNypxZrKjZSmV+wzSQI2YUf0Y+yiJQskymwLQSfHBp
    Fy80P8l4MoxHdTEU6ebx1bXcULuZuBGja/wCAVeQG+u349N97Gx+kjfP/pyIOYGUNnEjwV2LP89N
    9bejqzqXhlsp8pezYtYGdFXjnfMv8fyp7yOlQFUUEolRHln9VTbMeQBd9TIw0UWRr4zVs7ejKi7e
    Pvccu1p+Tsya4GS3n3AizIMrvoRHC844tzPyEaBOtpMowLIsVMVJCoVjo7QPnmXMGKXUX0o0GeHi
    cDt9oxepL1vO1gWPkExGcLn8AFi2wYnufUgpKfQUMZQYpX3gBP3jl5hbegNbGx/BNA00TQdgINzB
    qa4DWEJQ7C0kYcU53vU+9yej5AfL2L74CZLJCVyuAAC9I220DJ8haZkU+SoYjg/SNnyGofFuKvLn
    AWDYBrqizyj+jFxfyNIwE1wcPknPaAdBdy6zSxbjcQUo8JfgUdyMJ8fAluR7Cwh4CrCsJJeGz9E3
    fhm/O0Rd0SJ8eoDyYC29Y51EzDC2NMj3lxL05GPaSdr7TzEw0UuOL5e6gvl4NB8loWoYOEbYmEDa
    NnMK5qKpbhJGlPahU4xG+8jzlVBb0IjXE6LQX8IFRWUiMYEqNAr9JfjdeSSMKG0DJ+gPd1EUKKG+
    eCnqN77xjW/MvNwZuVYg53TnPn5y5B94p/UFTnbuwaVozC5eTJ4nn1hijIgZoyw4i/X1tzO/fDUt
    fR/wo71/xu62VzjWsQeP5qauaCEFvlKGon0YtkFZsILNDQ8xu3gphy+9ybNHvsOu8z/nXO8hVCFp
    KF1Frr+Y4XAHloCiQCV3Nn2aivw5HLzwKj8+9Pfsa32F5s73CHnyqCteilf3MpEYImElqc1vYOPc
    u5lVMI9DF17jP97/Fgc73uJ03weE9FzEDC15Rn6V/POuP+Bo70FcuodYPExlqIpPrPg9GituIm6M
    0zd6CZ8rl6KcKiYSw/zXwe+wq+0lCnxFGKaBR9P46m0/ojhYwXh8hKGxTgr8JYQCxYzHhvnPg3/N
    wUtvkOstYDwxTmmojt9e903K8+pJJMbpHm0jN1BGnr8Uy7b4i1cfoWOsE7/mZTA+wKLi5Xxq5Zep
    yJ9LON7P8EQvub5ScnzF9I9d5On3v82hngMUe4uIWglyPKEZqDMjv9riSwV0oYE0QQVLCOyUvbSl
    IOQvRVc1bGkhpIIiQVddgI0qJKriQtgmUpqoikpeqAwhtBT2TyKEQBE6Ugo0RcetathWHAATyAtV
    oQkN27aQ0kQTbjQhsKSFprgQippJ0CrCTY6/FFVxO98Sjn+iqW4s28alqAjxMeL4U9lwU2dLZoe9
    yKpTYJKiKrJn9grJKji46tcf4a1d+1uTP+DDfvjRL/trGUpmxRQ/4miZTTxNDU4NMX0kmZ3lvMZl
    VlRupGO0jeHYAAoai8pvpLawiYGxS7xy8kd80PkOFTmzuLXxMyyuuplFszZxpOcAccsAabOuegN5
    gXL6xrt47sh3Odd/nIpgNXct+TzzylfRWLKc1sGTDMVHCWleGoqXUF7QQN/YRZ478o+c7T9ORbCK
    +2/4HeqKl7CycgO9E11ErQRuxc3iirUU5VTRPtDMKyef5Hz/YeqLFnH7gsepKZzPkqoNnBs8TZIE
    muJlQ912RCwSlUKIjAcvP+wFpOZRAooQ6C5XptLGMExsy8KWNqpQ0D1uFCGuWAgSmSEVZc+xZVoY
    hpGZeF1XUVU9c2nbMjANa8qLSlf1TF2EThZRUVV0TZty/RSLCWmbGIaFbUk0TaDpGkIoKaLVr9dU
    Os+VnMwmpui5YtoCkVO1Fk3XUVXHMtm2iWGYSDvr2a9yrfStq5qGpmlTFTmt+NLGNJKYlo2qqGia
    jlDSBLMrFd+5XcfonO49QGvPEYqClSysvhmP5uO5Y//My6f+E6/mxbQNKnLq+J0N36YgUMK57iOc
    HthPsbeS1XU7UDSNf3v3qxy+/C6KqjGeHGdl5XoeWv7/UBaq5Uz3fs73H6U8t4YlNVtIJuK8cPxf
    efHcUxR4iolZE8zLmc/vbvlHPO4ARy++ScfAOaqKGllQuRrLMvjRgW9x8PIu3C4viWSMxeU38cTa
    b+DWvZzu2k9b/zGq8htYXnsL2u9+4QsZ5VEUBZniVzukH+lk6oSjqgqQSCSoq6/nN37rtykuLyUa
    jfLUv32fvfv243a7cXnc/Pk3v0l+aXHmRaZL0MRVTEoiHufdt97hP374JB6PB0VR+PRjn2btxk0Z
    fdn/3l6e+c9niEQmcOv6VKtlO0UGQlHxuDwUFhayfOUNbNi2FY/Pi0AgbIlMKdzQwCB/97d/x/mW
    VjZv3sj9Dz1AUXEpvzadl3bqvhUOHtjHj37wQ5LJJLqmoWZlEdNW2VFuO7N7RiIR7nvwQbbdfjv+
    YICTx47zX888w6VLHfi8bhQx9W/T78vj9hHKy2XxksXcesftBHNykekZTz1ceGyMnz77LC+//ApN
    85v49GOPMnd+w4c+iyJSpYLA/JJVzC9Z5eweimA40s1YZBBQ8Og+kkaEmBmlf7yDgkAF88qXMbd8
    BTJVqJKwIvRNdOLSAyiKJCD9jE0MMR4boiy3joaKm2goX52iGKuMmQN0j57DrwfQhYrQAwxG+xi3
    JlDxsqRmI0tqNpM2owPjHYzGhxFC4FNdmGqS8eQYQ+EuqgobWVC5ngWV61PG20T75ZtvOKvbtDBM
    M8WrZtIy2SmLgHSSDvE4S5csZcf991NcXkoymWT3u7vZ+fbbuL0ejESStWvXcc8nHsTtcWe42NN3
    kPQO09XRyfe+/2+8sutNAl4fmqKybNVy1m7ahI2Fgsr5U2fYuWsX4fA4ArBMM7MgHcvvjKsgEKqC
    5z89LGiYzzf/7M9ZfuOqKRTZweEh3tq9m6PHjuFya2zZtoWi4tJMhlJ8zCUgxCQku3i5gxdffgVT
    2igITNOcUo2UrbxpxR8dHaWqtoY1N6/HHwxw+dJlXntjF63tF/F4XBiGCQikIjILQBEixUeXeF1u
    /v47/8BfffuvuXHtGvRUXBwB42PjHDz0Ab98+SUGBwbZtHnjpOKLq8BXWyAVsCyDA60vcLJrHyU5
    1aybdz9FgUrKcmahC42xeBgFi1nuEqry5mFYBu+3vMjxrr0Uh8rY0vQZ8nwlNOQv4PL4K2BKokaU
    ioI6ikOVxI0oB1pf4lTXPirz57Ju7j0EPEFmly7naO9RNKGTsOMsKLuZXHcumEneOPk054dOUVc0
    n3Xz7qckp46yYBXn+48zEo8gJJT6KyjLrWciOc57Z39GW+8xaooWsGXBI2guXce2TcqLy6mfMxu3
    7kJaFgiBisCSEinsDAsvkTCYPWcOpSmLLgG3242u66m/tXnu+ee4bcedeDzua0SHHent7eXk8ROE
    vH7cuo6mutBVB6YoqQoaTVXx6hoJRaVx3lzKSkszu5MQzsJMJg2Ghke43N3JyMgIzc3N/OGX/4gf
    P/ssZeVlma1cU1RcmorP7caruVBSiurgYMGvE+VrqoKuq7gUjfpZtVTVVGNbNto0QJkdWJuYmGDp
    kqV4fD4H9mkqHl3D7VKpr6lm9uw6kKCiOn8nJZaUjI6Mcam7k8HhYVouXuAPfv/3+cGTT7JoyeLM
    LQmh4NI8uNzeDJy61iKUimPv3zr1NL849wwJI0y8czfD0UHuW/JFNsy9DwU4cultynJr2Dz/Ebzu
    APtaX+SHR/8WDQWzx2Yg0sPja/6c7Uu/iKZ6ONd/mMq8ejY3PEKut5hfnvgBr555mrhMcnzgCD0T
    nTy25n+xpeFhpJHgdP9hKkN13LnsN3GpHp59/9vsansFWzE52P0uE0aUOxZ8ljsWPoZP93Ku9xD1
    xYvZPP+TGJbBm2ee5rnm7+NRAxwfPs5ovB9NlSpxI0FDYxN/+Td/TV5uHrZtT2btpnlrUoKiKnjc
    7ikKLITArWrgcnH8VDM9fb3k5OdNh5mpzx3LOjI8zN49exgaHiGQE4IUJLnSigosBNFEkk999rPc
    fffd6C4XUtpTHDkBtJw/z7e/+Ve88e47tF28yH/86/f58v/+00mYjURVNSzHoGUs8BSn+2MmfCY/
    U7BNidvr5tZbb+cPv/LHRGOxa64vKSUulxvdpaegHFgSTFuydv0G/vpv/4aJiYlpPQVspC0ZGx3l
    T772NfbsfpfW9nZe+MUvqKiooKCoMLMzKCk8rwglXcX6ocEKB9xKmvuPYhsGXtWLRKF18BSXRs6z
    pHojtyx4jC1Nn0ZBoCkuxhPDnO1+H2FDQA9iCJvzA82MRgcpyankruW/hZWCg7riZjDcTcdIK4YV
    J8+TSzQZoXv8Mn1DZ5hVvJTty3+bO6SNIgWaqpMw45wZOoqmgkv4GVclbQMnGIp0UV3QyD3Lv4SU
    FkKoaIpOz2gbLb3HcOHCp+rYQuFoz340RUiwJSqSQMCP1+/7iGltga0IDGlRWVVJMpHkTFsLLz3/
    ApXlFQRzQlPVPQvrd3Z08uZbb+Fyu2iaPZsLHZeZiMZRpnnYtnBerhBg2SYenwu3++r3uWTZMv7o
    a1/h4NHDRGJRDh36wKm+TK08hRSmBlQhUFLRol8Xxs9e4Q7+tkGRCAUUTSUQDHykfcOBIKTu2fFn
    AoGrjxEM5fDV//mnnD33GUbHwxw7eIjhTw5nFF+mFr5UBCjKlAWYKQ5PWf10OxBQCLpDJElgS5W4
    MYEvWEfAlQtA3/gl2vqPkecvpaniRjTFRcBbSDQZxqd5iZpxij35uHUfApWOkfNcHjhFWU4t9cWL
    CbpD6LqXqGWgmwksy0RRFHw+5577Rtpp6z9BcU41DeWr0BSVoJLLZeMyQoGEEcPvysGtOfpwefAM
    HcPnKM+rZ07JMlyqF787RMSK4ZJuTMumyFOAZgmQiqMMtm195NickBLVkggkBTl5zJ49m3MXWtj1
    2k4e/exjBHNCU1tMZEl7aysnT5ygclY1W2/Zxr98/3vOixHTnSxSlsp2HFXr6lYqHegpLi2hOK+A
    1nCYcCzqqHaWRU9HsaQtP6xA59dh8FOcRicsIKT4SGHhyShmqnpUCieykzX65GY3CVFmz55Nbk4O
    aCrj4TGSRjw7EoCCRLGdghE7w1Wf+oqFSDvhjlHY2vhJoslRzvc3MytUx5aG+6grXsjJzr28cPx7
    dE5cJqgHubF6A/cs+xJr5t5Lz2gbF0ZbKQ+UsX3RE+T6ijh28Q2eb36SvsgAbkVnx4LHWD9nBxtm
    7yAcG6Rl4BjVefVsa3iIAl8ZzZff5anDf0fUiKLqXu6a90k2ND7E9kWPkzj6XbrHLzGvsIGN8+6l
    OFTDnpYXeO3MjxmM9VPgLWTz3AfY1PAgmxoeYDjSS89EJ1XBKu5Z+iU0ka6SB6dvSdakZsJc00Jj
    6fBgBgsKGykVTNvkvoce4PlXX+L0uXOcPXWaguKiDJbMfsm9XZ3se28PkWicebNns3rtOr7z3e86
    zrAtp+mSnLy8TcYBF1nhzDRV1vEJNBJmEgkU5uZloNWkMqbCq4rMKpv7OJGcLOWV0xdZanGltXba
    jiA+DPeIyWd3WtaIKWtUZhWAyGlzIS0byzApLC7B6/FNXYgyqxmHFB9q1yYbOUkq8+fx8KqvETfC
    aKqLfF8Z0eQ4x7r20DZyFq/qZSwywJHu/dxQeyvVBfN5dM3XiRtxNEUlz18KtsHbbS/QH+5AVVTG
    EmMcuvQms4uamFu6lEcD/5OYEcGtusn3lxKOj/Be24v0TXTjc/lIxmPsbnuBFbXbqC9dwufW/yWG
    FUNXXRQGKhkOd3Kk8x16whfxqD56wx0c7nyHheWrmVe6gt+4+S9JWnF01UO+vwzFxmn/oLp0PB5P
    5rEzL2R6kyoB0wvXZWo7TiQTLFyyhNk1dYzHY7y28zXGRkYys549THv7Bd55Zzf5+fncuGoVZeXl
    WKblvGNFuWLjkVJgS4E/FMTr81/hA4i0atuSnS+/QldvLx5d544d251lI6+amvjoSSGnhVLWP5nq
    35L6OmshZaZQCNwpn4irwItrR4lSda1S4vF6rraxTPn8uZ//lIvdl7FMk01btlBSWnbF9a43JZbO
    lyhCJd9XhEfzkeMtRNfcSAlRI4phJXC7XKAqxIwElm0AkB8ox6Np5HoLcGluQBKLjSGkQFfc2LZF
    wgxj2glAIddbgFfzEfIUoKkuDCtB3IhjSAtdcQMKkeQoEomq6OR7C3Erbgp8ZShCxbZtosk4pm3j
    Ut3YEqLJGLblhIrzfWW4hZs8TyGaoqEJbDRFYWx0mAN79hAMBrFsM2uLJRM6NE0Lr9dLbW0duYUF
    TGaEU7FyKdHdLjZt3MihE0d5+61dPPY/Hie/sNCp4MmKdZ85c5rW9gssXbqEzVu3kUgkruFkCRA2
    uiZoOXeGfXvewaW5sC3LccAEJA2DgcEhjp88wauv7sTj8XDPndu57a4dU6DOf1fhs+FEupA5/bkT
    WbpSDW0pUVQFW9pc7ujg+OEjxONxpwHHtOcUQpBIJigpLqaqth6315sZA+Hw0Xu7ejJjgJVJEhqG
    zfDgIBcvXuCnz/2M2HiETz70EJtv2YI/4J82j+IjLH6JEAoD4Q52nvkJl0bOEtRz2DjnHprKV7Oo
    bBVt/UfoGe8g4M5hcdkKynLqGAp38dKpJ+kcvYDfHeKWeQ8yv3wVa+q28+KpHzEY68OTyrhW5M6h
    e/g8u87/jPbhFkoDZayvv425patYUbuV1sFmxmJDuHUPa2p24NNzuDx8ip2nf0J/pJsiTzFbGx+m
    On8eS8tX0jvaxmCkn3xvAUvLV1EQKuPS0Fl2nn6KwYk+CgNF3DL/YTSEgu72cLallb/4y28hFAWE
    TMXvSXVMcVKviXickuISHn/8cW7bfmcGV8tUOwkpJZZlceeOHfz4J8/Q0t7OyRPHmVVfi8vlykz3
    +XPn2L37XSxp09TYyLymRo4cOuwoghBM582lGxa53B5e/OXLvPPuu5l2FiLl/ZmWRTg8ztDwMLa0
    uW3LNr72J39Cbl5uphPAdD/jel6+uCI7zJReLeIatAgHoSgYhslb777D8VOnUzvDZFIve+zIRITl
    y5bxB1/+MrNqazIDqoqCqml8cOgQf/hHX86E25ykqsC0LCKRCQYHB0lEY2xcu56vfO1r1NXVT7sn
    wRXb0XXInpbnOdD+KnErgmEaYJsU+stYPmsTHt1Lx0gbIU8+TaUrUBSNA+0vs7v1BVQ8WIqBZcSo
    zG9gRf2teNwh+iY6yfcV0VS+CiE09rbv5L22lzFsk8vDZ0mYccrz5rF81mZcipveSAdBTz5LKtaB
    Injz9LMcvvwOFgat5nE8uo+7lvwGa+fcRZ6vhL6JLooCZTSVryJmRHiv5Rfsv7ATXXXTOaZjWkk0
    iYIqBOFwlOHB01hpFGyL1AuSqTYlgng0ynBlBUNDg9PNYqYsTdo2c+bPY9H8JlovXuDVV19l5Y03
    UlldlZn482fPcfD9g5SUlLJ8+TKHjGSYjgJk+RxZbpzjaikqPd29dFy6RHZD0Mn+Lja2beN2u+nq
    6+Xlna+wY8dd5BcWXN3kX4OYejWLfL2+bXZDO0VRMG2bgYEBuju6Uws73fpFydy3IgTjo+PkBHOJ
    xxNTr5u69MjICD09PRnnfAodATBNE1VV6R8d4ZcvvcRDDz1AaXnF1HmU1/8saVNxof80tmUQ1EOE
    CdM1doH+cAfl+bNZWr2JJZXrEYqTexmLDXJp+BxJ26LYFyJpx2gfPEM0PkZJbg2r6m/FtOJoqgPb
    hiI9dIcvkTDjFPiLGU2E6Qp3MBLppaZoESvqb8Uy4qi68/sJK86FwdNoQsWnehm2xugcPs9YpJ9Z
    RQtYVX87lp1ETbUt6R5poXPoLFIohNwhkqbFuf7jaEJIzESSuXNnc/fdd+H1B1Isu8nAvUxNlGma
    BENBVq5cOTXqcpWOtDffvIG39+9lz3t76e7uTCk+xGMxmk+cpKe3l81bNrN+482TBkhM8y+ywYNt
    YyTj7LhzO0uXLMbt8ZDK32QWRiKZpKuri4Pvv8+efXs5fuIEvd09/OFX/tjhr1xpzq9p6acr/+jI
    CD/9yU8wDCOTOEuH/kzTICeUw4bNm6meNSujk5Zloesu1ty4mm3bbiWZTDr5B4TDk0lDIgSJRIJZ
    s2ZRWloy5eFt29lJl61ayZ3bt2fa9klFICwbhCBpmAwODrJ3316OHjvOkePHGOjt5ku///uUlJRe
    xWm+/thsTWETbSPnGYmPYkuT8pxaikOVmGac5q79XB5tIejNY0HpSvL85dTkN3Gw6x1G48PYWCws
    XIrPHSJmjHOiaz89Y5fJ8xayuGINQXeIskAFzaqL4fgQAoWyQAV5vhLi5gQnLu+hb7yLgC+PZZU3
    E/DkUVvQRF/HLiJGFNs2qcitJ9dfRCQ+zImuA/SFuygOlrGgfCU53gIq8+dyavA4I/ERVKGzuHgV
    GtgkLIPq6io+/9u/ia67r3NKUvZATs08pj/bcvst/Ojppzh69ChHDh9mwYKF+AIBmk8cZ9++/ei6
    zuKFC6mpm32F53yFRUp9nUwarN9wM5/6zGc+9L6iExF2v/02//xP/8T+gwf5wQ9/yJbNW1i+euXV
    F9R1wJz0DtDT3cPXv/4NYtEYiqpOSc7FolGqq6spLa/IKL5IEfDcHpUbbriBTz/+2EdIB2TPr41t
    WTQ2NfH455/40L8xLZP339vH//m77/Du/n38+7//BzeuWccd27dfScuU17d7Aayds52YFefS6DkC
    ei4bZt9FcWgWhy/t4sXj/0b3eAd+dw6dNa08uPx3WFm7lcFoN11jl/G6gmyb+wBBbx57zv+cF0/+
    iMHIAG7VxVi0n23zH+amuttJWnEujrZR5Ctlbd3t+N15vH/xZf7r8HeJJCO4dS/jkX7uWPg5Ns3/
    BIaA4Wg3+e5C1s29B787j11nf8Jrp59hODJInq+Aoci93Nb0KGvm3E3EiDIY6SM/WMItDQ+iSQsQ
    CrZlEZmIkpvnzmZic2VfEieMk90N92pSXlXJ0sWLOXmqmddf2cmGjZtoaGyk+eQJTpw4QV1tXQbm
    ZBZO2oR/yLYrgUQiSSwaxevzTXE80+IL+Ll1+x0YSYNjx44TjkT45S9eYMkNy9H0j1d+4Pf7Wbt+
    HYl4AkVJQS3pzFEikaSkuJj8/Pwr1qwtbcxEckqu4VpxfDmlv/VkUMBIGtOo2mLKItFUjTU3r2d4
    YIjz7W20trXxwYH3WblyJUWlJcjMf3AH9IoJAAAWTklEQVQ99UcOY1VSklPDQ8u/xEikF787F78n
    l3BsmOPdB7g4doFCXwnRZISjXQdYW3c7tcWL+eSNX6N//CJ+bwG5eg5SmrzX8hLhxDghb4iR2AjH
    Ot+lqXwl9cVLeSjn9xiZ6CfgzcfnDjIU7uJQ+5sMJ0bI8xSStBLsaX+RDfMeoKZwPo+u+DLjsUFy
    /IW4dT+DYx0c6dzHYGyQfF8+I4kRjnTtZ9WsbdQUNvLYTX/K6EQXQW8RXncQTRVO+01hZxsBkUk6
    Sa4MgUkxjfGa2rIdJuakbNiwgdff2sXRY0fp6e5hdn09p5qbGRoe5NZbb+WmteuuyBqKqzi3TpOu
    yWukAe6HYVWBYPHSJfg9XkYSES5euphhcTLF0bt+B08gqKio4G++/ddYltOyLk21FqlssNvtpqC4
    eNpfTdsVp/Hrr5q8mjIHKT9GTEK6NMCUU2ogJl33BUsXEQyFEKrg0sULjI+NUVRakllESpbfcO0d
    L73EbMZig0SMCQxp4tZ9KELBo3nQVY2kHUNi4tW9qKrbUYqL5ynsGkbkjiHnzEPqOm53AKnYJCyn
    +MSjedFVh5oxEhtkwhzHilm4NDeaquJx+VHRMSwLW1oE9BwUIbCsJKPxfqJmFDsqKQm5URQFr+ZD
    FYKkHUdB4FN9KEIFLEYi3UTMCHZMoLt8aHbaUZLiGs6anFIGIsgK7YmUE5TuX5ilrZu3bqHhqR/x
    6utvcPr0aYxojOMnmsnJCbF40ULyCwuvcAqn7CIp/rqTQbQnoz7C/pUvLTw6hiEthzPi8aXGSS9O
    x28R6QTE9dSyCFB1jVmpSMn10BZkxi47HXz5kAz2hy20tDWXmZ1QXhWKTI9XjY0MY5oGSInP67li
    pxNZ9IUrjFr6vWaZv66Rc7x45F8423eMnEARdy54jJU1t7Ck4iY6h8/QGe6iwJ/Pkso1VObWkTx9
    mL7/9RXi+86iVIYo/pOvErzjYdbX38lIfIDh2AgFWoDl1ZsoCdXR0neUV5p/yLm+o5SHarml8RGW
    ztrAqrrbuDDSStyIoGp+1s/eTsCTQ2vvMZ47/n+5PNJOeWgWdy95gqbym1hetZ6hiS6G4gOUhapZ
    VrmOwmAl5/s+4OeH/y9d4xeoDNayfdkX0JQU7UCooCriQxIk4kPxr0glJYWiTM0sShtfKMjixUvY
    9/5BXn/tNd4Phmhpv0BD0wIWL112BedHpuLhMmsnSXN10pZTUxV0Tb2m0owMDvP6ztcIR6N4XC4W
    L1uSYWFmLy5FFajXGEv86qKua64WkTV34iMyP7PDr5nwqKpe0zKDQyF+5623GRoeRlE1GpoWkldQ
    OGmwAEtK50CHaSeGXAm7nId5s/lpjvQeRBMqF0dbeOPMsxT5K1hUdTN5vlJa+4+Q5yujoWodIjzG
    yH/+C5HnDiHIh94wvV/+c7wr1nND7a3keovpGDpPaU4lc8tXYtkmu1te4FDnXoLuHNpHWnjl7NPU
    Fi1iSfVGcrx5tPcfpzBUx6KqdSANfnnye5wdOoNHddHcf5ics7kUB6pYN/duSoOVdAyfpyKvnnnl
    qxie6GbX6Z9ypO8IRb58WsZaeP7QP6BZ0rFGtg0jI6PYqUSUmAJmsiGQRFVV/H4/qqY506k4PdGz
    IUp6Ajdv3srrb+ziyNGjuFwuopEIK5bfwNLlS6dCGUDYNoqtpDKxWZt6KrJko2DZMDYaRnfFHa5N
    xjY5145Eo7yz6y3+9fvfQwBVFWXcft9dKKpIhUQnj46xbMH42ASR8ARJI3nV3Sd9f16vF4/Pe/1O
    adYYiiIwkgbxWJxYLHrNxWNLicfjweP1phZ+6ogdBLZpYZkW4+Njk/5X1kKzbZv33nuPp378NCND
    w9RVz+LGm1YTShEFhZ36l16QyodHskSawmJLwolx3MKNpqhYqk00GWYiMeb4crl1FAWrUBSBrqiY
    yQTJzi5UcgAnACDPm5hjI2hlldQULaSyoAFVqKiKi9HIIHEzhqrouBQPUrWwbJPx2ACFwTIq8xsp
    z52LIpyzswzDZDwZxqt50FHxaT7CyVFi5gQANcULqSpszPT/SVgxJhIjhHQfuqKjC43h5DCakBZu
    VeHc2dN87Y+/jCfFqU9nBe1UWj59xEsymaCiuprPffELzJ4719mKLQs7lcmchA7OrN645ibmz5tH
    a2srY4lxyopKWLZ0Kb5AYNLCCOeFp6uRrCw06+woDiXA69F55umneOvN19EVdcpJGEIRxOJxOjs6
    ab98kaRtUZBfwBOffYLKiqpJiygltrTxeTycPnGcv/zzP8Pv8yNtM5WOyD7CRmClYuP3PPQA2+++
    5/qBcereVEXBSCbZ9ebrnGs9iy0lmhRXDaoIYGIiwu133M72++8jv6AgQ4lQNIV9+97jc49+JkNZ
    FTgsU5G6Rk/vAOfbW4kbSbweD0888QRzGiYrrOwUN8lCYgr7OgI7NkJRmFeylNaRc8SNMKZMUFs4
    n4rcOqLxcfa1vcCRjrcpDlSxsfGTzApWEdy8lehPTjlli5rEfd983JXVjBtj7D71E870HaYqNJsN
    jfdTmlPLrLw5nO07yIQ5hipUykO1VOTMIRwbZvfZZzjTe4jy3NlsW/g4RYEyGoqX0Nf2CjFhYQqF
    +qLF5PtLGZro4d3zz9Had4yawiZunnc/OZ5CZhcvoXmgmYSdRLE1VlXeiGZYEoHK5d5eznVcTpUb
    Zm3NDoh3srhCEovGmDd3HrfceSez58514tC2jRU3nbrZFK5V0pZIVVi+dBnv7d1Le+clmubPZ/68
    hqlKkmJPGYaBpqqYpjllC7ZsG8t2rP7x5lMYx45PSY/K1PUUCSgKXq+XuqpZfO7Rz/DYFz+PhZ0q
    aXGshmlaSBQu9/TS2tmZTYFLZVbTDEWJZVroikZlff11Kb6YZv1N28Y0DZpbznL0TDO2BCG0q1Ih
    AMbGxsjNL2DD1q3kFxRg2c4Ylm3Tfvky51taAAUrdSaIFJNloxoabk2nuKCYzz/xOT796KNZtHDn
    9wzLxLIMTMvEtKxrOtYOHUSypfGTKJrO6a4DlAQq2dDwIIWhKl4++STPnfgeprQ5M3CKvkgPv7nh
    O4Q+8UVkcpyJt95Dm1VO8e/9L0Qwj1cO/R27zv4MG5sjvQeIyhj3Lf0tNjXch0fVaO46QGXhPDY0
    PIwQ8M7Zn/Pjk98noPo4PnSKcGyUJzb8BXcv/018ipe2oXPUFzWxqfET+NxBnjv6Ld5pfRFbKJwZ
    PM5IvJ/H13yDbU2PoCoarYMnqc2by7YFj6HdsnkTahZMmQIzJFjCTkUwnAlJJpNUVVdTU1UNgMvl
    YvXaNcQtWLK4CaEoV/RD2HL7LXT1dHP6zGkefPiTNC1dPDXAIaCgoIBbb70V27ZpbGycUvk/d34D
    WzduJDYxgculZ+pMlawoUJrTEvD7WbBwIbfdfTdl5WVO4UkqAiKA/Jx8bl67jvn1c1BVp4RPSKcT
    MFcJM1qmiapprLph+UcOf1ZXVnHHbbdhGgaaojqH/akKUlpXKH5aIpEIN62/kWBuDgBV1RVs2bSJ
    eZc78HvdmYSdojgGJ7tO2uP2MWfeXO69/z7KZ9U4OyjOswkBAX+AlUuWMritl4amJioqK6/um2R2
    LafmVlF1tjZ8iq3zPpV5X8MTPXSNtGHbJoWefCaMCGORYXqGTjO3fAV5n/86eZ+fDIubVpyWnsO4
    dQ1d8SAUje7BFgbGLjGnbAVbmx5ja9NjmXvoD1+ipecgfsVNwBXEg03rcDMj0VEKAsXsWP6lKe0h
    OofP0z3RgRAq+XoeYXOM3kg3XSOtzCpoZMeSL0xd4KZpSiGZhCpX4+HbU2m/6e017f1L23S4PYpE
    UbSrxqql5USQhSIm2ZfTKLrpMKGiOId6KaRfnI1t/+qqDaebg0OsUlQlGzhnaL0CMC1rqkM3JR7O
    FUSuNMpGEdfnpMrJGlbbsjNcJnEdDnLaeitCyYwhbTvTYeJDSyTS3RZSxmBK8i0rL2NbDqRThJqB
    iFJO5Q5dbdy2vqO09h2mMFRJQ8UadOHiuaP/xKtnfoxfz8Gwk5TkVPO7679FYaictoGTnOs+TEmg
    jMX129Clwj/v/ipHu/ejoTJqjLGych0PLf8dynPrae09Smv/MYpC1SypXks0Oc7PjvwrO1t+RoG7
    iJgZoSGvkS9t/Xt8rgDNXXvpGjhLZWEDc8tWYBoJnnz/mxzq2IVPDxEzIiwoW83nb/rfeNwBWnsO
    0TZ4kqrC2Sys2IiW5sor13ol6rW3dqEoSMUJOaZhUfZkCiEQmriyvkVMfWGTtAI7o4xKilGoqiIr
    YXPtzOt0xzTdYSHTgkNVs0gWk4eW/UqW5nUovcyibguEU4sgrp8KPK2KxYl0qWoqAiOv5IT/qtqA
    dFlmKv3hvO+pcylEqgZCTPXP0pN8uH0nz574VwYiPQihcdu8B7lzwaNsmHMXscQYJ7r2U5lTx5am
    hygKVHD80n6+t//rxGUcENwx0sr2ZV/k3kVfQFpx2kZOUZ23kFvmPUh5bj17z/+c55ufpD82QI4W
    ZP3gbdy/9He5rfERItFe2ofbqc6bzT3Lv4jPFeDtU0/x3Ol/J2zEcUmVh5b9Bhvm3cu2hk9gmAla
    B07SVLKC2+Z/Co/Lx57zz/PM0e9iAl7NzfZ5F7MbSn2c4jslo6ZXbpe/Ikworj7etb4WHwFnX82Q
    TQsSXtezfxSS2tW++MizK6Y9ueCj36eY+qfiKs8qssvcrmCvOjDpwOXXicTG8Gk+YmaME117aSxd
    TlPFGj65+is8aBuphJSficQYhy6+RMScIOD2I02TvZd2sqbhXkryqvgfa/8CSzjBE68eYDTSx8me
    IwxFBsnz5BBPxmjuOc7q2nNUFTTxuZv/yjnjCoHHFcAw4+y99Aa2ZRJU3IStMEc7dtNQsoLZJUv5
    jfy5WNJCESpe3U/f2AWOXX6LuJUk1xXEsm3euvDCzOFvM3KNRSTTETcwhYkqFEgxYNNBABUYnugm
    nphAEQoSG0VVSZhxVOHCEgLDMh3/RmhYwmJ4vBPbTDrus3DOvzVtAyltDGGQlEkU6UqFsnUGw92Y
    KWUWQsGWpuO7KSqWJbFswzkxHbDMJCPj3ViWOQmhkdjSREXBxsY2jZluyTPy4WKlQv1e1cPFgdMM
    TvTg1nQ2zL2bVbW30TfSxn8d/gd+efrHNHfvR5EGc0puwKN4aO8/xmBsEEUKbm98hIUVa7g03MxT
    7/+/7Gp5nsMdbxNy51GVPx9bJukZbaFnopMcdz6b6u5g6azNdIyc58n9X2dX60sc7XibAk8+pbn1
    qKi09B1nLD5ASA9wS+OnaapYw+nuAzx7+B94/dxPOd9zmIDLT33JIqSEtt4jjCZH8epe7l7wuZmm
    sTNyLQDrAJ75Fat51BOiZ+wyOZ486osWYkuLvRdf54OOd1FUhUh8iN1tMLf4BuaUreALN3+LvrEO
    Au4gtcULUYTCrtPP0DZ4Aikt+sKdvNfyC0pDVayouYVCfzkD4S5yvAXUFTYSNcbYfe6nNPcdw6t7
    CUf7ePnM09QVLeWG2m3kBysYjfaQ7yuhtmAB4fgwe9pf5XT/ETRV4ezQIK42D7MK5nNDzVYK/CX0
    jndS5C+lvmTJjOLPyLXcDMfZ1VU3c4qXMqtgAbrQEQqMRvoZifYTtaOUekuJJGIMRYcZjw5RkltL
    XdFCqnLnoKcKSCwrQff4JWxb4nX5idsmg5EewrEhynLrmVOylPrChSipZmID4cv0jl/GAtyaj7iM
    0DV2CcOMk+MpprFsBUkjgkt3SiuHIz0MRfqwbJtcdy4j8WEGo/1MJMbI8RfTULaK+uIlKVKcMqP4
    M3LtyJLTzcGJSblSTEqAoCeP2rx5HO18l56JHjzCw/ziRZTl1hFPjrOn5RdcHr1AwJ3Durk7KA/W
    sqDsRrrGf85gbADLNqkpaKIkt4bx2ADvt79G+8hZigOVrJ61xQmblt9Ac/8RBmN9KBbcXLkerzvE
    SLSPPS0vMDDRRbG/nJtmb6cwVEF9/hwuDp6id6IPj+qiNm8O+cHJYntddWcebEbxZ+T64qpZHF0h
    QVV1ltdsRgpJy2Az+d5CVtZswuP2s7/tZV448UNMaSBRCCeG+NSKL7Nx3gOomo+esTZyPQWsrr8D
    nyvErjPP8trppwknRnHpAcZiQzy07He5qX470hZ0jreS5yllw5zt6JqLt5qf4e1zv8Q046AILGmy
    tfFTbJx3Px49SMdoGxU51ayYtQWvHiCbzyVmTj2ckV+p8tmc7Ksc8FwQLGdTw4Msi27ErXsIeYsI
    x4c43f0+48kJCnz5JE2DYx172b5wiJJQDdsXfpZoYgyvHkTVdEYivZwfOMlQfIgCbxHh5DhnB44x
    GL5EVWETOxZ/nnB8CK8rhKbqmFaC4517MWQSryvAUGKYM70fsLx6E1UF89nW+DATiWGC7ly87lxS
    DRazwpczBzzPyHWg/Oz+qVfpoIKueSgKTZIAVaGT5y1CCknEiGLZJhXBalyaDyktLgycYDDST8id
    S13xQry6j4A7hEAwYUxg2yY5egCvO4iN5MLAMQYnegl4cphTvARNuMj3ldAV7iBqRDFtgxx3AS7N
    Yc763CF87lC2jk8SDzMnlsxY/Bm5ptpfqejTk4mZzg2pHcKj+7mhbhsXwu2MhLtx6162zr2fPG8R
    zZ3v8ULzD+gebSPkLuD2hY9xY91trKzezGi0n+5UEfpN9beT6y/jdM9+njv0T/SOXyLkL2J74yPc
    NOdeNsy9l5gZZyw6QKW7jtV1tznd2kjvSCClkmkDM/U50otg5vC3GfkY8N9Oab0yjeTXPXqBjsHT
    5PkLmVu2CmlbfOeNL9E8dAyPcDGaGGVR6QoeXvEHVBc00DfSTudoK3n+UmqLFzEe6+OZD/6W3a1v
    kOfLJ27GqQyU80e3fp+gJ5eLAyfpHb9AWaiWioL5aIqWOpQj3bslC9xcpa55xuLPyMfaEhQ5WROc
    OfYJSXluLeW5tZkVkrSTJOwkLkVDERouzY1pxTCsmBMl8hUyS3Ph0jyp7hQ2sWQUl+ZCUTR0Tcew
    DMxUY+OaooXUFC3MUm4rxf26klh2NbrJjOLPyMcz+SLVyTnrRBkrDYjSxD4Jbs3Lyuq1DJ3rYcKY
    wK25aShdSUloFn1jF3m75eec6z1KRaiGjfPupzKvgWWVG2kbOou0DXTVw9LqdQQ9odQCsxEZaqVM
    Kb0yhSQ4Y/Fn5P9HL8BhdTqOo1Mco6YP2hNKps0hNmye/yia4qN14ASVOTXcOOcu/O5cfnn8SXae
    /TFC1WgfPsdIYoQvrPsWa+fuACFp7TtBWW4lGxs/k+ILMQXKOG1eVCabR0w9Wu9q/MIZjD8jH9fm
    Z0jdGcs67UhHKSQ2InM4R7YMjXfzzOH/w+Gud8jz5BNJhikMVPC5VV+jtmTxlRdLiSWcc8VECmp9
    1FOcZtiZM/LxkI6cbEGThhO2yDrcQ9gIaaagjxNVzz7Uwu/LId9fiK5oJOwkQqgU+IrID2RFaWyc
    o5vSlxWpUlI52ab9ozZ/n1H8Gfl4SCdddCMdJXWsv5g80BsF0DPaNtmJz9FVj+Znw+y7WVm1GcUW
    zC1YwB0LHiXHX+KoslRAcejPSrpFBIoTrkTNNF2d2ozPnrIQrrYk/j+2JV8XG2BgOgAAAABJRU5E
    rkJggg=='''.replace('\n', '').replace(' ', '')

_rwth_aachen_logo = u'''data:image/png;base64,
    iVBORw0KGgoAAAANSUhEUgAAAL4AAAAmCAYAAAB6QSGzAAAAAXNSR0IArs4c6QAAAAZiS0dEAP8A
    /wD/oL2nkwAAAAlwSFlzAAALEwAACxMBAJqcGAAAAAd0SU1FB+ADCg4mH7bFiE0AACAASURBVHja
    tZ1pjGTXdd9/99631NrVXb1N9+wz5AzJ4SaapChKUShHkm3JRhLZhmIgsIPACYIsX4I4iJEgH7I5
    QD4kcBYHTpxESewYXglHtiVLifaFIiWSIjnkDGcfTu9b7fW2e/Phvnr1qrq6Z8ZQCpjprnq37rvv
    3nPPPed//ue0MPYFgBCC/O+D1+Azw/A1aCuEAJO/kjUY+b4UAp22H3xP5K7nP8/6zb0Gn2ltkHJy
    m/G+QKSjzo1l8BRmMEQxPly0Ngwey97D9jFok83HSB8HtLGfHjDO/D0Ou87IMwzGe9dnNZPnygwb
    jKznpJfOrfG9zfd+ERhpk34oJq1t7h7igHGCGPnyoJ0xJp3ng8dJ7jmc8YfOC3/+syje5HbHITQy
    +9wREk9JOlHMQD5cJZEIQp3YgRtQUuBLRTeJs+/6SrFQKFB2nQOFfXwMjXbAt76/yo88dxIhJ2+M
    fB9vXN3i9nobeUi/BkPRczh/cpp6rUCSaL726ko66SJdfM3yXJmHT9cpeA7dfszLF9fpBXFuCQ2e
    p3j2kUWqJQ+Ay7f2uHxrD0dJfpAvKeHJc/Ms1ksAtHsRb1zZZrcVZM+qjWGxXuR95+ZR6f21gZXd
    gE6YZAIzXVTMT3lMmqHNVsRuJ8ree45kecbHU6PykmjDbifivd2QRjcmTDTVgsPilMvytI/vSmJt
    2GyGNHsJAFNFxcKUh5J2zZr9mLW90MqVFNTKDrvtGD1JqeZey9Me682IRNt2rhIcqfkUPfvMUaJZ
    3QvpRxqAo9M+5YLCMRN2/STNf/m9v8fbvY9zpfsY2ljNNuN7hImmnySE2qAEFB1FQSlakZ0AAVQ9
    F0cKOlFMrDXzxQLPHZmjWCmDyXTwPuEdH8PmXo9/+9uv8/HnTtqFO+B0Grz/H3/4Nr/++Uu4rhw9
    riYcTtMVn7/08XN8+mPn+RefeYVrdxqodHNFsebJ8/P80t98nicenGdtu8sv/odvcmejPXJSlIsu
    /+xvfIBPfeQBksTw4leu8q/+x3cpF90xFXbQQO7SJr3uuYr/8Pc/wsfefwKAlc0O//IzL/Py2xt4
    jl3wMNb86HMn+Xe/8ALlYioEccLn3thhtxtlp+eDi0U+emGGasHZN5/futLg8loXEBgMtaLDjz5W
    58RsIdWwEBv4zrUmr95s0w01A20nhEBKOFH3+egjdTxX8vL1FhdXuoDhsWMVPvLQNCVPghDc2Qn4
    7OvbCARlX/K+U1W+9W6TROtUHhk70QzawF94ao7/c3GXTmA3VNFTfPh8jceOVQDoBJqvXNpjdS9E
    CPjJp+c5XSgiJx0J4xp/bef32Wp+nXn9q0y7bYQQlBxFmGhCrSm7Do4UFB0HgF6cUFASR0LJVRgM
    u/2A2YLPo7PT/MyDp7lQnzlwncfHkC0IUPCcycfY2E9jQCqJ40hcR+I6Kv1nP3OyzyTKkTS7Eb/8
    W9/nF/7t1/iJD51GSpF9r+grLt/a5ZW3N7KN4iiJ44hhn64kijUvfuUqcaKRUqR9qLQfieOo9N/o
    /fNtXEfhqnyb/dcdJRk/xJSSuNl4FK6SKCVG5rMdaLY7EY4UKGk16243ZrcTj08oYaxZb4QoZdsq
    KQhjw1ojHNEYr91s88r1Fv1I40jBbNVlYcrDc+y9b22HfPnSHmGsUVKghEBJYU8mMZBju24yvaak
    NVqUxM6jELiOPXE8R+A5At+RuEoghf2eTL8XxJrbOwFaZ+oUicjubS0Tg5PXrnltOfg9iNZ4b/u/
    EsYNZLLNycIX6elP4SvFTj9EpUdd2XEAQ5AYwkRT81x8ZbV/M4xwpOT0VIU/u7w4crIcZNYcqJqZ
    bJceZCoN7OPM3iQ1nA1DAVVWO5ULDt99ewPfVRxfqLC23UUKEFLSDxO+8foKn/zgKaS0/cmBn5Kb
    s8u39vjOW+s8//hStq4CkSrC3GYWIAe2u7F9DK5q7PgGfQ9+ps322fxDu3j4nIPLgxPUGLi13R8R
    lARDq5+w1405MTs6n+vNgF5ktbpU9vthrNlohlm/7SDh6kaPOLFzvzTt8uFz07hK8PKNFm+vdBHA
    Vivi5lY/NWuGKyiyGRoK/gRbFCnh9HyR+YpLogFhEMbOR6XgZI87sFS22xGb7YjFKc/eY2xOILXx
    DxO+7eaXieIdCt6yPc7FN3io9hPc7PhM+a61/aRkynVphhFhoik4ishojpSK7AURUggenqnxwSML
    hzqvHOQ8j3hlk4V9v28ihk5s+lNrw+x0gfdfOMJ8vUijHfLtN1bZ3O2N+CNvXtnmQ08u87+/dh3P
    VYjUvn3l7Q3eurbDmaNT1jvIfMrUeQP2WgGf/fp1nn98KTURrL1d8h0++swJalUfYwyJNvyvz19C
    itFNcfLIFM89egTXVUgBL721xpXbjezI3+/mMuHcNCPAweCzG1t9lLCacq7qst4ICWLDdicijHVm
    JgHc3g5IjBWw+YpLq5/QjBP7sxczVXTYaIS0+4ndlMbw1MkqyzM+AE+fqnJlvUecQD+yJ40Uo2tm
    cltgaMiYCT6N4MJymbMLxX3X+lEy9g3BXifmzk7A4pQ3Ni1DM8kxhzgP7f477HW+hTExQrjppz0W
    +F1W5M9hkqFGEunmlelxolIv3WA4Ui7y4eVF3AMc0nE/YxxlOuxkGBf+fUCAsBMsECTGsDRX5q//
    xUd57IE5okTz2a9d55/+2ks0O2GKNFl79sET08zPFNlrBRZIkLDT7PO1V+8wU/WRUqb3tMs3uEcY
    aV69tMk7N3dwHWuSaAzlosff+unHeeD4NMZAP4z5jc9dwohhH1obLpyp84t/5RkqJTvf/+TXXuLG
    SpPEpIKSajrPHQqp5yqkHBemoRlhjCGIDWvNEDAUXMXZhSK73ZhelLDViugESSb4xsCdvdBqeyk4
    u1Dk+lafZj+hE2g2WxFTRYdGL6Efa4wB3xHUy0M/oehJPEfSjxJEDP1Q4zpiuGZYGcmEO/1cyslr
    HSeT5XQoOvZUUhKC2LDaCIlTdG5cXQghhqbOfhgrYmX7N9hqfBFDTBTvWeXmLtMLf52a8zDvBE/h
    ypiSo+h1rXMRG0OcaGqex2q3R9lxeGZhlqrnTnSiJ5k24yjTgYjMATCoMWYUes0ddyLXp6skf/GF
    s/yb33yVZifMwZyCYwsVPvrMcX7985co+taM81zJ51+6yeMPzmVwqMmDlcIu3HvrLf7w6zcoeGpE
    BofPa0+WgemS08+pUziGson8cxi0MXzxO7dY2WpjDKxud9na61kTbOy0IzV11pp9eqE9NUq+4kjN
    o1Z0aPYSdjsxjV7CTNmuUasfs5OiOVLC8bpPO0i4tR3QDRM2mxFnF4oEUZIJl+dIXCUzs7LgSj50
    rkYYWXOy5EmubvTSjS7ox5qNVkgxPVEbvXhoBomJB3eGVuXXNb/OniOol11W9gI2WiEbjZCpopOt
    TXayGDM0dcaFZ6/9TTYbf0gQr1Hyz5DoHq6aIdFNtO5QT36VWf9X6CUSJSWNIKToOLhS4kpBbDRB
    ojlR8TlVrUwU2INs84OuC8FEG3/SBhJDWczFG8i08+D16qUNuv04m12DQUnBY2dnma76vPjlq8Ta
    ICUoKVnZ7PC11+7Q6UV26vN4fXr6dfoR335zjaPzZeuImoEuNiOayhirwUUutsA4uGDSzZiOTQhB
    Emte/Oo1PvuNGwggSTT9IElRqNG4ymAr3NoKMo1YLShmKy4zZYfbOwHtwNr5A6h7tRESpPCfqwTz
    Ux5b7QgwRAnsdCLixBDr4a0GJ/0gvuAqyYXlUrYmnSDh8novNSdhZS+gcTFGSrvxw0gP5d2MrqPW
    hldvtriy0RuJoZR8xbOnq9k0eY7k6IzHbidir5Ow1gwp+2qCbZiz8fNCH8U7rO78Fp3+FQQOQbhK
    wTuKNgFx0gKjMeYdTvi/yS3x87TCJgBBHFNyHZRQtCOLrZ6fqeEIOdGWv5utfzds/6CA1wCqG1q7
    QzNACkEUab766h1eemuNL71ym0Y7HOnnkTN1zp+sUy66PPXwAt94fRXfUxYNUJKvv75CuxdlDoTW
    Q5PPYLX+5Zu73Fxt4joCbQZO9X7H2wozGGNNrEmAukgNqaFcCLrdeLiRxo5zk/MgjbEnxJ3dwJ4m
    AmbKDgVXUi/ZkyzWgq1WRD9KKHqK93aCgf/PXMUidtNFF8+RRImh0YvZ60b5fYlSAkeJiebrwNzK
    P38/0vRCnfUhBzC2GVVcBoEG1poRa80oQzSNNtRKDk+fqo5M2UzZYWnG4/Jqj9W9kIWqOyGAaCbj
    +N3gGlHSpFZ+FoGDMRG10vvohbfxnKUhhOZuYdjlWlJASTvB9YJPQSnKkaLoOJyqljM8/H60+/0K
    //73ZlRlIjIHtOg7fPONVf7nH7+Tau4h+jFd9fkHP/cMAAszJT7+/pN8/bWVrF8prAObYigYY5if
    KbLbCtKI72AOI7pBbN9rkw8Vjwn1QEjMIUG2/ElnlzHSSSacFgbMIVj51kLQ6g0hSyVF5vRNlxxc
    JYkTw0bLBnmKnmK1EWYO9/JMIQs4lX3FXiemkfYnBxvOGGRqvu1fk+Hc5x8jD1sCJCn4oHMrZje0
    PRVjbVIzZ3Bya6LEjIY4jKHsKY7N+Ly71mO1EXKiXtjvMIsJqE4vvM3tzf9Eo/0dEtPDmAhXzbLZ
    /DyOqtELrwMaUBRNFy/5ZQz/gFbYx5WSnSDEE4JmFHF+uoan1D2hOAdh84fGe8Y0izkoBC9GNY9S
    gg89scyLX75KsxNmeDfAVNnj0dOzmdP47IVFHj5T5/LNXVxH5RzvgR1l8f4PPbHE/33lTqb1yKBH
    kdqX+WHkHVAzFP4DAhsi76Ck/b7wQ8c4tlDFGMNeK+Dli+tsNXqo7OQaCs7KXkiQGCQG31VMl6wT
    6ruSakGx143Zbse0A02sraM7CPEfmXJJUju+XnbYa0cpShMTJkPlEiWGKNJ4aujXxNpgUjNxEFkV
    whAn8MiREh96sEbJV4Dh0mqXL7y1l26moelnDLgOfPhcjbMLxawf63/Y2IXJrb3nSJanfSoFRbMX
    s9oIiLTJAmuDaczh+GBMzHbzS6zt/j6eU0dRJNIBSvoE8QbaBChZJk4a+O4S/eAqkm2W3O9xJX4S
    JTVaG1omQSBYKBVwxmgEh1Ej7uf63TdNTkpyQK4QaST23DyPnp3l1nqL3FxaG/71O/yZJ48CcHq5
    xgvvO8ZbV7ZxHTX2LMPff/YTD/MnL93CUc6E5xUTQch90mzuEl7OBav+8o8+xEeftZHbd27u8Iv/
    /pus7faw/qXIaXy4uR3YPZVi77/zyqZFmzSEiUYKQTdMaHRjNuIkQ1CEgT95azcbWhgblJLEiabR
    jW2ASYBO4wR6DIV58dVNtlq23eKUlznbpCanqwRuOl4lReZjGDFmnBhBpegwnTrfedSvFyYjSsZg
    N+jRGZ/Lq13u7NqTTI5ZmiOR225wnTvbn8GYmDjpIISg4B0lTloYHRMlDVw1g1JVCw/qCGl2qJnf
    Yd6PUEISJvYIUkJQygnBaFR1v2a+2/V9Aj3BTDoI78prWpMeqQA//PRxahV/KHBpUOu//9E72ddL
    BYcPPLbEkbkySWJSxGhosw4U9TOPLPLwqTpJrPe1YcyUyVM0Bn1gzEQy28A1MLnnEGNktEHgR2QR
    bpM9dxAZNpphei/7/L1A0+kndENNkpg0GAdrzYCrG32L1KTj7AYJ3UDTDTVxYg0RKQV73Zh+ZDLz
    Jk50dgIIYaHjTqDpBAmdIEEbMsLcgUpggPOPoXJmgiWw34IYXi95iqPTHgjBXi/OeDp5v0cOWYQh
    u+1v0gtuUfCO4agaBe8ERf8kjprCc+fxnXlr+9Y+gevMMlV6jHLhPFNqj3Pl15ktlpkr+swWPISA
    GLOPgzMJqsw/zGFQpjgkcpvv/8B7ZMiDSAX/GKeXp/adLq9d3uDd23vZ+wtnZ/nAY0eIE51h/SIX
    SbUbxOUnf/hBosSMtAEbAxCTIq1C5qKxB2j8VIBH52ryopNjSYr0WXd7EXvdOBuTryRFT1LyFRVf
    4jsqRXskt7YC1ptR5qx7jqDkKQqupORKip6ym8xYqkOiDb4jwQi6oRXywTh7gd1YUgiUkBRcYXF8
    xMDyS83AHPiQMVVHsOcMKcqfnuk+zp5dIEamb37KY6bkYHQ6v7lTN2NnGmPo9C5zc+NXSHSbWLfS
    ULGH0l3CeIvE9MEYPGeWXniLbvAuie6lbDofKb5IEJ1kLZxFkOBLyX+7vcrfKvicK5cOpTtn2PsE
    R5VDFn1SoGvcuU117/Cz3ORUSh4f+aFjXLy+Q5AemQDtbsxvfeEy//CvPps6uUU++PgyX/neHeuw
    5qDLvG766DPH+S9/8CabuymmnkZPjB4HM/MR1mEvRkxWgyYXmhrf/ONMU7uZTGZODGBMAMeBH3+y
    zvyUZ30dKXh3vcfXLu/RDTWNXpLFEhJt+PD5Go8sl4cQdzfmC2/tsNGM6IaWe+O7Anp2LS6tdqmX
    HZQQXN3q0w0SpBD4nmS+6rHVitJnNPuDbXn0SpghhSc9pdabEZ7Ts5SFPHav0hkcm7v5isvStMde
    t4s0YuR6CggItOlyc+Pf0eu/i6NqgMZRZbTuE8VbSOkjEEjpoFSVvfa38Zwj6SRJhCxjwm9Q54/w
    pUAhaEvBf7qzyn++eYdgjHWZ/3nQ7wefAGaflp8YwBoDdQafmTGz6FMvPMBcrTi8rg1RnPCVV++w
    tt3J2j39yAKPnK6TxCYLpJg8AgEsz5f5kedOEsU65zdNUs1DLNqORwzHOVGcBzcSWfR2kvkvTM58
    SjXgzZ0gG6MrJcfqPoVUe3uOZK7ipMzM1BlOTzIpBCdybQuupJZSibWxzqoxhpmSg+vYE+PNOx2+
    8OYuX7y4yzcvNzM6xlzZ4XjdJ8kgTTHqtI+ZdYPo9NBsgu9ca/Fb39nkd1+x/37vlU1+55VNXrvV
    nkBrsRj/YkqWM/ttTGvjb7e+zNrubyOkAhPhyDJSumjTI4w3kEIhhMR35oniTSAi1jt4zixCuAgM
    YbxKlS8z71xEKY/P7TboRzGdJNlnf98PonMvbScGsFKMPL8JJjFPF2dLfOiJZZSSWcTXGFjb7vK5
    b9/M2j14fIZnLxzB91WGTubtZoCCp/ix508xVfEzwcBYZGP0tiKHbZssaDXpWQ2MXDdmEL00+9is
    Ojf+QdBoqxVl0c56xRk5XUlJXhVf2e+mfRtjqBZVRgAb3MN3JXMVi/JIAZvtiAcWizy4WMRVFrW9
    vN7j4p0O/dj6A0dqLu87WWGq6KD1UPFkPtCYXsg2L6PBRkcNWJnCRonT3x0lcsHB0TjI0rSNTutM
    6Q03nhPFLVa3fo+CexyBg5A+vjtPknSJ4h1cOYsQPqXSWbSWYFZw5SxgKBZOY0xEEK5RdM8hRcgZ
    7x1WwvM0jOH0VJmfOXoEX8r7wu3vB8Of1G4QwPrIU0epVwuZA5Zow/JcmfmZ0sj3fvaTDzFbK6TO
    23DjzNWK5FOtPvH8KQqeotOLckQrC4EO7nvhzCx/92eeZHOvn91XG8NUyWO2Vhgm8SjJ3/ypxzPH
    dIA6PHyqPsLD+fATy9RKHlGiM4FWSnB6uZa1mZsu8lM//AAffHwpo0IYbTgyP8Wp01WkkmhtOFLz
    9/kZZV/x2PEys1UXKYaPWytaLT9yugInZ32ef2AKR9nT9tiMz8nZAsvTfmYCaWPNoHrJ5fSCvRYl
    mvNLRbv5sChPwRv2P1t1ef/ZKaQAT0mWpj2ePV29awrDfMXjSM0jiDWekkOKAob5qsvTp6upj2Of
    azoN2olGZ8f8y8/+a9qBjey50mGmPE0v6tPstdKUQc2x+nE6/QaNXifTNnOVGTy3xO3t2ygpMTpm
    cf4B3p1+gj9px/yjB0/x8yeP4U4Q9IOQncOuX7vT4Bd++Wv89i99ckjKOjCAde/Q571srnvlDR30
    ihLDXjdGa5NGRN1UW9nXXi8mik2mWW1MRTPQGTMlFzfXvtVPUmfS8mIcaR1MUh67NoO4mcFTIoUC
    TeZA9sLE2vQCVEou1Gb0ZNJjz+RIQcVXtIM4s7UHEKQQFu40xkZPHWU58xliGCaspWzOxBibZVV0
    WKp5uI4EDN1A0wqSlItv4c5enspwl5dK+xxGkKHTT+jkIsQzJQfPkTgFb4py6QV+87vfoui6FDyX
    qu8RJR7toEgQR0yXSrx8S1MrLvDeXgOjDUpKjtdrtPsB3WiZVq9PtVgg7tVodht84Ogin1iYy4T+
    fl9/Ghx/8gkwRn29S9bWePvJ4xhDViaYcuOc+Tu7fV6+3qYfaRJt+PPvm2Om7KRUD8233m2y2Qox
    Bs4vFcEILq93s1zUx46VefJ4BSntafb2Soe3Vzs4UnK87lN0JW+vdm1mXMkliDX9KEEbwXRJ8WOP
    zzLYN4k2XN3o8cqNNlLCVMFBSbuZtDYZOiRyebzaGOaqLo8eLfPy9RadMEEOroshBu8qwVzF5amT
    FXxHoo3h9nbAd2+02OnE9GMb61FSUPQk81WX585MMT/lcWWjZ212Kah4knrF5cZmenKOpxSbsXEK
    g+8onj5V4fR8MduIr9xsc2urD8JmZ/3YY3U8BxzPUfzcB5/i6+++w9pek4ovaPXbVAo+taKkGwjK
    Hux1+3RDzXKtyHqzxcnZGuvNPTCGeqmEjg3F6TJXfBfZD/jrx5c4WSoemCV1N17+ZKrxZFTnIF9g
    Mip0r9fNxBTMPL/9sOsml4I30N5b7YhuYANE1nSx1+PEsNOJWG9GGGNYnvYwCDaaUSromtduah4/
    VkamUd5mL2GtEeE7kqmCIvIV640QhKDoKvqRpQ9rA62epBcmVH0FQhDFNg91rRGiJBQdSQCsN8LM
    yU2SAT9omOSPgWDBsNWKaPXtaRMnJpPHQfBsdS9ktxPxiSdm6YeaL7+zx1Y7yjaHEoIosRtztxPT
    6if81NPztIMkHZMgKNp8hJXdAKWGiT8jJD+GhQyEsDyqG1sOp+eLgGCrFXJjs89GM0JjeOqE3YzG
    pFydpdoUf+3PPs8//YPPEUQxUZLQ7gdUCz7z1Qq73R6x1jR6fSqFArPlMt0goBOESCFxgoCluVlu
    Fcus9vr8/OnjPD1XP5RPcxgd+fDr5p7py4fRne/l/Q+qj7yNnOfV5FmnUggkNgoqc1FMkfoDO52E
    9WbE0Rk/4wypLBdi2LdJHcGFopsmfhv6kaHRjammzmqUGLY7MW6aEVWvuPQiCz3qlJG5POcNx5cK
    2UzZngwi9UmkFDx5osJC1WWvn/DOSodO33qwK3sRb7zXQQIbLZuMNF1SPHWywvK0x9urPS6udAhj
    w+peyPXNfpZ6OPg5V/U4v2zvE0Z2wwUp/39+yqFatP7CRjOk09ckmtTPSCilnKNGL0lNSsGjx8rp
    3Kc4vhSCF86f4/Un7/DVS1eoFQtoYzg6M43vuCi5S5gkmQHwyccv8Nrt91iarqU2q2FuYRFRLON0
    ejx/ZJ6ZHP/+MBPjoOT2e8HxD+PkH4b1HzSuw/q433vszx4zQ36PMAfQFgaoxnj2ub1wcaXD0Rk/
    QzrMRODT3nthyuP6Vj+Npho2WxHH6jZJvBsmNLsxCIFUVqBNx66jNtYk+PiFmVxiiu3XUYLVvSAH
    4xpOzPqcXbBw8Mm6z2df2yGIbVbUnZ0glyppOD7rc+FYGd+RlAuKWzt9tpqRZV82wsyZNhgcJXho
    qcSzZ6oIIdhsRnzx4g79lvURLhwtc+FoGTC8s9bji2/uIhE0ejF3dgNOzhZYbYSEKbT84GIxdWxT
    f2WwODOlIh+78BB//MZFdnux1T7aUC0W2G53CGIbh52vVPjW1eusN1t0Apu57pdKvDczz6X2Lhvd
    Ln/nG9/lxRvv8f6FWX723BmOloujuQUT4Udx6PVJGv+wRJSDHOfxNne7fpBA3/c9Mv6MGIFX95ls
    OdqOYJjSqKTk2maPfqSzRG4xRgEatE0MLE65eErQS32NtUaUwVWdIKYb2ShrxVPUig573TjjMgkB
    vqtGnOnxewyo0JkJIgQnZgv4riBM0nFom/kl03VTUuCkoETJU5w/UmK+as2xmbJDN9Q2hpA+lKNE
    mtxiA1UqN4eOFHhKIKXk/GKJl6426QaW6ryyG1JwJGuN0OYWJ4YLx0oUPJWxOGX+gR47tswnH3+U
    fhRT8n0inbDZalMueEhpE05cR3Flc5PZSpnEaJRSyOkZvtfpE+uEiufSiSK+u7nDP375Db67ub2v
    Nsq92uZ3w/EPSz0cL1Mynpt72PVJfRxGk7hbH6PUWfYVcprYJhVSrQ1TBYUEOn3Nze0++byV8cJM
    g/jCTNmllGWAWR9Ca0OSaHY7CUFkEaSpoqJaUMMA3oTNuA/dyjEn8+dNnGibe5AKlu9aesRgXHd2
    +ry3E0AaNX7/mSqfeHyWH39ilsePVzI+0licb1TR5a4P1tJ3BY8ul0m0IdaG9WbEpbUeja6ty3O0
    7jFXcTOhF6lZmb1qxSI//sQFzszNZtUTmr0+jlQ4UrI4VWWj1UIg2Gy1WKxWcAsFtkoVOkFIbAwF
    pZgrFtgOAh6t13h6fhZ1F5z+IEG/n0SUSRsiT4OYpMEPun5Qm/u5xz5EKA3aIExWDGs8mGT2ZWkZ
    EuD0XCErkPT2SmcUUDqAvQ9wZNrLTpu9TkyUGDSC7bZ1mk3Ks68W0qBcqsWjxHB9s8+V9V7279pm
    n81WOFIlASGyKG6i4c07XXpRkqUsnpkvsFizkV4hBdudhD9+Y4cvvdNgqxUR69wmy4MCYjIdw6I4
    wzTSwbxJIXhouZTl9K41Qt5e7WQcn3OLRSq+GtnU+xJRHlle4tPP/hCf+ca36YYBRc9lr9vj5NwM
    /chCWJ5SxFpTLZUw1Rov9UOUknSjmHq5RMV1uBbHhFoTJMlI+Y0fZCLKYfb4QRygu13/Qfcxqs1T
    gtZYPni+PAhilKFstIUR93ox3SBhdS+0AZlBGqmZ7C8k2rBU83hDGaEiuwAACcpJREFU2LzcSBu2
    2iH1sstWK8qCVVNFh6KnMlqEENANNJ99fXsfLeLC0TJPHCuPjPn1m22urvfY6yasNy0q5CrB2fki
    jxwts9OOeHulY1EgBN1A890bLd54r8PZhYJ1jqe8UbPK7GN9p2k/o9dFjsBWLSjOLZZ4606HJMu9
    gNmyy9EZf181O2dcyIqey49ceIjrW9u8+d4KrrIMwoeWFrmz26DkDp3Wan2GaGaW+uqmraoAHCkV
    mPV9ikrRSxJutDocr5RwpfyBJ6L8/wg23a2Pe7nH4WMXGKPHiDbjubij30g0PLRU4r2dgH5kuLbe
    yygZIA/M2lqetiX6BgnhG82Isq/YTdMGy75K0w8HQm8yNMmRo5lcSggcOc4aFdzaCXLZafYbS9MF
    PvDAFFJYZOaTj8/y1UsNdroRYUoRTrR11i/e6fDM6SrvPzuV+QxCmH3HmclIeqMlSAb3dhQ8vFzi
    0lo350PZejz1yjB4l7EzJ2mxI9M1Ts/N8uL3vk+YxCxOVbm9s0etWGC1Ycu6ea5LUqpyY2UDYQyb
    /QDfcXknaeGpDq0woheG/NJrb3F++jmWy6W7autxTbnPzrxHATusoOz9RnQP5n/fPUI8sY0YDaaN
    05EngQBaG87Mlyi4Dbqh5vJ6j8UpN4eYMHJiDL4zVXSoFBT9lk07XGtELNY8OoFGCRuFHVRWGHBF
    wUZ6Hz5aQo1VpzhS80aSTQyGYzM22ymIbdygG1o/5Atv7fHJx+uUfFtv888/NceltS63tvtstyOa
    vSR1dAUvXWvZSK0jJqzz/mChmbAWEpiruhyv+1zfsoWzKgXFibqP74gsYj1YT+cgaO9jFx7iW1ev
    8fL1W3hK0ez1MRhmy2U2W23qi4t8JTYYnbBUKtCOYyqeIko0u0HMfMGjnyRcabTpJ0nOhpvMvT/I
    th+HBPdX5P3T0wgOih/cDcu/1yDYvvRHMyr8Y2mo+5vmyFoFV3JyrsDFlS5b7SjjmJt9iRyjfS1O
    eeykxVfXmyFbLS+bw7KvmM7VwhHGoI2g6Es+8tDMxOe9sxuMHFZPnarwwEKRKDF8/3abb15pohGs
    7Abc2gl4aKkExtbZefJEhceOlVnZC7m40uHKeo8wNniO4O3VLg8uFrO4wf4sNzFC2puUxFJwJMdm
    fK6s9xDS1vmZLjsjkz8Qfpm3T/PCuDxd4yeeeIxziwu0+gHaGLqBhYeW5+rcKFUJo4hYa5phxMlK
    GUcIWlFMFNsiskdKBZ5ZqDPn+xOTRCY5g4dWbp4QwDITEJKDqqsdlKgyvhnvp4/x02nSPfKRRovn
    5xKn8xtkzLY1DPvVxvDIchmROpObzShNCpnMnBj0aXH/tNp0L+baZi+tSWkdWxvJHIVrjLFVhsfZ
    n4wyvTPHUgjLmHx4uWxpwGla2s2tPq9cb/H1dxt8+2rT1uKUguN1n488NM3ytJ+VUQxjY3k6QhxQ
    aWLogwgxSWasbDhqWKZQZubXfl/MOUyb/blHznNrZ5cvvPUOURrAKroOZ8+d4+p2k7M1BwOUHcXJ
    SpkrzRa9xB6jsdE8MztH1XUpu85EDXo3usK9aPzDosM/uIoNd39/mIM+IFDJbH0EvUCPJGBEiU4r
    0Vn8Oo5N3irCGGuz1ysOW+1omCc8OQU4ey1P+yhp/QRj4PZOiEyzq+plZ2Ifg8jsQZDsKKtpaFB7
    jkh5NbbFbjfiykaPTqDxXXvPxZplsnqOpFZSiJ3hMSXvRkcT5vA1E2JsE3NglT7nsKhqwXV5/oEz
    fPXyFTZbLRt4WDzCFxtdJILNfoAUkGiXtV6fXpwQJja/cr7oc6XRZqPX55+/epEXlhY4PVXGlTax
    YD2KebfTZTMM+XNzdR6pVibyeUYebkL20b1EYg+z3ydr5/uP9t7NCa6VHCsUxmqiKxtdTs1bmvJm
    y6YHSiEyBmRTJzBSTcdungcXi2y2YuvTprUtx7Vi/v6VgqJaSANUxlY+II3OzlbcMVvLOrSxtlWR
    B0jIADv3HKuZ96EsqdCJ3J/BEAZ8R1FwNf3IEtO2WhFRonGVpBdptlMTzGDwHEmloCZEJyaYg2b/
    iS0mPMdBNJisktphWvCJ40d5/Ngy765v4EnFVdfje9u7HK+U2QsClBAUHcUbO3tU0kpq7ShGILjZ
    7jBb8PmVi5f5tUtXeaI+ze1OF0dKpmanebXb44X6ND+9tHhXngtwqD64W/BoUpu7Xb+XiPK99EFK
    Q64VnRTWg0trPcJkC0dK1hthRvaqFh0Wpzxa/d5EhOeBhSKv3GiltOAJJVbHJsmyJa3gD+p7amEo
    uWIo+HntKAS9UPMnb+4OsnkyE61Wcjg5O8hvGKdNmH1/uEQKWJr22WnbimlXN3r0Qk2loNjtxKyn
    lZcTDWfmixRcmauDP2HBTZ6aMUH5CZMby8E8MCHE5IJS46+/8sHn+Mblq8SVKl/qRxitudPpcnqq
    wkY3oB8nBGGEBDylWCoXaUURYRyz2tUcKxdZ6/ZJ9C4rzTYoxWIcMVWv8amlBY4U/Hsis02Cre+F
    V3MYKnMYTfl+uDuHU52ttn7mTJWt79u6NVob3l3rp9WU7T/flTyyXGZ5xuPSapfEpAWWcqmT9YrL
    fNXl9k6QkdIGGU2DmjN6LIVxadrnndVeVhteCFtGsOzLEYRGp6mXWsBOJx6rRWqDVUvTGm0MyXje
    JYNkcki0RgtJEBuePl3mxlafdj/GRIZrm/3MBjfGlno5Vvd47myV77/XIdFpyZIJNbYMgkTbsiiT
    wJDByRRrUJkfNVlBOvcSCV2arvG3P/oC//H2Gmfbvew4Olou8li9xqW9FvOp8C6VixSU4k6nR1gs
    2DonvsfzR+a4uNNkanbamkyOy6ePLvEzx5buGQuXQqQFXLl7WF0c7gvcL45/GEfnbmMfvD85W+BT
    PzTHKzda3NwOsvo1roDFKZ+nTlY4NVdACkG94nBm3hZQqpVU5tApKXjqZBUlJFLaOZmf8ig4kjPz
    RYyBhdogacX2f2quwJnUrDJpQsmAujvQJLMVl9NzBQ7yogxY+LPkcLxeoJ4mwdiCUMP5uXC0xEpa
    adlzJI4SfPrZeb5zrcmt7YAgNhlprVZ0eXi5yKNHyxRcyUzJ5cx8ASFsBpij8ieXTSUsuBJt7Mk4
    aV1qJcXZeTuHR2peFnvYB1CYewS3B+HzSX+Ia/gXS0RmEI5O4FiQJv0v+8sY9+icamOyJIb7cUL/
    NBHi+31/P98xBhJj6IdJRrjyszS/YZt8UaiRevditJTGiLOZIipy39zlZj+lEIhJqNMhDnr2dywG
    jQflWib0M+TJDz+PtSGM7Inhpn/BZbyNGYsqiAkc/CHAN2ENRkR0fx+D1/8DhDCvobavS0IAAAAA
    SUVORK5CYII='''.replace('\n', '').replace(' ', '')

_ei_logo = u'''data:image/png;base64,
    iVBORw0KGgoAAAANSUhEUgAAALUAAAA7CAYAAAA96csBAAAAAXNSR0IArs4c6QAAAAZiS0dEAP8A
    /wD/oL2nkwAAAAlwSFlzAAALEwAACxMBAJqcGAAAAAd0SU1FB+AHHRQOHFtumB0AABJxSURBVHja
    7Z17fB3Fdce/V5IlZBtjG9uIIAw2BAyGuDwaSFEgIUDMo7SZoaVhaIBSSoACIQGSJiF8GvIgtAWX
    RygEEpN0iQmZDY8AxoFSiHi1gYTwsIlfGMsgxwZjGVu29cofcxaNlt37kO6VhLTn87mfe+/u7Ozs
    7G/OnNecyTU3NtQBuwPdDE9qa2ppXU9GGRVJNcABwG+GcRtvBs7PXlVGxVIV0DPM29iVvaaMSuXU
    I560MjOAsQOoogdYasOgY4jaXwvMAHIyyFfZMNjuna8WEXIHKbPGhkFbBuqRTTcARwyA63cDhwKv
    D1H7dwMeA6qBTcBxwArv/HjgdhElxwJ/D9yTgXpk007AxAFy6qohbH81MFXeV13Ce8sBk4Fp8n+H
    TPwYXbQdWFciSLuHgWzfM8DzGahHMP0/cCSwY4mAeneIAd2TAToDdRp12jDoBjYOUHnDhkHJ5ZKu
    08oAJNYXnSuRqhLqydkw6Cm1/ZXol6S2RM+f0F99ymagrowlAhsG73W8VuZoUSL3BMYBHcBq4Ekb
    Bo/EX5T8/4gorjlgiw2D+V79c4GTgEbgHhsG8xOA3VMMt9bKNAD/IO0bq5XZBrwC/MSGwStxEMYH
    l1ZmOvAJYH+R16uBd4CXgQdsGKyJXyPP1wicIqLeNuAOGwadWpnPAZ8GJmtlOoHF0pYX/bZoZcYB
    5wGHA+O1Mh3AEuCnNgyeLzRwMlCXSNLpOwPfAC6QF502AN4E/tKGwXMxAB0NXCfF1gPztTJ7AL8C
    PuxVsayfIkanVuYUYEFC+04GvqKVuRU4z4ZBd8JAnQtcDcwpMMBvAT6fALD9vOfbADyvlbkXZ3b0
    6STgUq3MFcB3pS0G+O+E20Vl59kwuCTfrFCVwbR0Tg18G7goH6CFdgV+o5U5swAA9wGeiwF6IO/n
    fODuAu37J+DWhOf7MPBQIUALnQv8TiszJU+ZccCDCYD2LTffAr6ulTknBdA+fUErc2c+bp2Bungg
    +9PsXZ41ZBUwDzgL0AKoR2Pc9QatzCEpL6EWZ4PeOXb8VeCNIjl1XJE8Sr7bgCuBY4ETgJti152t
    lTnJb5cNg6VAs1fmQeAy4FTgs8LBfaV5DvAdcQClPV+D/L4TOBGYC9wSK3cl8J/e/58Kdz4W+PdY
    2b8V8aZi4kc7sEVGXCVoc5nrmyAybTEexhzwpg2D12IiyGNamYXAVTYMnk247matzDHAw1LHeOA0
    rczzCUrPZO/3KuBiGwb3xgZUf6wbi4EjbBhs8I49pJW5G3hAOCgyIH8Zu/ZqGRhfS/CiLgD+RStz
    v4AOAdjXcKbStEE314bBIu/Yw1qZRYD1GGx9VJ8NA+uVfUQrc5vI8tXy0VqZe2wYdFaCU/8QOAQ4
    rAKfw4H/KDOoDwJeAJ4u4vMU8NUUS8VJKYCOZL1H5EVH9FHyO0XuA/b3AZ1iEekpwqzXA3zShsGG
    uJJpw+BxYL53aC+tzC6xtj9gw+ByGwYd8eu9/+d5M8kkYN88z3aNDYNF0bVameg+IXBHrOx3bBhY
    /75S9lXgKq/coTILVIRTb2hqaV1Vyem/ubGBppbWoZI+tiUoiomA08rsYMNgq3f818BWAfNuefr7
    LeACGwZb+mlqiwP9RzYM1uYxJ94oSm5EewNr0xRjubYW6I44ow2DFq3Ma8CHPGbRnFDFFuDeBAtJ
    dP4W4Ayvr228b73fT4l1aYzct/oDa/0oM6DbgBeLUPKQMitTLCB7amVOBP5OZpUxHieLzFWve2JZ
    fR4RLRCQlMt2/FTSoPP+/yFWfoeYSS0HfFKeba6v5Mnz/RH4P2CPmEKYRNvpG6cSB/abPoMkv/+g
    TZjEGFy4QC5TFB29YMOgSeyvhT4fB66PT71iLntCOF5TBOgYszhQlKK6Itp0fYqoMRA9J59ZsjuP
    uXKsyNgPA+ekWC2miTy9W5GzSEeeWcAHZhf5wxG6i9EvRqOduls6dFupF8pLnyFmpwisHTJ1PyaK
    Ur1wsEOACfSGg+ard3kRXLrirnCxYNwolpLonpuB54Hfywy0MzBbzI9ji5zxBpUy50sJZj0B3bUe
    oNcCZ9kweCjlmqNEAZyQp+q3h9FjNoppMgL0bcCXYxYU//l+BRwz3N5VZqcugUsL/bV3+NoI0HFt
    XWgjhdd+dpVZ9BgIzfV+Lwe+GllQUp6vczi+q3KAuuC02NzYUNLx/pYbBG4dV4ZeyGN6A5iOi+Wu
    yMxRAfGk0fu90YbB+ujZEiw99Z7lY8SJH3s3NzZ8KkFZimhpU0vr8jSrRnNjQz3OHp2mUK1qamld
    XG6Zup/cenMMTEfhnAh9goJE9h4DfJfyOKW6RXavxy12GFchzu5bIqZqZabaMFjnPdN7ANfKnC6y
    9YgE9Wflk0ZX4Hz7aTQN5z5NY8c3AheW8Zl30cqcTGlrFnuAx20YtAILvWn6Qq3M4wLsnAC/Rytz
    kJSbVqY2bwZaPdncamWOxwU8lTO1xf30utKnAz8QS09n9HxamSpc7MuXh6uoOFwUxcEMct8fcQaU
    SCfgAn0uEQ5dj3N/L8TZvV8CarQys+UePpcdqJi3HufY2Ef+z8TFhoBzs5dLtn0T5208U97JX+Hs
    wotwDqJdcAss6rz3lhtuoB4timKuDIMuUuiWAJ+JDcQDZbb6Gw/QncDllGHFjMSLXML7nRjtKZw6
    15/+EW/h+UDo1VENHA+cjgsuigB9B24VERmoh4ZW4zyDywbw2eLJzA+L+HILsAZnltso36txscS1
    OCfGCrl+pQfADWJdWEGCxzJJKbRh0GbDYC9c/MNKnDlxjdTZDbRInWtwK84L0TqvDVs85bPdhoEW
    IP9Wyr0jn3XAM8BxNgzOlNlimdzbN/ttEXFpubQ1n0OlA+d5XQ68VmDW2Sb1LZdPouiVa25sOBgX
    y1spuqKppfVbeSwbe+CCh3ZNk6mbWlr7LVP3ZzlQAXDFV4vU4Lxu9SL7vp7vfinLuQq2MSGOYyxu
    1UzR9/GtJklLyuLKrvyeAkyRmWmdDYO309pcaFnaYJktRzyoK2ziG/QXOFjgyHefwQRoBuqMMiLz
    KGY0Amm05NKrKXIAdyetpBgp4s1QiQ1paSEq1ZbREtB0HXBwghbuKzrVIgZdNlwaLUppFc4u3Y0z
    s71hw6A9DTw4h0+UqKdNlDu/vgmi+OVwsc6rKw10uW8DvTHXW20YrMk49cBoDvAXRZTbPAzbPh5n
    OmsTIJ6Mi+VO1JGAfwVOk//zcfZt3/SlcJGGVVLv4QyO82seLr68B3gWZ/POQD0AapNPl7z4iR6n
    3oSzjdZImWGnzAsAJxb5zsbS605PCgWow60phPwhsZUYnOO93x9ombq2CGU130PWlaENF3tTci1u
    uVM1zgV8Ic7JkKM4p8VwpzEpvzNFsQS6C+cyrUnhMksKXN+Ky5dRl3L9yoE0ThSS5d7/HbzpthtY
    GaW9Gkplqoz0S9waQmTw9mSgLp2WNLW0PpSvQNpqcDnejku3VfL1xSop/SkveeSiGeRdGwavy/F9
    cKuvd5JBsR5YUoziIzHIs3GBQbUiw68Fltkw2OwPKq3MjjhPZTyGe7pWZl/PmpMD1toweEsAvBB4
    Us5tikAtac3q6RsNWQfsp5WJp1f4g8xkewpGqnA7KWzP82zVwCxPfl9pw2CrVmYWLkbFn43rtTK7
    x47lovx+ceYizztTRLBuXDjCMhsGK5MYUTlAXXCNWhogiwXqEKVHuBaX8y4HPKaV0cDPcYtxJ3vP
    vR34o1ZmgQ2Dy9Kyi8r1V+EWq44XoHTgAp7Wa2UeBi73LBtH4lKHbUlQuLbTN+Dom3I8B3yd3pQD
    twNfESDcAHyMvrlHpgOPx+rvxOXw2AkX2NQgg2EOLt4ijSbgnHiRCHcMbhHFY4IzX36fjds8qyrG
    YCfF+mwGbknZR+Rc1OcdwAatzPPAOfGV+JnzJT2ibSfpyInCWZ8Ry8HU2ECuxa0YuVQrc28E6Fh6
    2s/JgNhPXm6VJ/NOwi1i/Wdgi5d3r07AtHNCu6aKWW6KXD82ppBNlo/PCSdK+fExhjQl9pkqfVIt
    dUyR2aK6iH6s8+qJGGbU1toYgKfF7jsxBujP4IKtjpbz1TFdYRourn2ZVuZTfi6RbOFtceSbA1eJ
    SWqbiCEf886drJW53obBRd7UOZu+WYjexAXjb8Ithzoct0lRRBGI3xXRpF3EAP/69hhANhTxDG8I
    SCI7dTTLtMTKdVHehQcrZBDv6g2+bbhoxjinjgA9V2YpvHb+QkyQtTifw3He4F+olTlO0sFloC6R
    LsGtDIkSMuYE1L7d+FQBdpSG90veudX0OlJ66M0LdxguIeJaGwZRmrVHBcw7iTId0dlyzudcxewa
    doa09x9FFEGU8IMTlMmtCTNEf2mODYN2rcyDuHBWcLlXDhMdIy6bT8Tl+qvyDAmfENGnU56hRhjN
    I/K7BrhPK7O7DYN3MlAXT/NsGMxLCNH8tSwPu8+bbg+iN7f0AV4dt0ny8aiOTq1Mpw2DJ7Qye0VK
    oZzvArq0MnGr0LZ8ClseBXib1N0RUwrbk8Je+7mDQdJ92z1LE/7vFM/odR6gO4CmWF6UHq1Mhw2D
    x7UyR4qFB+m7LwFXZDJ18XRV3JriyXEPAUs92XKmBwpf0TtXK7NvwiJdJAffWyPApDgQ82sDvWmI
    EcX5fYl+vD57mt40vzngWK3MjmWzfjQ3NtRW6Fm7m1pahzrI6FUbBm8nAc4D9mJ6k6ZPkE7uwe1n
    GL2oDwFLtDJ34RYjL9HKdEdpwEY5oMFlttrVs8IskGC0xJlDjv1MxMJq3FK6CeUA9bnNjQ2ayi3A
    /Al9U7gOBa2Jc+kY9dA3O2qVB/p5WpnjPHkSXALzU0VJWyjy5gM2DLaPVmALc5jimRy349JEF9IX
    xskAqMZ5jceVA9SR+ahSNO2DzH0EoBr4Iu9PFdEoituZwHKtzPE2DFaOYo7tmxvHxhhB0XVkMnWF
    uU+kENkw+La8tCtxyRbXecpTDc7hscKXuYepiJBG5Zipt3q/u2SGXF3CZwPQnlk/Bpdrb8Z5/74p
    Gwb9GS71wMn0mghv0sp8Wqwfw4lqCgy2SWUYMG/RGzHZiTPlvVNCNTngrQzUgyN+vO+YDYOlWpml
    Ngzu1spcRO8mPjNwHsw3hrj5HSLXRjQHtwdj2jOeWAaZugVnl27EOVWOsGFwR6l9PNrFj1wFFdzo
    Re0leedyKedzuLRrvjXJd6zEvXu1SeJACXZlfyevOkmTllSuTThnRF+M38szSY6nr5Mp7b6+vPze
    3i9eSMFrwO+8ct/XyuxWQBSq0cp8TzZpHR2gjnLAeUDaGuvoLSXIjaXeuxb4EfBjYJFWZnaCSbAH
    F/cR0Sa8LSJsGMRjvE+IDYparcx+Jcjh0WIJgBlamf1j+URmaGXGyH1/6113qFbmhzFdIXpGS/o+
    iRH5XtF9tDK7RMkmpZ7I7HlJDPz/K6EGSQxhJs6reDlwv1bmAq1MbsSLH8KJbsBl9u+KccJ64Hat
    zEY59pQNg0vLdOse3FZsH5f/xwAvaWVeFM68WMxRZ+OCdiJ61IZBW2xqfRbnSgeXlHK2cLSZuHzZ
    F0h9xdArMpCjqLmXtTJ34uJMPir9tIuIH9fgkrBH/XWWJKZcgItBOVB0gmLoF2LpAWe2W6yV+Zm8
    kxNxnthxNgyWaWX8jUv3ln77Oc5ru0FEtFNwkYx473IWUDVaZOqD6Rt45M9UB3r/y7lGMYcL67wV
    t7tsRAfiUvwm0ZO4UNE+1hNcOKkfc350bCD0lDDIfy97IPpT0mkxTh4NqKVamTPouwttA/CFhKqD
    WJ3x+z6olXnaew+TcDvmJt33BxIDco13/hTSNwTtAL5hw+Dq0SRTFys3+/1RyrKoKMjGl4uRhQPn
    A3vhIvPy0c24HHVbE/YQfAS3b2Ea+ZF2NSm/fdn1dJxjI4neoe+q+wDnEc1nhbg4QaauShDpVExm
    9mllTLT5N2EALxfot//BhfR+L7rXaOHUn8fF6/YUAObbMcVocmQmKlB/F87+/H35/5p3ry4bBitw
    Yak7ikx8sEy3W0UcuMeGQUsCh/bjHP5LK7NAuOEsqf9l3HZ173o7417jKZ6r/WeOxZycrZWZh0vX
    u5vI8s8AYcIehk8Ak7QySgA+DRf++hzwYxsGmyTJ/Fx6I+lejd9X8nsfJB7Wo6R/1wGLbBg0+wss
    pI0vAQdoZf5czJ6zxCqyTgZHGK048uPXByPt2EBpoAkiS/LOFZNAsRLtyJdcsdD15fZA5tlUtKT7
    l6ueUvvVB3X3MAX1TU0trReRUUZFUg1uJce5FBdoPhS0OHtNGZVCfwLteMXEzr1bnQAAAABJRU5E
    rkJggg==
    '''\
    .replace('\n', '').replace(' ', '')

_valid_html5 = u'''data:image/png;base64,
    iVBORw0KGgoAAAANSUhEUgAAAFAAAAAPCAIAAAD8q9/YAAAA6ElEQVRIieVXUQqEIBAdl74D57cD
    BB6mY3iojuFhgg7Qrx+eYD/eIjJF2rYVrI9B5KXjPGdUUlprqgkNEXnvnw7jJjBzg94y9OfddW4O
    1pz3cxHacSKi19Nh3I3qBXduXvc7N8PESMEUAqUlOjDBbw4WUwSfxXaGhZJl6GHpFgjmDHDy0/MP
    GftigjWwQ2s1heMgbH23fXfbZdMSrGnHCe2+k6sEQxgSm/JrpgSIMpvAkq/Yl/KlpWBUaWxBxrqN
    TCzmn7xnhUhTGjfraIaV1tp7X8k7zMzVP0t/j09JPx3GTWBmVdvf0hvVzH8tNHhSIAAAAABJRU5E
    rkJggg=='''.replace('\n', '').replace(' ', '')

_valid_css3 = u'''data:image/png;base64,
    iVBORw0KGgoAAAANSUhEUgAAAFAAAAAPCAIAAAD8q9/YAAAA7ElEQVRIieWX0Q2EIAyGC2EBuoAD
    MIcrOIejOIcrOIcDuAAjmHvopWlaLycS9YHvwdQ/hPSnrUQXY4SWCACQc347jYdAxECRH5f67fap
    T/NWv89NrEMHAP7tNJ6mOcNBve9Tz+3N8T71pMjOJ7FmFqjH5BSQIkWl2AWlaMOEtA3ClTwCFZSy
    Dh17UIFdw8iDuOb52LDlVz3rv3acd5o3VXOrwFFfFHF2hv24+HHh3masUgp3KQCkeWOT55UidIXJ
    FT9JtDPM5i9X2Fbv7wAfKqW4GGPOuZF7GBGbu5aaM/xt6bfTeAhEdK39LX0ALc+PofdKE1EAAAAA
    SUVORK5CYII='''.replace('\n', '').replace(' ', '')

_favicon = u'''data:image/png;base64,
    iVBORw0KGgoAAAANSUhEUgAAACAAAAAgCAYAAABzenr0AAAAAXNSR0IArs4c6QAAAAZiS0dEAP8A
    /wD/oL2nkwAAAAlwSFlzAAAN1wAADdcBQiibeAAAAAd0SU1FB+ACBBQjLyFlNIEAAAonSURBVFjD
    rZdpdJTlGYavb9Zv9sxkTyYbCQlJCGEJxLAoBKVIASsVrW3Rql3sOfVYa6vV2hbFLqetLVr1SCla
    awtiq0DF3QpEwlLIRiKU7CSTSWYmk8nsM98sX39gyaE/WuX0/f2u1/vc9/M8gizLMlc49nftoHXg
    IGtrtrB6zs1XtIfi00yWZRl/1IsnNAbA0f7XSaQkusaOkpbTeMMT+KNePs2bhE9DYCrsYm/bdoa9
    57hr2VYmQ0763V3U21eg1xh58cRPydBl882rf4ZBa/5Ee6o+yaQux1E0Ki0GjYVx/zDxZBR3YJS6
    wqUoBAWzc+rpGG0hFPcTivuRkTk3cZpoIsS8wmWoFOorv0CP8wR7Tj+BQWPhhvqvcftVD+MNT1Bv
    X8FTh+7DE3Jy3tXOLYvuRasSMYs2psIudp/6FbKcJhIPsrxiw6eLAU9ojHfO/hl30IFCUBBLRAjF
    p1EKSpJpiWDMh1YlEooHkJIxwvEAaqVIWAqQltOolRqCMR+huJ+UnGQiMMLbZ/+EP+r93zEgpeL8
    pe0pTl14n2xjIQ+t3cmApxtBUKBV6Xjh+Damwi5unH83C4tWcd7VxtyCJtpGDvGX9qcAeGzDHqYj
    k/hjXoqss3np5M8Z8HSzpHQNmxfeg0qhRhCEywn0ubvY1/kcE/5hzLpMlAoVmYY8pGSckalexv1D
    yLKMWcxEEBSYRCuu4AiO6QG84XEMGhNalQ6zaAPAMd3P2PQA8UQEi5iJgIBVn8101MP+rh20jRy6
    PAZ2Hv0hUiqOO+jgjqZHKLTMYm7BVZwZa+X17l0oBAW3NT7ErYu/w2RwjNqCq7j/1fWk0gn6PWf4
    7rVPo1GJ5JiKmI54eKXtyUtUb1p4D3WFS1lQdA0vHN9Gl+MoAIuKV81cYFFxM4OTPczOqaff0023
    8xhSKkZhRjn2jAoMWjN6jYke5wnGpgfIMRfRUNLM8ORZ6gqacAVH6XR8iFm0saT0Osqz60ikJAos
    ZZyb+AfdY8cwiTaqchfhCoxiM+RdHgPJdIJBTw/FtkpePr2dTkcLJtHKtg0v4/QPISCQTCd4ruVh
    wlKA9XV3sKryJnrdHVTnLeZw72vs79oBwLYNLyOl4gSiXrKMhew4+ggOXx8Liq7htsaH6HN3km8p
    wyRmoNy6detWgL2nn+RI3z4AqnIX4A46WFm5CY1KZGfrjxiY7KYiex4alUhaTrO2dgv7u3bwYf/f
    iEgB5tuvxhUYYU5eA6WZ1exs/TE9zhPkmovIMdnxRTysqb6VXncnB878jl53B0tK18x8QfvoIRIp
    iQHPGZqrNlOSWY1Ra6F14CBTYRfheABPcIz1dXciJaOIagNtI4dIyynOTpxm3dyvcHvTw+jVJnrd
    HYz7h1AICoYmP2LDvLuYb1+BXmPicO9r+CJufBE3wAyBDF02ofg0zXM20+loYceHj+AOOriu+lYm
    AhcoslZSb1/O88e28deOpynPrqM8u45gzMfa2i/jDU/wi3fv5tTwe6yrvY1Q3I9JtLKq6iYO9jzP
    H0/8DJ3GxOLSa5nwD9NQsprK3AUzBGryl1Biq0Kr1tMx2gLAqK8PpULFDfVfR6VQMx3x4I9OXpSZ
    r5+Gkmspsc3Bosuk/WNZTUVcxJIR1tR8kbScAmAyeDF5jU3301DczC0N9yGqdJcH4fYP7mPYe5ar
    Z3+Oq0o/Q9voYWrzl5BKJ3nmyIPoNSa2NH4fKRllbHqQ1XNu5tkjD3Jh6jwNJau5sf5ujg2+gVWf
    g91awa/fvwcpFefWhu+QY7LT4zzB0vLPcnLoHd49txuLLpNH1++eMaKoFAQgloigUekwaszo1EYS
    qQQAyZREWk6h05gwai2oFRrCH6+JSiFUSg0GrQWD1kJaTiOl4gAk0hKi2oBBa0ZU6Ykno5fWXEbA
    F3HjDbuw6rP5+z/3cmzwTQoss/j26t8wKo2hQkkqEuHlU0/gDjrYNP+b1BUuZTI0Tr6lhO6x4+xt
    244gKPjRuheJSiGklIReY+SvHc/Q62pn6azPcn3tFtxBB0athVxz8QyBcf8FjvYfIBCbQqcxAVzU
    aUrAev/vyXhsNwZZRKXSAmAz5OH0D9HSv5/J0Pil/K9ViYhqAx2OFo4PvYmAgF5tBMCiyyKWiNDS
    v5/zrvbLCTyw7wakZIya/CXctfTHuIKjFFjKCP1+N76vfQ/BZMD2zE9JbVlHPOQny5jPd19dTzKd
    oMhayf3X/hZ30IFZtDHq6+OZIw+gEBQ0V93Murm34wmNkWsqYlfro3Q7jwGwffM7MwTm5C7CJFqx
    Z5Qz5D3LG90v0DbWgnpBLcr8HDRVs0iW5XNy4C0Odu8iEJuiKnchZtFGWWY1voib987t4UjfPqz6
    HLKMBdgMeRRklNE91sqBrp2M+vqYlVWLWbRht87+DytOSfzT1U5Fdt0lKzZqM3h8416mBj5CkRbw
    F4jseP8BIlKQ9XV3snL2Jj4aP0m9ffl/WPFeookQvoibwoxynmt5GMd0P/PtV/OVph/Q7TxOsbUS
    iy5zxogO9jxP+8hhEKDYVkUkEWJJ6XWIKj2v9D9Pf2IQu6EEvcaIRimyvGIjh3pfpdPRQiwRYXZO
    PYHYFBXZ8yi2VXLgzE4GPGfI0GeRZcwnkYqzvHwDI77zHOndx4ivl3r78hkjOtJ3gFQ6gVal46vL
    H0Wt1DAnt4G2kQ/on+hAISiYn7+MZeXr8YZdFFjKONL7GjIyiVSMlZWbWFn5ebKNBUyFJ+hxHgeg
    NLOa1XNuwabPoyZ/MbtaH2XIe5Yh71m2ND44QyDfUgLAyspNnB75gH2dz+H0D9JctRkpFacydwFl
    mTW81vEs757bQ5G1gnr7cgQE1tR8Cad/iJdO/pz20cM0V21GoxLJNRXRWLqGtz96ide7d6FSamgq
    W0skEaJp1vWUZlbPEJhXuIx5hcuIJ2McH3yLRCrO6FQfWpWOZeXrL5ZrySjuoINUOsG4f5gVFRsR
    1QbKMmtoHXidsBQgLAVIpRM0lq4hFJ9GpzEyERghLadwTg+yomIjtzU+9N/7glFfH8cG36SxdA3J
    dIKnD38PUa3njqZHCMUDXPCe44b6b/CLd7+BKzjK3IImtjR+n/fO7SbXXEyprZqfvH0nADcvupd8
    cwn/GH6PVVU3kWOy/++yvMg6m1sW3QtAp+NDlAo1GqVIVApTYClDrzEBMik5hUK4qGRBEKjIrseo
    tRCWAujUBtKyTDwZpSyrlrKs2ivrjGRZ5sTQ22jVOrIMBfzh+ONMRVx8oeE+SjNrGPB0U5O/mF53
    J3tOPQHA4xv30uvqJJoI01DSjPbjzHdFjYkgCDTNuv5iWg05MWgtTEVcmLQZuAIjOKb7yTYVYtJm
    8O+6AmBh8cr/f2+YltP4o5PEk1HyzCU8uO9G4skIZVm1fOuaXzIZdqJWaMjQZ1/6mv9rd6wQFFj1
    OeSZL0q2Nn/JRa3bqlEqlOSairAZcj/x4QD/ApX6khedphFNAAAAAElFTkSuQmCC'''\
    .replace('\n', '').replace(' ', '')

_omnipath_logo = u'''
    data:image/png;base64,
    iVBORw0KGgoAAAANSUhEUgAAAXcAAAD1CAYAAACrz7WZAAAABHNCSVQICAgIfAhkiAAAAAlwSFlz
    AAAuIwAALiMBeKU/dgAAABl0RVh0U29mdHdhcmUAd3d3Lmlua3NjYXBlLm9yZ5vuPBoAACAASURB
    VHic7J13eFRV+sc/585MKqkQAoTeu4LYsLe1AGLDtZdVsawNK5AAF0JA3XX1t1ix7eq6umBFwIa9
    IaKCUqT3lpDeMzP3/P4ICSlTzp1MMpNwP8/D8yT3nnvuS5J577nved/vK7Bo20zVhxLBVnS9LNSm
    WFgcUUyenIQ96ixgFFIOAdEPSASZBEIAJUhZimAriC3ASiRfM0dfC8im3l40dQKLMGbq1M7gWAGs
    IEJMQNeNUJtkYdGm0fV4XMaVoF2L5ASQNtNzSHYjWIAUrzFHXxWoKZZzb6voegxO+RUwCgAh/sZs
    /aGg32fKlBTmzs0J+rwWFq2JqVNTIeJhBBNBxgZvYvEZ0niUObM+NX1l8IywCIgp+kA0+Tcc4np0
    PS9IswrS9ddBXln/qLyD2bOeDdI9IF2fCDILhxiErh8M2rwWFq0FXY/AyUMgpwAxzXinpQj3vcye
    vUn1Aq0ZjbFQQZN/A8bikr8ybdqxQZkzQ5/RyLEDSO3/mDr9nCbPf9ddkaTPeAXk80AHqpje5Dkt
    LFobU/WjcfILyEya17EDXIC0/UbG9LtQXJRbK/dQMmXaKWja13WOVCDFX5mjvxzwnOnTLwUWHtqw
    8UQRmnEymZm/BzZ/ehrY3wGOq3PUhSGOYq6+LqA5LSxaG1P1GxA8AzK6xe8tWUSEuBpdL/E1zFq5
    hw6Bpj3S4FgUQr5E+ozn0fUI0zNO1UeCeNWHYweIx9AWVccITTJl2klgX0l9xw5gR8i5puezsGiN
    TJ3+MEK+EhLHDiC4EKf8rnqh5R3LuYeK9OmXAaO9nJ2Ii8+rs10UmTq1M8j3UXs97IlwLEbX1V8l
    0/WJaNrnQCeP5wUXkqGfrTyfhUVrJH3GowjRcFEWCoaDfRlTpqR4G2CFZULBxIkOUrqsBdnPz8hs
    pLyCObO+8Dlq0qRoYuK/pPGK2jeSt4gQf/aZIqnrdpzyceBu/xOK1TgYaaVcWrRJ0vX7QD4eajMa
    8AvO8tN57LHihieslXso6NhpooJjB+iIEB+RMf2vPsYIYhL+jVnHXn3lZThlptfzU6em4pRfoOTY
    AeRRVHGdaTs8MWVKStA2mC0smkr69HEg/xZqMzwwEnv0y3hYqJtPsD9SSddf5ZTT4+iUsoZ16wKv
    HnvooTg0x9uAai6sDcQFnHZ6b44Z+THLl7vq2zVjJnBHwPbAKZxyxg6++bJ+sUS6Pgq0zxEMNTWb
    EMdy/Kjn+f57Z0DW6Ho8J502FWF/A6kN4psvXw1oHguLYKHrXTD4CPXPbMsiGMxpp+Xz9Vc/1j1s
    rdxVmDo1FeTVCPlv+g/ZRLo+kQkTAnswRsQ8CHQ0fZ3kOmLiv2ey3rv2WIZ+GchpAdlRF8F8pk4/
    q/b79OlXgvwKQVfzk8k0ImIeMH2ZrkeQrk/EyUZgOsg44HRfMUULixbBKV8AOoTaDJ9IMYeMjF51
    D1kxdxUypt+KFM/VPyjWI3mETWtfZ+FCt9I8ut4RJ5sPOa5AyUPKq0DLQchvCF5+bR6GOAUb1yHl
    w02bSpSDcwBZWbv8Dq3ef7iRaofeePdfyJuZPeulptljYREgGfpYpPwg1GYospSsmWNqvrFW7ipI
    caGHg4OqV/KDfyFDvwiVB2UVs5ro2AGSEdoSBMsIbuFEMpr8temOHapTxOy6zyG6rjFVv4aULuur
    i6E8OHYAqV3adHssLAJA1zUkfw+1GSa4gHT95JpvLOfuD11vB5zpY8RwpHyXdH016dMn4M3JZ2QM
    QMibgmOUtFUrywUd87n13rmBadOO8XgmQz8bJysR8jWQffzMczaTJzfH/9XCwjcuLgI5INRmmESv
    +cJy7v6o4gIgyv9AOQzEAqbqP1TvrDc8bX8UsAfdvvBFw7DVzy5I108mfcZXSPkpyBFq00gHtoix
    zWCfhYU/7gu1AeaRZzFFHwhHlrMJDE2ON6WsLOTxIBaRof+ANOaSNesD0vUTwLjwyNvikGdUxyzJ
    rt74lQE6ae1S4LWgmmZh4YuMjD5IY3Sr/Mxq8kbg4VZoeQsycaKDlM7ZQGLAcwjxFZJ49ZVqW0MU
    g9HOjySCPypwiBR/WhoWFkEjXZ8KMivUZgTIdrJm9rLCMr7o2OUMmuLYAaQ87ch17FC9gdwkxw4Q
    hdMY43+YhUWQEJwXahOaQE+m6P0t5+4LKceH2gSLGqysGYsW4q67IpGydVdHa5xpxdy9I5Bc2BpD
    bm2UC5g0KZonnigPeIapU1PRtGRmz14fRLsswhC5lPhSSW+biyQJSVLDsLkpMaDYZWNL/Di8N5dJ
    TByOoZJEEcZIhrdN5z51+hg0rRw736LrVQHNMW3aKIxAKjQtmgcZS0zcecC7Ji4SpOvHII0xoI1B
    yGOQvAjc2kxGWoSIinfpg41zDfiTgBEVLrrbAET1lqiQILXqrx0SyhZRICQ/ovGtAUtjx/FL7WRu
    Wx9Ek/tThxYhh7RN5y60W5FyHE7KSJ/xPVIuQzMWmVqxGdpFzWihRUBol+LPuet6DE7jLBBjQYwB
    mVYd8q/9sHZvZiMtWgi5gOiKKK5C8Bcpq+WzVV+0BSQiOBfJuRpklr/PRgSvOl08F/8zvf3PEO6I
    tLYZdEjXf/GyibkVWAZyGc6KjzzJZB6eY8ZaYHBzmWgRCKIYBx3R9Yp6hyfrvbFxNshxIM4BGelj
    jvVk6dbvtRUjvyCqsohbpcbDSNR7HqhRujS///qbtl48qsDVmiMzIr+NOvcZB/AvzlUBfIuUy7DJ
    ZWRm/kLN8m6y3hebVG5Ea9GCSDmWTes/ov+QExGMRXI2SM+VsJ4pI2tmeKr7WfilfBGnIZmPoH9z
    3ifXFcO0XWfxr5wRSNkq3aSrVVrtE12PwCnLMV19K3aA/BjkRyCOBqvpc5iyHUgG4gOeoTpn3vuG
    mkXYIb/AXlnEo1IwiRasLFpcMIDbtlxIrqu5+18HG1Ha9lIhK+hCQLIKsgcwEcQ7WI49nOlJUxw7
    QJUVd29NFC8lpaKIz6XgPlq4ZHRs4gZ+GDqfQdE5LXnbICDL255zt1kZLhZ+0Czn3loofZ8udhdf
    ITglVDZ0iyzk88EvMzpuZ6hMCITstufcMXx2BLewAKNHqC2w8E/xO3TUBF8Bg0JtS6K9gncH/JeR
    sXtDbYoiYksbdO6a5dwtfCOFtXIPc+THxNrtLAb6htqWGuJtlSwa+Dq9ovJDbYp/BBvboHP30vTB
    wqIGaYVlwp2KSp4Dwk4CoL29jP/1+x8xWmAtglsMaXzX9px7QH0/LY4shBWWCWPKP+Bq4JpQ2+GN
    YTEHyOy+LNRm+EBKDOe3bc+5I6yVu4VvhLVyD1cKPyIZyf+F2g5/3Jb6E8fH7Q61GZ4R2vfMnZvT
    9py7tMIyFv4wOjJpUnSorbBojKOKTKB9qO3wh4bkiR5LEWGpQWO8AW2vzZ4AEexyZIs2hxBEJXQL
    tRUW9Sl/l54CJobaDlVGxO5jbOLGUJvRAFGK2/kmtLU2e7qegtOXroiFBSTby8lzRvUAzH0ydT0K
    pzMFHJ0RdGS2vrh5LDwykTYmiVbmkyanfc0H+WHUQ1vyInPn5kK4/SDT9ZORlBDBQQoLc01rd7vd
    aW3vZcQimBwdu4+ney3mpLU3V8fdJ02KJjIhFeHujM2WAkYqBp0QoiOSTmgiFSlTQHTGKROqPzIS
    pCikqV26LGqRHxNbUclfQm2HWUbG7uWomP2sLusUalMAUY6bf9R8F17OHfkCgoE4gZh4SNdLETIX
    yUEQOcBBkLlIeRChHTpu5GCTB7HZcnEabUCq0yJQekYWYCDYWZnQ6JxdGPwl5Rfm9viEaOEkUVT+
    rSB9xpNAu2q9OA2kpFYAnOovq48Bjbqky9zm+n8ciZRXME4I2oXajkC4ruOv3L/9/FCbAci/8ejM
    2jLa8HLuQhxAyoGHD8hYJLFA93ofrhp9bgEgwBBgHPpgWhyxXNZ+DZndPmNTeXu+LOrF1sokAPpG
    5XF+4ka6RBxWeO4eXZBUUNqE1ZYUrU1sJKzRBBPCcWtShTGJG7mfkDv3zTjEo3UPhJdzRx4ItQUW
    rZeOjlIA+kXn0i/a98I6yVbh87xfhI82bRamkBJR8QGnhdqOQOkRWUC/6Fw2lYcqyUc40dxXoWeW
    1T0aXgFqQ1jO3SJgUh0lymMjNHdTb2c59yBR9QGDaQXpj74Y3S6EomLCuJ/MzJ8aHg6zlbtxwAqt
    WARK36g85bEHnE3s12Gt3Kt5+OEE7NHDEMYQpOiJFEloMglwISkBChByK2ibqCr7yVP3MzcMb+2f
    +hBKAj/J7FnzPJ0IL+eukd1o38rCQgEhJH2j1PY4q6SNbRXJTbuhYRy5G6pT9P4I40qEOA/EsSBt
    1HQrEtLD3vOhPTJHtIv0GasQYglu3mSu/geAoPX3LB0YEucuXsDB/d7OhpdzRzvQ+C/DwsI/g6IO
    Em+rVBr7XVF3it0RTbuhdsSt3AXp0y9CaJOQxsmHshow+Xm1A6OQchQaM8jQfwD+7pKzutuFEXyL
    W5AUR5n/QcHlMbL0yfj4BYSXcxfuA8jw2gawaB2cFL9Deeyje08Nwh21I2flPnX6OQjtbyCPqk0X
    DQZSngi8PWj1PYXP936fM+O3BmfeEJBkM1eSEziiFIxbyZr1ur+R4eXcnbYD2KyVu4V5zk1Q62f+
    5L7RfFXUs+k3lEfAyn3KlPbYIp5Eck1zvlHvroxPGLP+Wi5JXse8XotJtreUowweQdig948Q34Hr
    FmbPXq8yPLyWyZWFVraMhWlibU7OTPC96it1O5i66xym7jo7ODc12rhzT9ePwxbxS7VjbxneyRvM
    cb/fxrfFrVGRWTRjXEbsQciJ2DlV1bED2JrPoABYvtzFqWfcD1j6MBbKCGBHVSKRwkWirYIYW3Uj
    hQJ3FKvKOvPs/uO4Y9uFfFrYl6CFFCLELL78sjQ4k4UZU/UrELxPCNITi92R/O/gMHpF5jM0Jrul
    bx8wCVrl2tl7zpiApAOCfgTnD20jiFk4uJ5ZM1fw5ZemNzjCDHkAiAu1FRathypp4/Wco3g95yig
    OnMmXqui0N1cawQpQajnXbYmpuo3I3gOZMgWflXSxl+2Xky+O5rbU1eEygxTGIJdZOnfAt+i611w
    cjmCi5DyeCBKfSaxAeTHIBYemi9gws+5V0sQhE3fRIvWh5SiGR07gFaArrua8QahYap+OUI+TxiE
    a6UU3L/jPKKEixs7/hJqc/yiwZbab3R9L/Ak8CR33RVJfPtjkAxEo/chEbqE6ksoQ5KHYC/S+B3h
    XkNW1p5g2RR+zt2QB6w6Joswp+3F26dMOwXBq4SBY69BSsHd28fQJyqPU+O3h9ocn0jJ7x5PzJtX
    CXx/6F+LEja/yMNYEgQWYU5bq06dMqU9mu0NCL9eCC6pce3my9jnDO9IrdD4JtQ2NCT8Vu6WBIFF
    uCPlANL1BSD3gNgFxj7QdiFceygs3HtotdZ6EBHzIXzbU2Y7Y7lr21je6v9GqE3xjGR31IWo5eK2
    IKHzohMm2Ojfvws4eoDRA0l3hOgOnAwMDZldFm0ahzC4vMPv9IgoYM6e5hIiFAeAvdXOn91IuRfB
    TiR7kdoeItmNrhc1083NMXX6+QixNNRmqPBGvwVclKycCdhiCHgy6kImhdqOhjSfc7/rrkhi26dh
    pzcGXcDojBC9qdaR6A10JyzfHCzaIhHCzYQOa5jc5Wv6RuXxSvZI7tg2LoQWubqTlbUrhAaArttx
    sgZkGPWJ806vqHxWD38KR5hJFUjB8THjCLu0nuA713T9TeBMkCnV7cg43FTDwqKFaadVcUPKr0zq
    8l29Zh15rugQWiV+D7ljB3AafwbRKhw7wLaKJF4/eBQ3pPwaalMOI/gxHB07NMvKWe4BUoI/r4WF
    Osn2cu5I/ZE7Oq0gyUM5e747hM5d8FHobl4X7f7WJtT35L7RYeXcJTzqd1C6fh/IxxWm+4ismUFr
    6RR85y7EEqS8L+jzWlgokGIv5dbUn7iz83ISfKhEhnTlbhgfh+7mh0jXR4EcEWozzLKhvAM/FHfj
    xLjQv/gAP0X/zPuhNsIbwXfu2Xu/IaVLIcjGXYotLJqRftG5/DT0WSIVRJzyQ+bcRSnFeU2qPAyO
    GVzRyhbttfz34FHh4NwlgvuETnhtANQh+Hnu8+c7gWVBn9eiTRNvq2R4zAG6RxYGPEefyDwlxw6Q
    64oJ+D5NQ34RFqmSkotCbUKgfFrYJ9QmIGFe9DhC/5D2QfNkqwhjCVJc2ixzW7QpTonfwe2pKxiT
    tIEIUe2YN5R3IGPn2SwuMLfX1zVCPbsw3xmylXvoQzK63gWnDL2HDJAdlYlsKm/vtwl6M7ImuoLJ
    obq5Ks1ToWrXlkD4vq5YhJ5Ym5N/9lzCxwP/xcXJ62odO8CA6IMs6P+madGorhHqq/7cUIVl3GGw
    mepidKhNaCqryjqH6tbZQmO8uJywF51vHueu69lA+Kv9WISEjo5Svhj8Erekrqxt1tYQIeCxHh9z
    VMx+5XlNrdxDki0jtvGIvjkEN66PZGCoTWgq68pDkpBXKg0ujBpLq2gZ1XxFRNVZM6OabX4vRGsu
    xievZ0ziBvpH5dIxooRsZyyrSzvz34PD+aq4J1JaOffBxiYkFyWvY21ZKn+Ud/A6rktEMUsHvsqA
    aP/yLHZhMDnta67cdLmSDZ0ji/0PAsoNOxVGCOrnJOFSCdrqG1JvaWqDc5NIKMBgTMxF/NiiN24C
    zfcXLlkMzGi2+T1wXcoq9G6f09lR/0PeyVHC8JgDXJuyio8L+nL71gvDXoiotRCjObk+5Vfu6rSc
    XlH5HPvb7V7HttOqeG/A60qOvYbzEjcRIdxUKciLt7erNcPJC9VmqgiDFEgAZGqoLWgqLZzttN0m
    GR95Eb+15E2bSvOpQmbpPyPZ22zz1yHW5uTN/gt4vvf7jRx7Q85N3MzXQ19sUlaGBXRwlJGR9iWb
    RjzBP3p+SK+ofAD2enloCiF5rvcihsWYE/2M0lzKvyvVJsUhirdX4dC+CMWNPWCieUR40oLO/cOq
    CI6JHN+6HDs0r7aLRIiPQd7YlEmGRmdzQdIGVpd15uOCxj08EmwVLBr4Ose1293onEtq/FLaBaeh
    MbLdXqK16v4KXSOKWDrwVUavmUhRszZ1aHv0icrjnk4/cE3KqtqfZw1V0ka+27PfeKDzd1zafm1A
    9+wVlc9mhddwT5WonghRjvu36HpJKG7sgYhQG9BUtOZP0s9F8GDUWP4VLVpnRUAzBx6NJSBMO/f+
    UbnckrqScUl/0COyAAPByWtuaTQuSnPx1oA3PTr29/IGcd/282vDL6mOEp7p/QEXJG4Eqp3U7G7L
    uHv7GLPmHZEcHbuPh7t8w4XJf3j9YGU7Yz3uZ4yI3ce0roEvWuM0/2nhDmHQTqtSmq8gNJupYRKS
    ARDFrU12oCE5rpgtQCzQKchTlwl4pkrwaPy41q3b37zNOhzap4DaJw44KmY/7w98nVXDn+LOTsvp
    EVkAwL+zR/BraePUpyd6LuXkuB2Njs/PHsVVmyfUi6sfcLbjzxv/zCeFh1f/N3dcydGx+8z8j45Y
    /tHjQy5KXu9zxbS/qnFIJkZz8kqfd5qk5GdXuDbJXu4186YhBa4QRCVkGKRA1iBkfqhNaCrbK5PX
    RMXRS8CtEBThrs1AhlvSO+pCHmztjh2a27lXa1b7reKK0Zw80+sDvh82nz8lbK73IS10RzJj15mN
    rrm0/VqPAkIrS9K4f/v5HleQLqlx05aLaz/cQsCkzi3e/arVIYRksEIn+v2udo2OTUn7ut4GqoGg
    3GSmik3BuScrhmSAUITi9jNH99yGLRS00F5Y8yK3iTOoiLqQ+dEXcrzQ6CNhkpC8C/jd2BGQD3yJ
    YJoBx0WNo3/0hWS1G+//2tZCC+SDiSUgG3vnQ7TTqlg6+FWOjfXcFzZr9+nkuGLrHUuwVfCPHh96
    HP/wzj/hkt6fWQedMfx938nM7latkHBJ8jomO/7UprJnojQXFyb9wcXJ6+gdmU+uK5qlBf15KXuU
    accK0D2i0KcIVw0Hquo79yEx2dxT5+G5ozKRazZfxvryjqwc9gw9D72Z+cPt4/dZQzsF+2oobnnn
    nkz6jKUI+R6GaxFz5oT4dVGuaf0S3GJL3e8O5Z4/eegfcgEJZdH0sblJNmwkYiA0QYlhkCcFu2LH
    t4UHnG+az7nreiJVXAiM9TZEQ/J6/4VeHfuG8g48d+C4Rsczun5FR0dpo+NL8vvzfXF3v6a9nD2S
    9LQvidZc2IXB2OQNvHCgxVPym4XT47fxdO8P6B1Z/837jIRtTOy4kvEbr2ZbRZKpOYcqrNqhfrhD
    CMm8notrwzEHnTGM/ePa2o3RH4u7KTt3FWdct8LVHyFYuUcA5yHFeQjHM2ToPyJ5D4P3mKtvbGlj
    kNrviNYdc8dPKEZcTiFHeCFlcJ27ridTxViEnIBT/gnhe1d+atpX/CnBe8He5J1/wtlg1dbZUczN
    HVd6HP9C9rFKZua7ovmooD8XJ68DYExi23Dut6X+xOM9P/QaF+8Xncviga9x4u+3mnJwA6JylMbV
    nfPy9mtqlfukhJu3Xlwv48WMaylUsNVMTL/K8J8z34xoSHkicCIaj5I+YyuwGMRCsvTvaImdzk1r
    V9N/cB7QspVAQUOUkrM3fETdw5SmO/cpU9ojIsdUO3TORUiHymXHxu5hctrXXs+vLEnjo4J+jY7f
    1uknohqk4EG1PvfnheqFdx8W9Kt17qfE78AmJO5WXLl6T+cfeKT7J37H9Y7MJ6Prlzy041zluXtH
    qe2/lRxywtGai8yun9Uefz77WI9prKoUKWyAOkys3I3wCkn0Bu4GeTfpeg7wERgLcWgfo+vKyQim
    WLjQTcaMZUjUSn/Dj88Pqc9a+CAw5z5lSgoi8nyEnACcB/LQPGqLjkjNzXN93sfm49Xw0T2nNDpm
    E5KrO6z2OH5x/oBGq3xffFPUs/brGM1Jn8g8Nla0V74+nLg4eR1zun+qPP6mjj+TufsMit1q6c59
    ovKUxhUb1fPd1ekHuh0qPNrvbMeMXWc1GpvsUN8AVSk6kiYctqpzH9VuDxvLO7RgGEemANeCuBan
    LCBd/xTJYiJ4L+gNtaV8C0Qrde7GG97OyAXYAJVXM7e4HPUVQStE3bk/rHfHYVwM2oTq10oTnrQB
    d3f6gcHR3l/115WnsLSgf6PjZ8VvIc2LONR3xT1M2bC9MpFcV0xtyfqw2P0hc+5DYrK5IeVXhsYc
    oNgdwccF/Xg1Z4TSw6p/VC4v9nnPVFFHjObknITNvJM3WGl870g1524g6Ogo5YEuhxOkHt5xrkfn
    qKrg6JQaB5yNs3AaUiHV/5RVflZCSB7r/jGP7j21SW8dTSAR5AQEE3DKa4DXgzq7Q3sfpzwIeBcC
    Ck+KcGgfeDtZEcVMIF1hnjeAq4JmVRii5qDTZyzBbmxHiieR8iTl6zyQ6ijhwS7f+BzzyJ5TPa6u
    rktZ5fWan0u7mLZlTVnH2q+HRbd8BlSCrZKX+7zLymHPcmen5Zwev41xSRt4qtdivh7yIin2xpvG
    dXEIg1f6vkOMZv4NdZSXTeyG2IT0+kBtiJAwvesXxNmqowkrS9JYmDfE41jVOfdVxSmttCvc6s69
    nc1/tOPS5HWcGLeLkzzUUbQsYj0b178Z9Gl1vQohXgj6vM2NEM+HUaVvWKPmpIVchnKJiG/qfvg9
    sacqnnc8OASHMPhT4iaP11QYdtaXmZcArassN8jHm0RzMDD6ICuGPcuVHTxLVhwdu4+F/d/0GbpK
    T/uSkbGBZXQNiVXLgOlgL/VpQ10GxeRwfZ3ag2m7zvJYbxBnq1JKrQTYXaXWrdHMyj3ez72jNReZ
    h1JlT4kPtXM3ZrJwYfOED+z8o7patbUgyrFXpzpa+EfNudu1Z0E0uWlhz8gCrvWx+gb478HhHjc2
    R7Xb4/WhoLq6a8j2ysMpgR0jfK+SVekVle93VToidh+fDnrFryDW8XG7uSHFczbXiXG76oU/zNI3
    Sq2LjaeUU2/8NXV5bTXpRwX9+LKol8dx3Uw01dhTFa80LteprvTYxc/vZ3rXL2rTNEfG7g3ozSg4
    iLU4tIXNNr2uH0Twj2abP9gI/o6ut/n89GCh5tx1vQKY3dSbTUn7ym/K2usHj/J43FcYYY9TzQE0
    pLBOFkYHPyEQf4yI3cdrfd/i9+HzaO/wLj3bNyqPxQNfo4OPMXW5q9PyRsfibFW83Ocd5RW1JxJt
    FUrjUh3qb8A1/UsNBNM9bKLW0M2EIudeD5IGnkixlylnO/l6SxsZu5c76/zMGxZmtSzGDHS9eTua
    2XkEaPlce7MIdlJR8miozWhNqMfON659CcSGQG+UFlHkNQRRw4qSrmzw0uhhiI9CmvwAtUKKjMMb
    fR0UtcDrIoTknMQtLB34Kt8Pnc9l7ddiE5I9lZ4fNu3tZbw34HVTpfIDog82ckbTu31erwBov7Md
    /84ZoewIwX9oooY4E5WfNbx5cBi/l3mXDDezcvcmIdyQmd2XUagoCDYkOttjWCjBVsHLfd6tp2Xz
    wM7zKDOUsnuDjPgdh/Zus99G1yswjJuBxvnFYYNw4zau4+9/D87r9RGCunNfuNCNZFagN7o9dYXf
    Vft/vKzaAXpF+sq1Dmw7oLROKmCCrUK5EMYuDK7o8DvLhz7PogH/4YyEbbXnKgw7eR6cjIbk333f
    Vk4rrMux7Q6/tYyI3cftHQ8X531X1J0Rv/2V27ZeyNWbJijPaReGx3qBhjSU9fWHWwoe2XuqzzFm
    ep3uU3hgjY7byfmJm5SlFaI0F7ek/lTvWLK9nPcbNBL5qKAfi/JC1ZHOmNbsq/Ya5mZ+A0IlwyQ0
    COYwN/OrUJvR2jCX5x7BmzjFQyC9e2EPxNqc3NTxZ59jpMTnB8l3E47AwhN187yFqC6E8ZV+2E6r
    4oaUX7mr8w9e4+X7quI8biJOTvuasxICa704LHY/5FRnrjzd64PacMy27AuvmwAAIABJREFUiiQm
    bLqitux/Y4W5rDanQncjs/HmhblD2VTuO6XUVFjGT8jNJiSP96gWXDSz7zK96xdUSRvfFvVgdLtd
    PJD2bb0QVLlh574d5yvPF2R+IWvWoha9Y5b+NzJmHIUMs/RAwQLs6KE2ozVizrnrukH69Bkg3jNz
    2YTkNSTafcd4fy/r5DOfOd7uPTzQJSKwDf/SBkU8/pzDU70W8+cOvsX9PMX/j2u3m6lpgS88BkRW
    ryZv77iCEYckikuMCC7feEW9xhOF7iikVMtrMhBKMWozmi0Ggkf9rNoBv38LddlX6Xvl/peOP9fK
    Nqu22YPq7KtHfVT0ztx9pmkNnqAh5XRaXnBdkr3vBlI6x+NDD6plEZ9h5/oWe4NpY5jPV8+a9T5C
    /GDmkus8SPM25NPCPj7P2/D+++2jWGTTkChb/ZCDP/VBFe33hnHvKM3F/N6+q3H90TOygLSIIqZ3
    +7z22NSd57CmvGO9cW4plHqNgrq+itPEn8jbuUN8NseuQTXeD9X7Cd5Itpczs+vhn0m5Ozix8eUl
    3Xhq/wlBmcs84mfmzApNI+35850U5V2GYEFI7l8P8T5FuWMOJXNYBEBgxUiGMU11aL/oXE5o5z+L
    8tNC31WAvja1Eu0VHrsx+aOhk/G1ko21OZXSB/c2SN1rqGd+wNmOSdsvYHlJN2U7u0cW8ETPpbWp
    oD+XduGlA8c0GmcXRm3Gij9KDTXpAacJka1lfh7QNcQrZupICeU+8tf/0ePDeq31Ht75JyoCkDSu
    S6nbwU2bLw6dzpAgg1C2SZo3rxK7uBIhHq3+DbQ0UiLE33BwGfPmmd/Nt6glMOc+Z9ZnwOd+xwFX
    tv/Nb5igxIjgh2Lfzi7XT8f66V2/UOrYU5e6GuBuKXyGZYbGHFBafR+sY2ePyALu7nT4JSfXFcNZ
    627kuQPHMmfPacp2RmpuxiVtqLXzrm1jPdoaq9hmrtoWtcwS1TcBUHfaqgVMbjSP+xdQrTpZN0T2
    WWFvXj94FF94ya1X5Z4dY9haGaJwjBA/MFsPfccmXTeYrU9GMgZQq3YLDtnApczWH0LXwzh7p3UQ
    eCcmzZis8mS/KHm936lWl3Ty60S2+GmQfFbCVj4Z9C+uTVnFJcnrlDYv4+o4wzy374fH8Jj9fucD
    qKxTBp/VbVltRopTaly56fLa/0eg8dwXs0d5bDkI+Kz8bUien4fl4XHq/UaTFGPpqjo4dmF4fBAM
    iznA070Py4uUG3buPdQL97Wco5Xm9sT87FG8nmMqVyDYTA/lzRsxZ9aHGFWDgefAxOaLeQzgZRxi
    EFmzmj/98wghcOeemfkTUngV8IHqEnuVsv7V5f573P5W5n/MiXG7mN/7fV7vt5DFA1/jKD8Oue7K
    /YAz1sdIGBCl1lKx8pAg3Ylxu7gkeW3t8Qe2n883RYfFzXL83M8T5YaduXu8b1iqaKbUkOtUc9oq
    ol01JCiu3H11ymrIqfHb633fJyqPt/u/Ua8Z9oxdZ9VqxS/KH1RPN16V74u78+D280xfFzSE+I7Z
    +rLQGeCFuXNzyZp5O5p7BIL/Etx8eAPJW2jGcLJm3oSuB7Z5ZuGRJvZQFdPA+07neIVVO8Dvpf4d
    92cmtNpr8CcDkFYnyyanyrez7a/o3CsMe62iYE046qOCfszPrt8MpNAdaTo+/FLOMb4zihSdK/gP
    c9VgJtCVpFicZca5z+n+Cb0j8xFCcnHyOj4f/HK9VMrvi7vz9P7ja793S8Hd28eaihavL0/hso1X
    mApBBR3DyAzdzRXIzPyd2TOvBlffQ5/7dYFPJjaAmIFL9GLOzAlkZq71f42FWZq2+zRH/410/X8g
    r/R0+px4712W6vKbj2rGGn4t68zWyqRG7eN8sc1P7LR3nYKibD8r1Lqbor6oNOxMSF7LqEOFR8Xu
    CO4+FDJoyEFXDF0V1RErDDv/2HOSzzEJPtJFG5KvGG55uMu3VEmbUkqkakpqmVo/F6BaruH3o+fh
    NLRGm8X5rmj+suXiRvsPXxT24vnsY7mtQaGSJ/4o78C4P65R/nk0H+JF0vVMNq59qdmEwoJBVtYO
    qqVIZjNZ743NOAWpnYiQ/ahuPJJ46B9AEZAHbEeKtcAqDJbxiL49FKYfaTTNuafrJ4Bs3C6J6uyS
    Y+P8y8q6pMa6Bil9npBS8MTek5jXa7GSaevLU1hf7lspsm4K5W4fAlXRmouukWpO2I2ol543fdfZ
    7Kr0rGyYa8K5v5Iz0m8Tb9WwSM29/XFy3A4uTl7Hvqo4Ois47u6KPVF3VCT6DZnVRUM2cuxSwq1b
    x7OjMtHjNQ/sOJ/29nImtF/jdd7virpz+aYrTO0rNBuCriCfp/+QO8gY8hCzdf9ttULNI/pWYCvw
    71CbYtGYwMIyut6VdP1VML4HPDYfPTluh9Jqb7+znXJ44l85I1hZkqY0dtbuM3yetwlJzzpvAX/4
    eBB0iShS3gS8qsNvtRID3xd3Z76P3qwqjZ9rmKeQd23GuftzaHZh8ESv6nRr1TBKt4hCpYwif5vj
    KmTtPZ0P8gd4Pe+Wguu3XMIDO87jYAPFyGJ3BDN2ncl5f1wfHo69HvIopPyY9BmfMlUfGWprLFov
    5lbuDzwQS1S7B3HKh4EoX5oup8RtV5rS32q0Li6pcd2WS/l00L98xtNfyj6G9/IG+Zyra0RhvdXg
    Op/OXb0C9vyEaoE9KeG+7ef7TK8sUWxzB2phlFgzG6p+Vu63p65gaHR1Flw7xXCPQxikRRSx08ub
    Sg0/lHRnEt+rGeqBV7JHMsfHxnINUgqe3n88r2SPZHTcTnpEFpDtjOXrop4UugMTm2tBzkYYK0nX
    38LN5EOrZAsLZdScu65rVHENQj6KlP53P0G5icR+E0qGUJ1CePa6G3ml7zuNiqMMBPP2ncDUnef4
    nWdgnRi6lLChwvfKXZWaTdS384aw2k+Gjxnn7qtCt3aMiSpYXw+LTo4SMrp+Wft9pIksuGExB/w6
    92WFfShwRynLDtfl7dwh3L19jNf8d0+UGQ7lAqvwQgiQE7AxnvQZ/8KoymDu3JbtKmPRavH/vp2h
    n41T/oqQ/waUHLsQslYDxR9mZGpr2F6ZyFlrb2TcH9fwz30n8sbB4Ty29xSO+/02Ju/8k5KA1PF1
    Hgw7qxIp9VG63tGk1rtLaszcfabfcSWKVaKglhtupo9qpY/K0yd6Lq1XvVvdUUltXpV6gHLDzj/3
    nag2YR2eP3As122+1FS2TRshApiIFrmBqdMfRtfD/rXDIvR4X7lP0QeiMQsp1XVkD9ErokBZHMqX
    dogvDATLCvsEvCI7Me6wc/fXf9VM/jhU7w2o5FqXmIi5q6zKzTh3bzIF13RYXa/w7MOCfjyz/3hu
    T11BXwW54qNi1TZKn9g3mguT/lDS6yl1O3hw53m8kn2kh6BlEkI8gpO/kq7PDovMmoyMARjaRX7H
    aRxk9qyXWsAii0M0du663hGnnAnyZo/nFRgSo95s2kxoIljYhVGbqgjVG5++MFPWX11opCYtoKo/
    DqApOHczkreeQk29ovL5R8/DmlUFrij+unUcAJ8V9lFy7qqNtysMOxdtuIp3B/zX61uelLCoYBAZ
    O88OqDCp7SK7Ac/Tf/CtTJl2J3MzTQn5BRf7MIR8xO8wyTrAcu4tyGHvousRuIzbcTITUOtK7IXe
    Ueq56DLARhtNYXjM/noVjt/WqRz1RIxNXdP8uf3HKYeaDBNxY5uCbk6xS/1NYFS7PbxxcHjt9yn2
    Ut4b8Ho9CYP7d5xfu+H9WWEfblXIG0+LKKJ7ZKHfuDtUV7+etvZmbur4M1elrGZgVA4OYbCpoj2f
    FvbljZzhjZQvLeoiBiBsrajBtUVLcti5O+VMEJODIUjX00ShkRkHFyzqhmSK3JGsKfddRGXGxp9K
    1VI1oVoYSxWVkEuBiQyQqzus5rkDx7GpvD09IgtY2P9N+tdRvVxcMID/1nH+Xxb1wik1pW5Vo+N2
    srNymJIdTqnx3IFjee7Ascq2W9Rg3MScWd4T+S2OaA47d4eYiVOeCRzX1El9t8Srj5lQQrA4N2FT
    7dfLS7r5lXdV6VhUgxmtcjP/d5U3nHzFHqJQrcy4fOjzbCzvwKDo7Hox+DxXNHdurd+vodgdwfLi
    bpwSv8PnvCVGROjkco8s/kHWrP+F2giL8OXw0lHXK5DOi5Co5TD6QKWasYaWdu4JtkpOS9he+/2H
    +f39XuOr9V5DzDSUNvNGoDLWX3u7hsRoTo6O3VfPsRsIbtx8iUcNmw8Lvf+sSt0Ontg7mkGr7mFh
    7lBTdliY5nty9k0OtREW4U19rzVnzj6kcRmIJonkJ5nIXzbbo7OpnJ+0sbZyVkpY7KPKsYaaHqUq
    mFm5mwmAqTwEd1Ym+EzpVGHOntP4xEvjlHdzBzc6VmY4eHr/8QxdfTdTd53TqBrUIujsxyEmMH9+
    y35wLFodjZek1Tvv9zZl0gQTPTJTTOaQN5XxSYfT/H4t6+JTU6aG/SaqaM30B/XX1q8uKit3A8FK
    EzH/hryXN4i5u71Xfm6vTKwVeas0bLyUfQxDVt3NAzvOCzil1cIMwgliArre5Ldri7aPZ++SpT8H
    zA9kQpuQxGnqq9cODvWmxk0lWnNxTsJhpUpf2iR12V+l7rjMPKzMhKRUx36QP1B5zrr8UNyNv2y5
    xO993swdzlP7T2DAqnu5c9tYy6m3JMK4hyz921CbYdE68L50zNl3J/C12QkjhNtvW726pDhabuV+
    fuJGYuukNb6b1zjM4IldTvXM0E4RJcpjm8O5v5ZztGn52m+Le3DxhquV8u6f2DuaB3eca6qJh0Uw
    EK8ze9azobbCovXg3bnPn+9EOi9HYqrztNlMCbOl/U1hYp087R+Lu7KhvIPSddsqkpQbOXRymNhM
    DvKGKlSndqpo69TwVu4QLvzjGgpNVMu2VobFHOC53ou4IeXXUJtiErEaBxNDbYVF68J30HfOnAMg
    xoNQa7GDudxtgL7RuQgTgleBMjD6IKfWUar898ERyte6pKZcIWkmU8jMg9DMKv9fOSP4+96TfY45
    6Izhtq0Xcu3my0xVyrZGkuzlPNd7EcuHPc/1Kb/ybO9FZKR9GWqzFBH5uLkEXW+5+KVFm8C/J56j
    /wLGraoTuqUw1eIs0VZBNxOqi4FyS+pPteGiUreDt3OHmLp+fZlapWSCrZJkxXZzZqpzzaggQrXY
    1yUbrmJ5SbfaB4NLaqwo6cqDO85l0Kp7+HeO+gOutXJJ8jpWH/U016f8Wq8Q7OG0b5T6+4YYA2lc
    bcn9WgSC2pIta9ZrpM84DrhTZXiZ4agX2/bHUTH7lcrVAyXW5uSqDr/Vfv923hCKTIYhVpSmcWl7
    tVaP/aMOsrykm99xpsIyAdQDfFjQjw8L+hFrcxKnVZLrijGVs9+aSXWU8GzvRZyfuMnjebswmNHt
    C67YeHkLW2YGMYM5Mz8MtRUWrRP1T3rOvvsQ4iuVoWZSBwGGKyoJBsrV7VfXaodLCfMCkJtdUdxV
    eWy/6Fz/gzC3cm9KsVep28F+Z7sjxrFflLyen4c/69Wx13BB4gblt6yWRyzBwZxQW2HRelH/tM+f
    78RdeSmIbf6GmkkdBDgqRk37PRCiNBcPpX1T+/2Hhf0DEqNaVdZZeVN1oGIzbbeCw852xjJ7z+nK
    rQiPZBJslTzVazFv9FtAe7v/ELVDGIxL+qMFLAsAKUfgdh8TajMsWi/mlnIVFWUgN/gbdsBlzrmf
    Fr9dqd9qINzScWW9lnz+Nhq9UWHYlfu3qmiUg+84+m9lqdy5bSwDV91L1u7TLL0WP5wZv5WVw5/h
    po4/ezz/woFRHPvb7XxaUF////R4v2uV0CDogqF9SYZ+WahNsWidqDv3KVPaExP/CXCev6E7vXSk
    90a8rZKT43wLUgVCO62KB9MO13z8UNyNH4r9x8K98XlRb6Vxx8TuUcoAarhyNxB8XtSbyzZeyQlr
    buWl7GPafCZLU4nWXDze80MWD3yNrl425hflDeSeHRewprwjt24dX6+T06CYsN5UjUEaC8jQH8FX
    w2ILCw+oOfeMjH5okT8ASsvemhJ1M4xLDv7r8V87/1ivYnTG7rOaNN9nBWrOPcFWSR8FZcyaDdUS
    I4KXso/hmN/uYMz6a1mS3990dsyRyDGxe/lh6PPckbrCa+FcpWHjoZ3n1v489znj6kk09I7034Ak
    tAiBlA+TMeNNJk0yV51mcUTj37lPnX4W0rYCZD/VSVeVdjZtyJUdfjMluuWPTo4S7u38fe33H+QP
    4Bs/TTn8sbI0TanYp9gdQaLCRt2+qjim7jqHvr9M4s5tY/lDsajKAuJsVSwb/AoD/OxvLMwbyo4G
    b5J1i9daWrguYCSXE53wBbqu1MfYwsK3c8+YfiNC+xAwFWfZWNHBtDphgq2S6zqsMnWNL57pvag2
    Q8YpNdJ3qVdtesMlNb4o9L56LzMc/GPfSQxadY9SfP6Twr48sXc0hSaabFhUc2b8FqI0l99xL2aP
    anRsX51OWTYhlRqQhAVCHo+TlUybZm20WvjFm3MXZOg6UrwM0rSGrFsKVpeZX71P6vJdUFbvN3X8
    uV4a3AvZo0xrnXtjcUFjsbEqeVghMX3n2eS6LNnb5uaCpI1+x+ypimeFh4ds3X60FYa9laWIyjQM
    21ekT7841JZYhDeN/6rvuiuSjBn/QcoZTZl4aYH/JhgN6RJRjN7t86bcluPjdvNo949rv99VmcDM
    XWc2ac66LM3vX7sh55Qa/z04nKNW32kpJLYgGpJzEzf7HfdhQT+Pexf2Oiv11vkglrHA22Toeqgt
    sQhf6qdiTJnSHlvku0h5SlMnXpQ/kNndlpm+7tbUn/ikoC8fFSiH+GsZEbuP9wf8p7Y6Vkq4Y9s4
    09Wovsh3RfNdcXdyXTFM33UWWxQ1ZyyCx8h2e0l1+Fff/Djf899QmuNwVk3rfSALgZQzyJgxgNKi
    v/DEE+FajdX2mDjRQXKy+h+ONKKVpHIlMUzW1bI2AERlIXPneq2YPOzcMzL6Ie1LkOobp77YVN6e
    9eUppvU7NCT/6buQC/64jhUl6lWhJ8XvZEG/N0moE9Z5JWckywr7+LgqMMZvuIZKQ72vqkVwuSDR
    f0im0rDxZVEvj+d6RR3OZFqnqBkUtkiuIDahB1OnXlwt9GfR7CQnt8MWuREIbgaE4FRscovyeMN+
    OuBVNaA6vpChn2k2I0aF/+UOC+i6WJuTpQNf5caOv/gdG625mNH1Cz4a+O96peRbKpKZsvNPAd3f
    H5ZjDy3+ZAUAvinuSYkR4fFc36jDi51AqpXDDilPRDg+QdetooiW4JFH8kHMDK0RYiFzM33KwWhk
    TL8RKU1nxKjw3P5jA9YJj7U5eabXB3w86N9clLy+UWbEwOiDTEn7mvVH/x+T076uF0ctMSK4fOMV
    QQ3HWIQHHRxlSnIV3oTb+kTl1VsErA4gbTf8EHvQjBvQdf/pQxbBYePaZ4Hf/I5rHipw85C/QXak
    lgKyWZ74he4ontt/HA/X0XYxy6nx2zk1fjsGgn1VcZQbdrpFFBKpeZYrkBJu2XIR68pTAr6nRfgy
    Om6nUvjyl5IuHo+f2G5X7dcVht1U6C9MWYmsupDMOc0n0GTRmIUL3WTok5Dysxa/txCP84i+3d8w
    jSz9MaQcCzSLqPq8/ScEpcuPhiQtooi+UXleHTvA3L2n8V7eoCbfz6L5ODFuF/d1/i6ga1VlKlaV
    eq71OT7usHP/saRrK5d3EO/iEKcxx3LsIWG2/jmId1v2puIAdh5TGVkdc58z60M042QQQRd4yXXF
    MGn7BcGe1iPz9p9A5u7TW+ReFuYZHbeTtwe8weeDX+bYdnsCmuOkuJ1+x+x3tmOfF9npcxIO71d9
    7qMgrRXwTxxcZnVoCjFuHgAqWux+kofQdaWF+OE898zM33EwCiECj6F44Y2Dw1mQOzTY09bjqf0n
    8NCOc5v1Hhbm0ZCMSdrIt0Nf4LPBr9Rmunjb7PRFO62K4TH+tf9/KfUckhkeu58ekQW13y/xUJAW
    /gg3Qt5J1sx70PVWUlrbhnlE34oQT7TQ3X4hgv+oDq5fxKTrBynMPQfEa8G26p7tY5olJ1xKeGTP
    qTxoOfawIkK4uTplNb8Mf4a3+r/BMbF7650vcZt37ifE7aq3ce6NX0s8b5KOSzqsVr2pvD1rW18a
    ZAnSGM/sWU+H2hCLOtiZg2Sv/4FNQUoQph7ojStU582rJEu/HiknA0FbGRS4ohjzx7XsrooP1pQU
    uSO5YvOfmbn7jKDNadE0Ym1O7kz9kbVH/5MXe7/nVdirNADnPlohJAOwvsLzZvrYxMPO/b381rYv
    I/agGacyZ9aSUFti0QBdL0HIqc16DyHeIEv/1v/Aw3gT1ZDMmfUoyMuBoMX0dlQmcvram4KSW7y6
    rBOnrLmFRXkDg2CZRTDoGVnAxqOf4G89P/KqrV5DIGGZEYpNULZ6eEMcGH2wXhOVd/IGm75/6BCr
    cXACmZm/htoSCy9kzXoVKX5snslFOdJl+uHhWzEpa9bbaMZoBGpLJgX2VMVz6ppbmJ89KqC+oNnO
    WP66dRwnrZnIxorgiIFZBIe0yCLlnqSBhGWGRGcrjdtWmdTo2LUphxVH15R39ChLnWQv5+aOP/Ov
    Pu+woP//mN71C78PqWZH8h4ORqPru0NriIUfJIJ7q8MnwZ7ZeJSsLNPJLv7l8DIzV2MXJyLFTwEZ
    5oFyw84928Zw8ppb+CB/gJKT310Vz8zdZzBs9V28nDPSajsXhiRo6kkDpSZX7gm2SrpGFPodV+CK
    osBVX0LZLgyu7HC43uS1nKPrnY/SXMzp9inbRz7OvF6L+XOH3xmX9AdT0r5m9VFP1QvntDD/JEJc
    amXEtBKy9OWgKW94KiHZTVXZ3wO5VC3JV9f3ouun4xT/AjkhkBt54tfSzly+8Qq6RBQzJmkDo2L3
    0CcyjxibE6e0sbcqjjXlqSwr7MNPxWkBrfQtWo54u7pcs9mV++CYbKXipb0eUiDPTthCZ0cxUK3k
    +UbO8Npzg6JzeKPfAq97AzGak1f7vcUpa29p2Q1YIb5itn5Py93QIig4eAgn44HgbC4K+RB//3up
    /4GNUa/gqF49/Jn0GTnAHYHczBt7q+J44cAoXqBxYwWL1kO8Zsa5mytsUw3JZHtQebyh4+FQ9UcF
    /chxxQLV1c//6/c/Eu2+3ziiNRczu37GZRuvNGFxE5HyNDKm38rsWc+33E0tmoyu72fq9McQYnaT
    5xLiB2bPfDPQy811KZg0KQrEiYHezKJtE+/HSdbF7Ibq4Bg1536gqr5z7xFZwNjEw/15XzxQvYC4
    vP0aFg34j1/HXsN5iZvo4Gjh6IjU5pGun9qyN7VoMsX5fwf8NxzwiZRI7gUCjuGbq72OjX8OKUcE
    erO2RqzNyeCobHpF5dM1opBkezmJ9grKpZ0ydwTZzli2VyaxqrQTe4KYAhqumOmiZTYs469Xag25
    7vo9pG9L/Qnboc5LWyqSWVbYh5s7/sz/9VqCZuJzYxOSUbF7AuozEDjSAWIBuj7K2lBtRcybV0n6
    9Mkg3gp8Eu1VsvQVTTFD3bmn6/ci5XVNuVlrJ1pzcUrcds5N2sQpcTsYHJ1d6zj8sb0ykcUFA3g1
    ewS/l6U2s6WhwYxzN5vn3sWhlrVSYRzuChmjObk+5XBIZn72KG5N/YnHe3wY0O5N3erWlkOm4uR9
    Jk062WrI0YrImvU26fqnIANp3lyCgybnzas593R9NMhHm3qz1ohdGJyfuJHrOq7izPitxGjOgObp
    GVnAnak/cmfqj3xR2IuMXWd7LZNvrSTY1MMyxSadeyeFzksAVXW09q9NWUXSodTMcsNOj8gC7kgN
    fDGkakMzMJKYhOeBI3px1erQ3JMwtFWYjZBI+Qj6zCZXvPqPuet6J6RcCJhPTG7FdHCUoXf9nE0j
    nmBB//8xNnFDwI69IWckbOOboS8yt/unPhUuWxtxJlbuZQ1i7kfH7uPWVM/ZttGaSzk2XimrnbuG
    5I5Oh2tKHMJokmOvniOUvyt5Len63SE0wMIsmZlrkeIFcxeJXURoQdGq8f1EmTjRgYsFCNrWEtMH
    7e1lTOr8Pbd2+ol2WpXHMfuccSwr6MNvpZ1YW9GRHRWJ1KTdp9pLGBydw1mJWxibtIEILw5BQ3Jv
    5+8Z1W4Pl224gkJ3lMdxrYkExVRIKatX0qPjdnJJ8jrGJ6+na0QRuyoTeP7AsY3Gd4ooVrah4pCE
    78Xt19G/TsclT5o0BsJU3F1F16Z5kY8zdfrvzJn1RYgNsVAlggyc4nKQihWXxv3os4Kyc+/buXfo
    8kQwmmW3BjQkN6X+zKxun5HoIbyQ74rmvweH8/rBo1hV1gnppYhqG0ksL+nGyzkj6eAoY3KXr7m9
    0wqvTuTkuB0sGfQa566/gVK3w+OY1oKZVMjdxzxGnK3+wzNS89xIqCZHXYUqw44Qkoe6eBY3rTDs
    zN5zOi8cGEWM5mTJoFcZrNjnV1PcX2lG7AhtIZP143hE3xpqYywU0PU8MqZnIsWTCqO/J2tWEzZh
    6+M9LDNVvwoh/xqsG4Uz/aNy+WzIK/yz55JGjn2fM44Hd5xL318n8cCO8/i1tLNXx96Qg84YHthx
    Huevv85nw5JjYvfyUp8W1vxvBhIUQydC0MixA0R4CVGl2NVrOCqljTEJGxke07hXdLYzljPX/YXH
    955EkTuS/c52PLv/eOW5AxE7Cz6yPTbeQddjQm2JhSJ27WlgjZ9RBoh7aELqY0M8O/ep+nCENBkr
    ap38ucPvfDdsPifUab8G1Su8R/ecwrDVd/HU/hMoMwJfVX9d1JNLN17lc47xSevrZXa0Rsxky3gi
    UnheuUfb1FuDVhp2j20dDzpjOGfdjfzaQFNmb5Xnph6eKDLCpSevPAonr4JVst0q0HUXUt7re5B4
    hSx9ZTBv29i5T56chOAdoE2vDGxC8kTPpfyrzzuNYuu/lnZm9Jp0CE0uAAAgAElEQVSJ6LvPDFqo
    5Lui7kzfdZbPMZndP/O4om0NDIw+aGpD1RPeNpe97Vt44pT47Yxq0OWpzHBw0YarPQrNqaayQmBi
    Z82HvJR0/f5QW2GhyJxZn4H4wPNJUYysmhbsWzZs1qFhi/wPyD7BvlE4Eam5eb3fQm7zkJ3x7IHj
    OG3tzawPsMH2Jcnr+GHofJYPe57xSevrnXv+wLE+9UlS7KXc2sSMjpbk3MTNPNlzKeuP/j9+Hf50
    kzccNaTHOcw49ys6/N7o2AM7zuNnL2mn7e3qe1dFQegFHFzko0ydfn6orbBQRLgmgfC0AprdHH1w
    6zt3F9OBlml4GiKiNBfv9n+9keN1S8Gd28Zy3/bzcUpzqgxQneO9cMCbvN5vIUfH7uOomP282X8B
    d3VaXjvGJTWePuA7xntLx5WmMjiag6Nj9/Fw2jdM6/qlz3Fv9Psft6b+RM8gFPesLevI7D2nIz1E
    GrxttHqi4YPg7dwhvJI90uv4XlH5ynM3lDYIPUKiaaeF2goLRWbP3oLgnw2ObqUo9/+a43aHs2Wm
    TDsFKYP+ahBO2ITk1b5vc0bCtnrHnVLjhs2XBtzAYVjMAd7ot4A+UXn1jpcbdhY36NP5Xt4gnuy5
    1OtqtHtkIce1283ykm4B2dJUrk1ZxTO9PsAuDL+NUCqlnWjUHW9D1pen8E7uYN7KG8of5R28jgs0
    v3xLRTJ3bBvnc0zPSHXnvt9L0+0Q8Qma+wEyMxu/qliEL1VlmThirgFZvfkjxP3Mm9e0eKYXDi9R
    52Z+C/IKEFt8jG/VPNlzCeOS/qh3TEqYuPWigB37yXE7+GLwS40cO8Az+49nW0X9xhH5rmhWlqT5
    nPO8xE0B2dJU7u/yHc/3er82NOJvA7E8wE3mD/IHMnT1XYz87Q5m7zndp2MHc3HxGgwEN26+xG8o
    xczKfV9YrNzFBpCXkzXzXMuxt0Iee6wYYRxaRIsvmK2/11y3qht/kGTNWkjO3kEgbgXUZPhaCdd0
    WM3NHX9udHzmnjN58+CwgOYcGbuXtwe8QaytceVqpWHjqf0neLxus59G4cfG7fF5PtgIIZnb/VNm
    d1tWTzPdn2OscJurqq7h68KeppqllwWwqf3v7BH8VOr7IQooh5TKDEeoC83ykHIyDoaTNWthKA2x
    aCJ27RWE+OGQ6mPz3abRkfnzncB8Jk9eiD3qYSR3g4xufGnrYVB0Dv/Xq3Ff4SX5/Xls78kBzdkz
    soAPBv7Ha/rfssI+7PegLQ6w2U97wGEecrRV6RuVx9kJWxgQnUO05qLQFcWK0jQ+zO/vMRXTIQzm
    937P40akP831ChmYcy8w6STNpiAWuKKYvtt3ZhJUq3qq5tB7+102REMiBEHsFCacIJ/FIWagzwyF
    cplFsNF1A10fi643ft0PIt4/nY88kg9MRtefwsk0EDeBtHkdH6ZoSJ7rvaiRLkyeK5rbto1XLkiq
    i01IXuz9rs9+oR8V9Pd6LtsZ63P+DrZSIoSbKsUftxCS8Ul/8GCXbxkZ61lvKM8VzcxdZ/JCzjG1
    /+cYzcnr/RZ6DQP5E/eqKfU3i9kVcJHLnHPXd5/JQaf/TF4z8XbVzdSrOvzG2vKOjfLpA2QxwnUv
    s2e32VDpEUszO3ZQUSur1pG+lWnT/omhPQKMbW6jgsmtqT9xXLvGUtjpO89RcgCeuLfz95wU77tn
    uK8Pt78ydiEg2V6utFocHJ3DvF6LGR3n255kezn/12sJx8bt5rat44m3VfDOgDcaFW/VpdjPyn1T
    RXtWlHRlScEARrfbyeS0r/3aC/is1vVEkYmHwfbKRF7MVuvo1SNCfSGc4+eBDNVvAjO7fcaT+0Y3
    1bmvxDDuY26mZw0FCwsF1JdemZlrgXFk6GcjeQzCv2lHvK2SDA/pfBvKO/DawaMbX6BAkr1cyYn5
    Cr2o5Far9Iu9IeVXnui5lCgvqYIlRkSjAq1rOqzGJTWOjd3DED/djfzF3K/ffGnt16fGbfdrbw0F
    LnNRPjP55VXSphwSMZPCud/l/0F7X+fv6BJRzMlxO5jnZb/FJ5LdCJGJgxfRM0OtUmbRyjH/Xj1b
    X4auj6JKXINgNsjQ5OwpcGen5R5DJ0/sGx1wTPSOTiu8qkXWUCVtPlennjJrGlLqow2dhuTxnh96
    LMLKd0WTuft03swdRr4rmm6RhehdP+eqDr/VjrlBUeag2EQ4JMZEZa3ZlbuZGH2KiaKktEi1BiAA
    OVW+V+5pEUXc2/l7AE6K34kQ0mTIT3xIeeGlVkMOi2ARWNBU1w3gVXR9AVXyHgRzUepN33JEay7u
    rKPnXUOJEcHbuUMCmjNSc3NHauM5G+JPYOqkdr5DKHmuaK+yBw5h8Eqfd7i0/dpG57ZUJHPe+uvZ
    Xael367KBG7acjEuqXFdyiq/ttel2MRGpr8HXl0KXOZi7nsq43FLoZQSmWCvwC4MXAqFaEk2dT/q
    qfF2XR7p/kntvk57exm9IgrYWpnk85oGHEdERARgOfe2gq53pIqzEfIYEMNA9gLRHuShPwxRDrIQ
    2I5gM1L8hOb+hj/++I2FC5vcPCAw537Y+Aoy9A0B7Uo2M5e0X1vbhacuXxb1Mt2cuYZT4rb73ESt
    odLHRmiqo4R+0blezwOsLu3k8biGZH7v9zw69kJ3JOM2XFPPsdfloR3nMT55PQkm9F/MhENiFRuZ
    GAi/sfyGVEkbe6vi6RZZ6HeshiTZXu530xrw+PfhjTy391DS+KT1XFbnd/JlUS+zjh2Q7XFETwJ0
    kxdahBO6nozLuBoprsMpRyJq0s1rFiZ1FygyGogGOiE5AeQ1GBr0H7yf9BkL0Iz/kJnpuYONAubr
    7OsjkHJGE+doFq7v4Dn08E1Rj4DnPDdRraF5lBd1Q4AzE/zLcH9d3NPj8Ud6fOIxZRHg4R3nNiqY
    qkuhO5K3cof6vXddzDhh1bBMuduutJ/QkG0mnKVqOzzV7k4ATsPzR6W9vYx5vQ+n2VYaNu7ZNkZ5
    3npIeT+67l18yCJ8SU9PI33GPJxyD1L8ExhF4P61E3A3hraC9BlfkaEHJAnTNOeeoV8KBLYz2Ywk
    2cs5Mc5zFsjmcsWGKB44O0HNufvSQlGR9X3PQ7XsVR1+q6dTU5ffylJ5Lcf/r+GLol5+x9TFzMpd
    NSzjDvBPzoxz76GY4mim56tda7y/KYTk6d6L6+XKP7HvJI/qk4q0w8VDgV5sEQLuuiuS9Bkzwb4Z
    uBMIdqXbqUi5hHT9EzIyBpm5MPCwjK5rOMNTi+ashK1eFQoPugJLf4wQbvpF+Q6n1BAlXB7jvv2i
    c/1mlfx/e+cdH1WV/v/3uTOTQgIEQhEIIEUBUWzYxcWCHQQFCyrqquzXXf1allUglEMaRfmpa1lF
    17aLDRERde2iiGX92gDpvYYWUkidmXt+f6SYkJnMvVPvJPf9ekXJLec8kMlzz33O83yej4v6NirH
    7598gCd7+VELBR7bc7ah1fD6AGX+R2JG4tZXla4vglm1gznnblRSwJdImT+6JzQOCeVmfNpAgG51
    eUdm7T7P8Ji+jVJ/QcrHalKQbWoof5fbFdxj4FLfMc0jUIrLyt4l4CaUUCxMvopsnyenTj0VXfs3
    0LQIU1hQw1DOn8mUk8mVj2KgqUfwzt2tjwExKOj7I8hpKf7L94NVXOyVdMiwxokQ0MFZ1ihP/faO
    Pwbcdp65s6FzcAqd5/ssItnP20ChJ8nwBvHGinRTfUPNFCm1Mrhy14PcnjEjV9DL4Mrda8K539bp
    J/65b3BdYdeErl9zf9dv6s57lMadm0ZSqYdc55eEW2UCd4U6UDOjs4ATwzWYEKQBaYEvxHfMe7K8
    A109QfhX6k2gEoG5ZE6/EJe4ASmbTPcKNltGw01mGDtChZVBKfl+z/VILApKcbGvgfTF+nR2HW7g
    3JM1DzcHyFZZVHBcI9vuPuo7TvVTdQrwUeExhitZy3UnBz2tDJXcu5VmapV9ZAWwP4J17q1N9Gc1
    6tw9fuLovjg26SBfDXyOdw8N4MzUHZzXZmuD83N2D+EnP5rx5hG3M2XKIzGpTL3//mRS2k7FeMi2
    X+BLAFQXMqc/a9wQ9TG5WQuNXx9FMqdng5oSQwsux62+RMorkNKvcwjOubv1G6pTe6xJho9X6FpO
    a72TN01uLIK5Js1Q7dzrc136yiYzbbZXtm0kT9vRWRqwYGpJocHfrRqMriwrTerGeIPQwDdKoubl
    ga7LKdNdhh4igYqzaikwWUzVP/kA/ZMbF41+WXw0ebvCKauuXCjndGBcGAc1xqOPlpMpO4C6M7wD
    i3bAeIPXluMVM8M7f5jInD4XeCDWZgAn4WYpkycPIS/PpxiV+d/IMWMcIGL51ApIB5f/QpYx7X8L
    qmNQksGVaS2dXA1Xx3/s3FiRspZCbxLXb7iuUf731IylTaYuupXGJ4V9TdlVaTDU4jYZXnAb/CgZ
    baJdn/u7LKdvUoFhTZqMhGJDKaubgt/4rCPfncq4jaPDKBRWi7qRqVPDFoYwhYspQOxEygSPMEtu
    jdn8/siUD2INx16DOgbh+piHHmrr66z5lfsxA28EFYUNhOARTYSLOrsOc1fn/5ouD/dX4u93noSG
    K/e+ib7DOgWeZEasvamRFslRrsOMC5BZs7a8o+nWb0ZDOE3l6ocyrkvopGpVhmsNeiYW8reuXwPG
    Qz8Ag1rlszRAdtCGEJ17qdfF6HU3GMqpDwIN3TEduDoSgzeJlPuYPC0HIR6J+tyIXVQcnt3osGI/
    gsYFHpFGUb2BN0WOROmzLNiTfBCOpJeo/pw0cHzmnPuYMQ4Ek8NnV2So0F1Nrnhze3zCrqo2php0
    +NvQ9MeRYZnL197MnB4fMaTNtrpjnxT24a4tI9jlo/Do7qO+89swupaVZZ1N2QTGUxarTK7cPSau
    75lU2GQv2Vo0FM/3eafOqZtJXRyUEti5f1V8tOHxjsSjNMZtHO23N2t4UKOYOvUMsrMDl0WHmwP5
    f6dj1ztBmYv7hYpiIo880mhTKPkqngOei6ottUiZgVv902pV+HUIRjJ52gPkZc2tf9hcWOaYgbdE
    /YcdBNsqm94Edwmd+ccs4K1jX2N0+m+GXuHNZtkc6dx/Ke3CxWtupedPE7h87TgGr7yLEetu8unY
    nUIPuPkKsKLMUNZXA9ob1F4xGr6pxYy2+3HJxmLiD3Rdzrmtf38YrjbRtPykVv431WtZX5Ee1OZ6
    pe7gpg2j+aAJWeewoWtZkZ/EB/PmuVH6X6M6pxLfkyfnR3VOI7h5HjCerhULhJbNlCl96h8y7tzH
    j3chVGbYjYoARle0V7Rbz7/6vsXOU+bw6YAXm3TypSZbyh0Zc69lnzuFL4p6NblyvbjtRr/312dN
    mXFnB9VvH0bz0R0+inb80dpRRaKJPqfnt9kS8JrTUnYxLeOLuu+rlIPbNl5jOCWyKSnj+kzcfrEh
    HZpaDnmSuXrdWBYfMlVPEgoXM3na+dGarAF5We+D+E90JlMKwQSsloKXOe0qUJfE2ozAqGSU86n6
    R4x/qjt2vQ3oHW6TIsGHh44xdb0Q1Up+TfUuLTNR0APms2vqM7L9msAXAXsNdgeqpZPBsnzAlLOe
    lvEF6QYeRrWM6bDKbwcrqNaoX9T/VVz1Nr7n7B7CirLOfFZs7CPYK+lQo7cnX3xfksH9Wy83lPb5
    bUl3zlj1Jz43aEPYEGIWsQr2evlfwLgqXLAIMZ9c+XXE5zHDmDEO0B6OtRnGUZcwaWpd2pYx5y5l
    Alg/1l7Lx0V92V7pcwPZL0XeRD4v8v9La3bl3iUheOd+vgH9GTBfbXtc8n7D1yY2oY9TnxNa7eV/
    Ov+Xziace6pWxcM9P0T4KAq7ot16Phv4QgPN+xWlRzFn1xAAPivq0+gefxhdvT+/71RGrR3rVzZg
    VXknbt44mgvX3MYOk5+r0FAKJX4AsZB77glO7S5UZsmNKPFU4AtDQZTjFtaLCvQbOAqUuZVirNE0
    WftHY4HSSu+paJq5tIwYUqUczNh5Af/ss8jQ9ZW6gxs3XNtk5yNf/Ueboo2j0lRWSC19kwrISDCm
    M15g1rkbzP8GSHIEdu4aiqd7LcEpdJyYSy8d1/EXOrjKeHzPWexxt2ZQq3xu7fgzF7VtWLfjVhp3
    brkKd03o5Muio/EozVA669mtdxgOn3xc1JfPVvyFM1J3MDh1F6laFXvdqXxf0p1V5VHV8tIR4lvQ
    F+LWFjJbNq0PHQ0SkLi5AYOl/aZR+kxmZ8X+79kYC6U9GmYoU6cOJDv7N2POfWb2t0jZDa/3ZJRj
    OIprQUUt6BgMrx4YxPB2awOGOAq9Sdyw/tqAmRUHg2jJ1yWxhA0mhcpOMLARCNUNmMtNbnoen2y8
    8XaqVkUbR2WTqZZ/7vxfBqf+LvVQqRymwjmXp63n8rT1TV4zZ9cQVtSTQC7yJvHj4a6c0Tqw9MoZ
    BlfutXiV4JuSHnxT0sPUfWGg1qEvQHnfIifXv35GLJCymCnTJEo8E/axFTtJ0OYGvjDKTJnSB6XO
    irUZQaEctwETjHuH6gYdP9Z8SSbK3jj14aCNQelnWzFN6PZNo0h3ljVIP6zP9yUZ3LrpGrYGyK4B
    yHe3Nj2/mWYQtQwwGDpxCIVDKFPFM6cbcIj16ZFQ5HfVemzSQWb0+Kzue7fS+Ka4p+GQkhG+L8lg
    zu4hjY4vKezfpHPXEbxTMAC584Kw2RJ+hBfBd6AvQPe8SW7enlhb1CRO7Tnc3Anq1LCOK9QE5Azj
    7bOihXLeYLW9XeOoMZhy7kcyS24GHgceZ9KkjojEyxBqDHAxEJv44BGU6S6Gr7uZ3O6f8KfOP9S9
    yh/WE5i7+xwe2X2u4UyJPSY3LwHSTORl12Kmr2eC8FJuMAVxQPJ+ehvUXKm7p9U+n87dKXSe77uo
    QVHRnF1D2FiZHjbnvtedyvUbrvNZHLXw4ECyMz5ttJyo79TNvjFFh/oO3f06ub7Lxi2JlDqTpt6L
    JpaFbSEnxLfkzHgzLGOFnXjIkPGDogdTpvQLrRNTLTNn7gdeobr1Xhpu/QqUGIngUsC8VwwjlbqD
    Cdsu5cn8M7kkbQNOofPGwRM4YDLMUuhNolJ3BCwsqo+ZZhC1mCnUSdQ8hkMzl7drOvzhi2s7rGKB
    Dx2eh3t+2EB5c2VZ57oVdm73T+gawmYyVKtRjt0wxu8eyNbKNH4p68LJKdWLXes7dQDxT1xMQMrY
    lfWHyszs5WTKt2pWhqGiI7z3YsXlsZRJuDnNiqYZR7sgPM69PtUf3vnAfKRMwsNFKDUbMF4OGgG2
    Vqbx7N7Tgr5fKUG+uzU9TaysW5toaVdLqtN41lmPxEIKPcb2uIanra37c5VykGAgNn5F23UMb7eO
    JYeq69Y0FNk9PmvQmNujNMZvvqpuhf3onrN5uOdHhv8OR+JVgls3Xh0w7r2wYCAnpuTHgVOvo0Nc
    O/Y63H8F5xVAcI0R6hAvh9JCLhDFb5PucNLbodFe99IGQBOUK0GhrrM55Sr8S616vSdA/CSQ+ESJ
    QeF37vWRsgJ4j8zpj0d0niix151qyrk7gnjye3Xjb7z9kg402Gz0xwmt9tbFqMt0F0/ln1Gn19IU
    QsCrx7zJWwcHsqeqNRe23dxITnnu7nP4pZ4uzjN7T+eGDis5pQmZYn94leAvW4YbynCZv/9E3ikY
    YErnPcZchJRJNb8T8Utu7g6myLmokBr1HEZVhTX1sXIJ/b2KywQMA04BOgMo/fcgkqr5jyag/F1K
    gR+Ar5Tgw+Qr+U7UilLpwpwanxURYmBknTvAlCn9UPFR/BSI/CpzESbNYHOP+pToxhcM/ZMPGLru
    r12W1/15wcHjWVRwnCHnDtXxdX99W38r68TM3Q2lbj1K4/ZNo/j8uBdMNaCu0J3cuvFqw6mL+e5U
    MCfUGWNUClXqD0DwrzVWobRoJiltbkERZFqRyCUv9A1k9REplRWMQ3C7rjjV5EZACjAUGCoU0yqW
    sLV8MS97vDzd+ketV3yHZAClukVOhLsO7dLIzxEd8j0mnXsQH5BDJjTGj8wJ98WpKbsZ02FV3ffP
    7D2NFWVHmZrHF2W6i3EbR/vUh19b3oEr197USMLYH6vLOzJk1Z3RLOmPEVqQnbMtxqOPlqNUkEWN
    YgsuHgtlerWEVmVL+FtFJVuU4GkF4cjgORrBdKeTre/2//c16QY1mKyLaBd55660+N11PoJ9JjNm
    zBYwAaaaK5+eurPJlMDWjipe6Pt23UNmeXEPfintglcJFhWE5kgf2HpZk0JeP5V2ZfDKu/iw0H+B
    3z53Cn/bdgnnrBof7UKh2CC4MtYmhI3crFcRonHnkkAIJoQSmqpYzLAKxUqhmAOYE1cyRvKwthtP
    WXXiE/yp8w8+q6jjA9U60jH3JNwqnC1qYkqB29xqt8SkHg3ASpNKj8/2WswFq//YqKtQG0clC459
    nWPrNfWunzM+/8CJ/LHTT6btA3j9wAm8vP/kgNftqmrDqHVjGZC8n6var6FPUgEOFDur2rCs5Gi+
    KOplSrQr/lG9mCT7M1OuDXyt5VHo3FfTY9ToD3EpOfLtoCZ7k4TyZOYqxd3B3G+WNGcFjx39AZel
    rWf85pGR0uyPIKIyss7dw3mEvKtuHYpNxMMBigyGJerzS+lReJUw3Iy7X/IBlh73TyZvH8bnxb1x
    CZ0r269lSrelDXLmfy7twsdFv+8TfVPSg29LunNWa3NVnOsr0rlni7kF6JryjqzZFYlFVhyicSXQ
    HJw75MmfyJQvg7rNwNU6mj4hmGkOL6ZzheAdoTDXYScMXJK2kW+Pf5aR624Mqn9C7FDlkV02qTgu
    BPDBYY+5lfi2qsCVr0dyyJPM8pKepu45JvkgC/q9zsHT8sgfPIvne7/TqBgqa2dj1dgZOy9AmXjr
    zHenMmrtjUGFm2xqUc0j7l6LqpoEwn/T4t95nuxs/70m/VD2Dt0d8BVE37HX0jWhhE+Pe7FRU3Rr
    I/ZG+p242WymApQp407NozS2G5A18MW7h8LbxfCTwj4+Y99fFh/NP/adbmiMPe7WXLn2ZjZXtgur
    bS2Qc5EyuA+GFcnL24vSAzSzFiW4xHSzQ5cupIvQ+BJBFLqiNE0bRyUL+73WoHjP0ii1KXLOXcoM
    Yly4FEs2V7arUzI0y7/3n0ShwWbQgahSDh7a7v8FKnP7sICiaT+UdmPIqjsMtcazCYgTtz4s1kaE
    lQTtUcB/CbTSZyClMUW82lsW09rh4j2g6Q9nFEnVqljcfz59knz3Q7YUgvWRc+4e/bKIjR0HfBuC
    smCRN5Gn888Iix05O4eypomslgrdyVVrb+Th3edS6m0oa7yqvBN/3jKcoatu99kO0CZYmklKZC1S
    VoF60PdJsYmSQ0+aHbJC8IyqLkayFO2c5bx+7BummrXHBCGWR07JsVqD4pqIjR8DLknbyDv9jLV4
    /POW4by4L/jPZqpWxfeDnjEt9lWf5cU9uGTtrYaVI1tpbk5O3UOC8LK1Mo0tFXYIJjKI/bg4qkZp
    tfmQKT9sLLilRpCbtcTMMOWLuQXBS2G0LOzM2zeYe7dY9hmt4xKdI7Nyl9IJ6sKIjB1D2hgU9dIR
    plv9HclhPYE/bro66FTBDeXpXL/xOlOSwGW6i+XFPfiiqJft2COK6ojXG7zQkVURnvtB1F/Sfm7W
    sRd9SHsE1tN3P4I7Ov1oOtMsagixHCkPRMa5uzkTaD6bRjW01oyJen1fksGeIPTffY1zWxAOflNF
    e4avu8m08qVNFFEOyy77giYnZw2KmoYewoum32d2iIQq8gDLq8BpKB7r+YFFi5z018BMg2xTNK8U
    yFo6JxhrMP36gUFhm/OtgwO5acNowxusnxf3Zshvd7AtyEwdmyihmllKZC16xXTgAIpnyM72LUrk
    h/J36QXcHhnDws+glHxGpFmuZOEwTu0NiJhzt1YKpBCKOzr9yOoT/87Xxz9nStmxPkbuK/Im8moY
    nTvA4kMDGLziLhYeHOh3Fb+hPJ2bN47myrU3hawbYxMJxH4QP4JYhBKPI9QrSNn8ynNnzTqEEveT
    gDR7qxI8gNG+zhZhUsZXsTahIUo8h5QFEIl/yPHjXQg2gqhEkQGqCzHszNTaUcXzvRcxon31E7YX
    h7i7y3f8bav554+RFKiX950SkSKfXVVtuGnjaLq4Sri03QaOSTyIy+FlV2UblhUfzY+lXcM+p41h
    PCB2IdiOYmvN/7cj2A6ebZSWbuXRR833XIxX8uS/zd6i3iS1QmGk0tVSnNgqn5NT9vBzPdnrGFJG
    Av+v9pvwO/d589zADfWOCKTsjNfbDeXohiIDQVdQ3VGiG9AV6AEq7OINbR0VLO43v5G4Vq8E8xko
    QqiAzasLPUnM9tHzM5zscbcOKQvHJmx8DCIb3NtYv343CxYYb9Fl04iKZEagiDcBFwDGdfzZGs5d
    qVnIGXXOLhqvQKqmgCGf6ubavpGyDZVk4CADxURQjevlTdDRWcp7A/7FoFaN21R2Tyg2Pd6ApAO0
    DdBZKXfn0EYCXjbNFVFErjQmim8TEKEYY8WtSSNckbae+7k8xlaIdZQXP1L/iHXiW1IWA6uB1Uye
    /idCyMDvmlDCB/1foZ+fZhYZiUakMBpyUdrGJs//3+FuPLuv+WW32fhDxZOKlKVRClGxhPNibUew
    dE8sol/yAdaVd4iVCVUoxh4Z+rPmho4g6H+lVK2KRf3m+3XsAO2d5aQ4zFWYXdN+td9zxd5Ebtl4
    TdByAzbxiLCde5ioWszxQNz0S/TFWakxzHkX6j7yZCP9buus3BsStHOf0/Mjn6GYI+meUMRag0/a
    45P3cXqq76YYSsFftgy3BbVaHPbKHQAp2+PhJBTHITgapdqBqCnyUKUocQjBZpS+Ab3qv8ya1WjD
    S3dwQrx3tRuQvD9WUz9CTtY/fJ1oVs49zVnBLZ1+NnRtj4RCw849M2Op33OTdwzjrYMDDY1j06xI
    axYNr4NhkjwOTd0IXIxbnUJtBKDOQdfz1LVFPkKAI1EnU64vK04AAA47SURBVP6G4D2E97V6efBx
    32O5X2yc+9O4xEP+TlrRuQuCfEVr5yw33Lf09NSdDZpX+OPStA2MbL/G57nH9pzNY3vONmWjTTPC
    7e4IWLQGPcxIqVHFaDTuQ6mzghxFA3UCihNQ2iQyp/+EEg9XerN7JmrxnWzUyVUa3QmFyCVHTmnq
    EusFiSdOTCPIh86OyraGmzKP7bACp2hat2lQSj4v9mncFUwpmLHzfCZtb17KrTZmcbWM0MwUeTlu
    ViDUGyE4dl+cglCv9f/13mub6rUbD6QZ1J0KHVEC6tpAjh2s6NydzqDj7R6l8VaBsRBJr6RD3Nfl
    G7/nz229jff7/Ys0Z8MfWqXu4K4tI5i1K243923ChdKbt3OXshOZ8nWUeh9UxGKP+VWt24xaN5Zr
    118ft3pICdF581iK8JxKbtYCIxdbLyzjFR1CeeTM23sat3b8OeCqHGBG989J1jw8tufsumbWp6Xs
    YvxRP3Bj+q+II9Ix15Z34JZN17Ci1FwTa5vmitZ8nfukqefg5g1Q3aI15ZJD/fjh8F282PdthrbZ
    Eq1pw4KAUohUEZbYBvp0crNeAeNbz9Zz7poIKVl0ZVln5u4+h4e6LQs8FYrJ3b5kQtev2VXZhnbO
    8kYrdQC30nhm7+nIHRdQprt8jGTTMmmmK/fJchyC50FF/cOe705lxNqbeLLXEsZ1/CXa0wdNN1fx
    BhB/Ah4ERoJyhGHYVSCewsULyCxjkrT1sJ5zF3QINS0qZ9dQuicWMbbDCkPXJwgvvZJ8SxJ8UHgs
    k7ZdzPoKy6uQ2kSdZrhyz5R/BvUEMQzZupXG/2wZQaEnmf/t8m2szDCFrrGdXPlfYDRSdsKjxqDE
    SBBnmZNWEStBfQRiQc14QWM9567TIZTqVKiOvd+5aSQbKtJ5sOsykjWPqfsrdCdvHDyBp/PPYEVZ
    8/v9tQkTWjPLdZ8srwf9SRoFJKOPUoKJO4bRylHFHZ38q5ZYBU2xqe4bKfcBTwFPMX68i86dT0IX
    /VH0RoiuQAKIFFBFQAGIHSh9Dcq9gpkzw5ZTaT3nrmnpqNArGnQEs3adx2sHBnHPUd9xdfpqurhK
    /F5f4k3g06K+vHeoH/8pPMaWzbUJjGpGVapT5FCUetkKjr0WpQT3bb2CPokFnN/W2jF4JfCtXV8t
    pPhDzVdUsZ5zVyqs8Y9tlWlM2HYpD267hP7JB+ibfJCjEw/h1h1UKQf57tasLu/Itoo09FBfGWxa
    GM1k5T5pUkcUrxJDaW5/eJVg3KbRfHf8s3QLQvAvWggPFhN2t6JzR6Sb2BA2jI5gdXlHVpd3DPvY
    NvGKWAnsBJUOdKK6MjrVxACdImJWtBEJz9f0XbAkB9yt+MuW4Yab08eAHUmj6oVlLIL1nLu5DdVi
    EHtA7QexF9SemofDDYFvtWnxCP0JcrKea3DsnnsSSU9Pp4oOoKcj6FT9mdTSUXRAU+koUfswaI+U
    TqQ0t6ljJTKnDQdGxNqMQHxU2Je3Dg5kdPpvsTalEUJhKO882ljPuYMGYk2Nw94Dai+I/Qh9D4p9
    aGofVY49JLPPp67H1KmnoWu2c7cxgLar0aEnnqgEdtd8NW/Gj3eBNjcSb8qRYMqOixjRfi0JwlpS
    BV54LdY2+MJ6zj1HnhvS/V7vdoT1Cm9tLIjwtgxdGH906HoDqLip+99Wmca/9p/E7dbKnvk25Sr+
    L9ZG+KL5ecG8vH0gWk6/SpvgcTgar9xbEkL9NdYmmOXx/LPCkUwXNnTBnFjb4I/m59xBgWrZKzIb
    A4jy2i7xLZJMeTowKNZmmGVDeTrflPSItRm1fNfqShbH2gh/NEfnDrA91gbYWB3VslftiutjbUKw
    vHbQEs8kXQnuFcK6GxbN1LmLbbG2wMby+G6t1VIQ6qpYmxAsRvowRBoBj7caTkjyAJGmmTp3e+Vu
    EwjRclfumZndiOPuRzsq28ZU60nAz4lOJsXMAIM0T+eubOduE4iWHJZxxH37sF9KY1Zzla+8XC0u
    pzJWBhileTp3dDssYxOAFrxyV/SPtQmhsro8JsXBJbpiePIotsZicrM0T+eu6fbK3aZpRAuOuQsR
    tyGZWjZXtIv2lAVKcJFVc9p90Tydu9O5AwjciskmXslHhVhBGur98U3ca+IUeKLYjk+wSYPzrL6B
    eiTN07lLWVUtXRAU+4CXgCfDZ5BNeBGPkzcjA00/CcRDIL4ATHaqcbfcsAwY6yJvYQo90fkrKFhc
    4WFw4gisJ2oTAOvJD4QNtQMw2v9xNUIsQfEeLr5BSp2HHmqLM2k8FpRBbfHovAMosrN/BX4F5iBl
    KzycDVyEYgSoAf4HEF5crmAf/vGPEC5LlXkGgRb59PJ9CB5oNRzLSlEGohk7d7EN1Jl+TpYB3yDU
    ezi1hUjZOP46e3YRmfIzUJdF1Ewbk4gNzJRrGx2Wsgz4tOZrIhNlbxxcBFwEXAqqdb2r8+NayTFU
    FIdjbUKo7PekbABaYXwBZ5QS4O+VGnPTrsR37804ofk6d8H2hg93sQXUJ6Deo/jQxzXqfwHG0Bei
    hO3cLYVaaOiyWXIzMA+Yh5RJeMS5VK/qL0LRch07gFCHrFtXaYytFWm/JVVwXUUyN6K4HTgbQuq2
    s0bBCx7BS22GcyBMZsaU5uvc0TeBWIpSH6Dp75GTs8b0EE5tMW71DM363ynO0PV3Td9TLQ1du6oH
    KdPCa1ScoZpDppDYIq6lCngReLF8CT2EzgglGILibAQZAQbYC/yqFJ85NT5KGM6vkbc5uth95QKR
    KT8FdWGszbABEHtx0RUp7UyoUJgsxyHUy7E2IySEupucrKf8nVYfkVJVTm8FnXGQoOukCI0CXafA
    62V7m6s5GE1zY4G9Ig1EdWjGdu6WQC1CzrAde6g4vCvR4zxRTqgm0xLFJZQCK2u+WiRx/hOOArrn
    bbBY65eWihCWlVeNKxyOX0Hsj7UZIXAYh+PnWBthdWznHoi8vL2gloc0hmA7iCVhsigOEe+DGgEi
    hF9IUULRwS/CZ1MLRkodoT6LtRnBIz5r0dlOBrGduxGEejvIOw+g1EScoh8uxoZQWBXHCC+a9yFy
    s5aQK09BiGHAT0EM9IGhDCcbgwhLNnU2hOL1WJsQD9jO3QjK+xamqj5ECULMxiX6kJc1GykrkPIw
    Qp8aOSMty0tkZ/9e3ZcjPyV3xuDqlTy/GB9Gt0My4WTf7iVUV2PHG4VUHW7Bb8HGsZ27EXJzd4H4
    3sCVZcDfcdGXHDkRKYsbnHVqLxLcqjVOEeXgnuHjhCI3awkucSqoa0GsCzCOG5f2n4iY2FKZN89N
    dR1AvPEMjzxSGmsj4gHbuRtGNFU8U4UST6HcfcmdcS9S+l4RSamj1MQwGaQDH4dprN9R4geEWBaW
    sQSPkZvrv5+tlDq5WQtwcTyoW4HNfoz6AikLw2KTze+4xKNAccDrrEMZyv33WBsRL9jO3ShefIVm
    dBALEN7jyJN3k5cXOKael/UJ8EFoxogShLiG3BmXEM7Vl2I3wj0KJxcgxOwQRzuAu9zYGFJ6yM16
    mf17+qPELSA2NTgv1Dsh2mLjCykLQDwcazMMI8TDhn7HbADbuRtnltx6RLbHp2j6KeTKa8nJ2eTv
    Np/o4q8g3MEZIjagcyY5strh7d9zN4gwZD6Ichz6SHJzdyGlhxw5EdRYEMG9AguVw+zZRabumTfP
    TZ58hf27B9Q4+S2gFE7NjrFGiuKDD4MwX70dfbbiZE6sjYgnbOduCrEQ+BhNP53cGcNqVAnNM1Ou
    RfF8EPO/j6f8NGbK1XWH5s1z48JA3LoplEIwjuzsHxoczs16Dc17Dn7DJX7ZjFP7R9Dm1Dp5F8cB
    1/kUdrMJD9UZSHcEv9iIBsIL4uYacTgbg9jyA2YYP95VsxEVOpMmdURL3ACqbeCLlQIxE5eY6rf0
    fqLsi0N9C3QwbYtS08jLyvZ7fuLEdjiS5htXyFRjyc16zbQdNrEjUz4Aam6szfBJoM+njU9s5x5L
    Jk97CCFmNX2RKAX9NnKzAuclZ8pzgU9BJRq2QfEWeTOuhYA6gYLJ0x5EiDyafuP7BZc41dZ/iTsE
    mfJFULfE2pCGiPnkypsJ/Pm0OQI7LBNLErRHG20eNkDsQPP+wZBjB8iVX6MYZzwnX/xIgrgFY784
    irys2Sg1AoR/nWulJtiOPS5RuLgDhZU2rz+k+ODt2I49KByxNqBFs3SplyFD9yEY7ePsV7jEMLKy
    Npoac9nS3/jD+QIY2uR1it0Iz4VkZxeYG//LDZwzdCGaOB/ofMTZD8jLyjE1no11WLpU58LzF6Ho
    BQyKrTFiMS4xhjlz7KrkILGde6ypdsYXAT3qHZ3H/j3XM3duSVBjfrX0S847vzdwou8LRDkO/VJy
    chp3NDLC10sLGHzyS7iSMoCTao7qKHEdy5bmBzWmjTVYutTLV1++w3lDBTCEqIdua/aX1q++i6ef
    tvAmr/WxY+5WIFOeCfo3oFUh9LvIyXox5DGlTMDNh6DOb3hCKZQ2ljwZHn2OTDkeeBLUfHJn3BaW
    MW2sweRpF4J4BUHXKM2YjxB3kiPfi9J8zRp75W4Fli3dyXlD09D0B8jJfj8sYy5d6uXCoe+jczXQ
    vu64Yjp5M54OyxwAy5b+yB+GLgPP2yxbFk/VjjaBWPblFi48/wW8IhnBYCK2Rye8KPEcCWIUWXJF
    ZOZoedgr9+bOlCl9UI7vgA4mMmNsbBoySfZHUw8BY4GE8AwqvMCbCM8McnJCqNOw8YXt3FsCk6YO
    QdNm4RLD7EIQm5DIzOwGrhtB3cDv+y0mEStR+mskaC8j5e6w2mdTh+3cWwpjxjhYsMDuKGUTPjIz
    u4HjPJR2FkIdA/QG0bGmziIJKKj52g6sAvULeD+tVlm1iTT/H23oWXCPZ0SBAAAAAElFTkSuQmCC
    '''\
    .replace('\n', '').replace(' ', '')


_uniklinik_hd_logo = u'''data:image/png;base64,
    iVBORw0KGgoAAAANSUhEUgAAAOgAAAA7CAYAAABxA8b5AAAPs3pUWHRSYXcgcHJvZmlsZSB0eXBl
    IGV4aWYAAHjarZlpkhw5jkb/8xRzBJIguByH4GI2N5jjzwMjVCqp1W09bZNZyoz08HAngW+DVzj/
    8983/Bdf0kYNRVuvo9bIVxll5MmLHj9fn98plvfzfZ0f76Vfjwex7xuZQ8Jv+fxZz/f8yXH9+YFW
    vsft1+Ohre91+vdC3zd+XFD8zpkX+7vI74Ukf46n799h5M+LWf+2ne+/u/LwQ/pd9e9/l0YxtnI9
    ySEfSRLfz/y5k7AKGTL5/fmZsx9RXmcp/BTRf6xf+Kt0fyhgSX+uX1zfM+RnOT4X+rGt+ludvseT
    /rl+r0p/X1HK31Pyzzf8y+xHif9Qv7v7veezu1nA0Sj1u6kfW3yvOJGSFnkfq3w3/imv2/sefPc4
    46Jrm61aiMYfI2UqflNJO81003m/V1osseSTG79zXlnesS4tj7xeU4p/p5tboDNbOv1YdE44nP9a
    S3r3HX4/bta5806cmRMXo8e/foffD/yn379c6F6HeUpezPKpGuvKji+W4Z3zn5xFQ9L91lRffVP4
    /Iq/f3ljhQ7qK3NngzPa5xKm6Se25PVZogZOLfHDl9T29wK+IO7FYpLQgVgBdqoptpxbStSx05/J
    yoF6NjqQNGjerDIXkUpzevZ785mW3rlZ8+cw8kIjVKo0WgN1aFYpWip860BoBhUtqlq1adehs0ot
    VWutrbpOzSatNG21tdbbaLNLL1177a33PvoceQgypmHU0UYfY8zJTWeZXGty/uSAZRMrplatWbdh
    cwGfVZauutrqa6y585aNBIRdd9t9jz1POkDplKOnnnb6GWdesHbllqu33nb7HXf+1bVvV3/t2u+d
    +9ddS9+u5dcoP6/97BqHW/txieRyot4zOpZLouPNOwCgs/cs9lRK9s55z+LIElCpzCrVm7OTd4wO
    lpOy3vRX73527p/2LVDd/2vf8p86F7x1/x+dC966v3XuH/v2h67t+eRWXoOchdQUhRToxwln7LTq
    zcaydzFJB+Xp6FIZ1JrNjn4i29+znjFHpYK3Ta3X1r47Bq1WJqXrl2VWo1RlN3bJVpC3vXg16L2l
    05Pw9lzsW0cpw4paqW3v0Vq6AHLunfT0UkfJYigce6xjWaP816IO4HPnQj7w2bNUZdyS7Pk7JeMD
    d1Oj1Y7pHEXpzcm3lawKxfcYq1GYgYB2v+qC2yWePIe2NlefZyoLk0wdI38F6qhb0lyV02h1tHFH
    uQj8plzTNB7+mmLss1DUdc/hukIxZuYCVWgj/Q0gv4xzhqVIT3355XCprf2e1fjsPW2/4/vyx6qs
    PnP30gfAnDtZpiNaw2Q99MpJVcw5clnWXXOCjjrm5pPcaPld87j0exN8ZPWM2wLuRFFa2bOH028c
    pnL9Ogn8lCpxb7uIYtZxRqoz7yrXmxSvtHrpRm4Tc6w25cwtp1oKgEBnb8AuAdGku7dbxX+ua/me
    ygt6KGVRpV7zNZlnr1P64WL03A4UOV7s3fJrUn2XodONIsGyqf1JSy18ymWh+3Z9iYYJjrz5yNSb
    0yCVSTC94/Ax3lujKe3TpZQVdmazTO/OmqXlc+sx61cXljC6zcYGF3WJVqmRBuol6zT6JHfOxbl2
    bMAP1lm0DvWmL5kXsEiJlvrA3W2XVidgp+WWWs0LZPM5WYPlz94Vx7FZbs3HJYK7V05onI+OJA8g
    NZnRGSArk6VkB4bXNPxSVFoB784p7Ufh2KxutgOMu4F6BwSJY5iBiQJyjhQYOmbYlqgyNF8j9crK
    xmCjSNJlX+gKG94rXxq8zC/altSE7oGmuuJahCKUwHoYE85VJGNTXBx1lZlSN4IM+iKsc2jmzKOp
    73Okwzb4hSz3zjV3XZZ1FRRSfC/qVeOstRtiOPSk2/tkRYYAXoSoUJyD7GSExKZLBYKYlnYnCrqd
    SlDvzareIv7zvKiH3RRFZxbpCDKji86SZtQJnkgZE92bdQvI7mlu2wSt0rCYTtM7DKIfTQXdXZsU
    AeLNdpu1bzsngoTRm1WWNFTjnKS+5RbpKtF9RfuKnN5AAxD0TIkHwwCB4KSLOpMWN7zZfM2oIXRV
    jAJJ52Z7pta5cMiOo/MYBzIyMpoga0VxeSceHABB6nRi4HB44ZqNzAN9Gtxa7Z4bERq1YCdT6EWA
    2TljF0Mv7QJncq52N+ummHWDX4qW475roHJO+oEidF1nXc4IwDHu6WKPPrixjKUFoxNMB1QLb5y9
    yakdIiVd8J0z694Eq4N7lae6qYd5Ae3EM3ANwVgh6oVGZfIDOB6WXp5uTtd9LAUASB8b6nXCAvpE
    Rot4P9Qu9eoGowBaiqUxSkermKeaLJIE1yFFrMuKNGO3B4EdBB5QMUkf1zG6W4CHpAgujJ71K8g6
    MMS+WCTUvEjSGbk4DtHMctiFNBSJHB+PNKyQrsaad/ACx4TRasFo+WTfZIiZ6TVBnryxL8pF/tiQ
    3K1lPCMYqVnbJB6TDcGZRVo17qn5TAaD5noKYZEHd5rRbLqcoTnoIxMeJuepFSKniexCOrN2RTW1
    cAqnzIV3CczFuXcnBKX1dsq9D1mEKBC3K1j2o3GT2w5lmqcBlpNeAQJcx0YLFcAagVKlLXoxX5eV
    sjEEUtXy0t5ITavPmhbx1wYVF4DoZ9c2LRCLNOo5TFBHL3obJ36ZySWFuE4dsMM3F9mCoAtowt7G
    9hMoOZCbsji6glsE3MDf10mcBpUID1vH6u6TJuSGBGDlJZZyYkssv7LVt/FOROR3P2EraDFnxokl
    d8A0RjvZd1dJMOcwzy/hMvhYwvwmskooyqhkJw8BwOh2WQO+Ls+lu1a3d+6+SYCElEbiS6s/OQLC
    +/kgbSP5lduXkYaIedfsYt8nwEvoyBT+8fpGYzNU91Z5EC0oL9ssk3sNXXBhVk+VhMOI+9ZYSJGV
    8BtIR+sWNIxFCiZwQPZGgDxbLzAhik7WiogsBAiFn8hPRy3IK5srkJF411oAgbn2hJOinLUSHdVS
    uugtnuDMGNCWiORJOWa8iBqWRlcIncpw2tDn5jWihgvablzCBRB9Hf3zCkL67yKE7OjyQDUJuMra
    2iT9zbKdlS7BMCaQ0HGBCbiVONoOsRZq3QQPDlcl6jQWAj0O/CSF8KfsYxFYQXEhJoBZImsgSbhC
    RUcDGzJmCIQ5T4cJZbmwz/eYZmSLbIUzDC4WJ3V1l8cl+pBG1+gdUBLnPlkblySIo+voFKUT9sZH
    0P+K/BNCr8BvmEwouKwAIeXH7DHgN2AIfpEd6O6CSAzcWCV1rGk2g25YFxG5Q8SH4vjBNNMsEvjD
    A8LHBDabItbSQJCB0hGEFmLPpE7MIu+Tiw2pgiMSx13wZjJS9E6pM36ZGEVhTj6eGTNhDRbj5qRU
    hN9xEznzOumANKLDwhdJrUIt6P8eSyDrmEpDs1miCl6ks0GB7DHbSGI5duJaWseHOhqOeHCGrFSY
    hi5Txc2gNpGFPTqh1EF9eacWSkh9WX3FsxcgY6QCAu4fL2UzchF3cDLWf4EbmQ3lJOLgaVZiJvpF
    dzVAVFA9ppS4mUIqI5gHwkyvdSbwK4wYuVnm8zNCqeUZHw4MshIrvkEphXnSX0k8NCifJIEwVvX7
    UtJ9bORjTJAwllWnQ1NaqTkifg99qAbJH/QT5ChkdLfiVup6zjgbJ3kCAJBn4DmjJO2nXVr8hmhM
    JQ0T0ziHuBXEbK0Z0R0SPWQidCDYkGiVVLsHYvyb8RW4EvAi4UrLJbEzpA3CNFMTR5dzTdzTKAQ7
    iCSftaILFswwPM0IRrTNEY9bbxaKUWYmMcpw1jAqS2RlpDy+IgIc/sJhBnEUdRP8GWoGVS1ozvHQ
    lAvsQN4NU21kxYlmF8Zbdrn6m9EoNlbDrI9nsUgmIcY9pAtzJogzMJtwreZjmjI0E0xBOHuIiW0R
    JzAd3IykEzA7SXX4Yz5aIhe2Vi4EmFHpz/j5cJQR2+lNux7cPJwyDOA7DKyo9hlEPx/xJ4N1zqWk
    OBgsPEhsV1U6xETBMN8oGXVRF15iUyM3I5ObAABgq48zgWyFZdAvBCQ+ab3Pao0sRfLA0TFkwhJy
    PTBsemXX5sZMkxuUd9NHhh7AYy0kayZsBmrfE5pOEJpgSQoxDBuTKR77kUJsxecSD6WsGuI7gSoa
    tcLEuLJjp1XBYi0hnmRZpABVct75cwDCt8CutQAvXRtQm3cucklZFVi0GZBin/qpEla2cP1OfOBl
    4/Y1plqJctct5src/iimKsl20azoGuH8Pb6/AOsW9pAhSEGGiKlM4cgF58BNBUy0YVcyo8JPJI2o
    jgSBVQqywQnlx0QqBsnkxDyBn+6u5khFoAYpijfox4G52Lq7JOsmM6ONkM4fCezmQcQSmI43kPgT
    hMlIBB9FpsAzMeJNasen0UKlCA7ciDjJNOImg5eRXAk1x8OrRGAc4CUI9Embc5klgQeQzJmpw/9P
    B2mE/fSpcIZoMFCZAzqFEeuBFSKLhxNhFLUdi9OxAJE33pjnZgSW4RRAd/6JZy+iMcPpwBiQbyje
    GznSJjB2GQ71pHFQ0e6U8DALnvZiiPZnZkx3bprELjbFdTaihByaS5BX6lEIaCLBgSWCA3cVkgL6
    ThCIAIVgRZBVXyT6sVB7boceTi8UWcWIooS+TYzno/hWyCwUM4I8TBmHGW7eATYTw+MsQttdWS6B
    FHgxmuAKVG4jtscNNPXMv3SqBeLUoeVwibUC1cQqZk7UpsIL5lTG5ujJMnmiZGSkDYQHBtSMCg6I
    AO5FMxlyk68P4xOup/4M3EddQoLThuE2uerGns9wjRmUfjC3eUifyBwUUH9MclIg/eOt0bUa49PP
    AwGGdrZGVGpldyONeo5ksDyuIcz6BRB1kiLjbQGP5BoL4MMf1RgdIUrX0pYrPo6MuzISiyMM3SaI
    Vx8RQC6p33o9Pm1RTqrtj7o1wNFEsQDEizPlxRqhBiSZq0zlMAZcg3gStERj/lvuiPWbqCbqh9zk
    cJgBuQljGW5BKIStAi4ZwxYDiT+HIJtk5n8C2SV2k9gQvVSpK/PcdsgsgjJOm/KWRXR6T0S4UIGC
    aQMa5oFDTkHf7CCMNAicEOWs+srRBF+MP7yCzdYDClI7WQWqYmwoDsFod7hfWqUWjMLUKj4TMceZ
    j7rLgzB2NSgs7GbusBsYBpDSgqK8+VhWZ2BofFL8MR1G2X2qyg6IhQRNZa3Aeq28I2cY9WHkYRZh
    KmSAJcT2FZl86FZhOGgsHjQRJJB5f5bEDhAOJPASwpkbXghs37F5UYRAbpHOVIVe+YD4ZsAEsNUf
    JZZE3iWhVeYYgjbHjj+7qeoDplscXsT04o/tAv5H/tjxcN3FdDq7W8gnu5OQyi/h/V/8DvHfPPE/
    vhAaczdOHv4X56PX+hWaXGkAAAGEaUNDUElDQyBwcm9maWxlAAB4nH2RPUjDQBzFX1OlKhUHO6g4
    ZKhOFkRFHbUKRagQaoVWHUwu/RCaNCQpLo6Ca8HBj8Wqg4uzrg6ugiD4AeLi6qToIiX+Lym0iPHg
    uB/v7j3u3gFCrcQ0q20U0HTbTCXiYia7IoZe0QkBYUyhX2aWMStJSfiOr3sE+HoX41n+5/4c3WrO
    YkBAJJ5hhmkTrxNPbtoG533iCCvKKvE58YhJFyR+5Lri8RvngssCz4yY6dQccYRYLLSw0sKsaGrE
    E8RRVdMpX8h4rHLe4qyVKqxxT/7CcE5fXuI6zUEksIBFSBChoIINlGAjRqtOioUU7cd9/AOuXyKX
    Qq4NMHLMowwNsusH/4Pf3Vr58TEvKRwH2l8c52MICO0C9arjfB87Tv0ECD4DV3rTX64B05+kV5ta
    9Ajo2QYurpuasgdc7gB9T4Zsyq4UpCnk88D7GX1TFui9BbpWvd4a+zh9ANLUVfIGODgEhguUvebz
    7o7W3v490+jvB2jVcqMlbKyzAAAABmJLR0QA/wD/AP+gvaeTAAAACXBIWXMAAC4jAAAuIwF4pT92
    AAAAB3RJTUUH4wkCDRgj07kTDgAAIABJREFUeNrtnXeclNX59r9nZrayhR2aqAhEBCxBRUFBUVFB
    YEHFFvQXE6NGjYgsKsGusUejIjZsrzV2RYwLoqKiIqIioIACForS2YXtZWbO+8dznZ1nZ2eXIpiY
    7PkwH3ZmnnLOee5y3dd9nzPwSzcToLk1t+a2leqys2/QovfJpLXdA6yNf2gtGOPd2xgb/84QrS5n
    8zuPALb56TS3ZgXdWRduOeACAqnp7m0OcBLQW5pXA9QCmUA5kAusBwqBT9xJ1RtWUj77lean1Nya
    FfTntuzD/4+U3Db+j84COgMLgCLgG+C3wEygHTAReAZ4AngAuAToqteuwB76/HvndYum3ONBZBtr
    fnLNrVlBt6YF8jrQ8rBTHIQ9GOgB7KZrVwIZwOPAmVK2g4DPgZbAMVLc3fT+ZF22A7BSSh4Gluic
    NTYWpXjqvbp8Mwxubs0K2mhrNexSbCwKkAc8BPwNWAiskkKmAb2AV4DjgJiO7QX8VQq5FhgOrAb+
    CLwK9APaAxEgHQgBf5Cy3wNEyr6bS803M5qfYHP7r27B7fKarTrTsv8f3Ns7gS+BR4BUoEzQtSPQ
    HXgM+AE4TPHmYkHeYVLKd+V5o8BrwHJgb+BTYANQIKVfBLwP3AKsSg23X5/epTdV333W7EmbW7MH
    9Y4Okjd4JMZLlZwEtAXeAsYKll4v77kA2Fcx6Md4JNE/gGrgZUHfYcB8KfQfgV2Ai4CFGAPWHqFY
    tBB4ElgGvAAsBfYDjgBuB2qLCsdbndP8RJvb/64HDQ+52Cnn2cAKwdEpwO+BvRRjLpLiHKHP91as
    uVke9FPFm7+R61sI9Nf3KwRldwdOAW7S35uAPwNXABNtNHKPCQQ+AO4AFmd0PXRTaoffUr1sbvMT
    /QVam5OvIpaaRWTtt3Wf7X7eA5TMKdym64Q6H0zb/FGUfTV9y6itZQdaHXcBERMkumF5k8e2O/0m
    aqqriBWt/I+dw7xBF5HZrS8ZXQ8lvUtvqr6d/fM8aDh/tDt8HHAXXprkISnoQHnOm6VgVkq5GLgV
    mAx8KAh8KPARMFTwd6yU70Ep5HvARmCS/v8XME3x62WCuBnAaOA2xbL/BH4qKhz/P688rU4Yi43U
    pmmuozZSU1E87YGEZ1lggCy9LUvJDtva0iIDtNiCTFSYQDAaq63GBEOZMvCVGBMJpmcRrSzNTGL0
    rbiEKoyhdNFMar//FIDWw68gVlOZIp6hKehWmrlbVyp+WmzwUnMBoCwlt41d++xVSYXfBEMZ4i5I
    0p8omMpAahobJt1Gq+Mvw0YjAY2/qVZWVDjehoeMBmNaqB+JLQpUmkDAli9fRNWXb9Z9kbJXX7K7
    9kb9SgcO0VefCF1GIuWbKXn/8bhh2jKsNeT2P9s9t+uB+6Wc7sI/AlcCTwEjgdOBr+RdU3VcB12g
    la8zxUAbYLo6WqXvK9X5lVLSZbpvVxmGkGLVg3T8P4GjgX0keP/TzUZqAa4DSoBpJpSabE5y9f0m
    oG1tWbEj+tbr88Zeh9tYFBMMISNbAozAWqKVpSicSXbeBmAJ1u6TvXdf8gZdBEDR+884NFbS9Mvy
    0xN/dcr1nT5vU1uyoZFILATwfCPXK/U4EDsjVlNlwkMuxkYjAF00H031ZVfpRIqcT7JjioFvbCyW
    m9mhe12fWg443ynnRZrnUuAdvcpElg4KtcitjzK26DkHXwxe0c94kT/dpRxT8dInKCbsDXwtwifi
    U34jOBvS5FYAs/XZj/KS/YBvFbeaBO9eBvykGHUY8HfgMynvZUAF1t6IMSOB1HB+wbxmT1rHmm0p
    YRxr5P1rei6JsrImyT2S3Xe+QpmAjHQ/hUALgVtNMHQlQKx4pf+cEuDFJB48FcDWlCb20W7l2L6S
    U3DyGMbLGhwBTMAERiX03+JlCrKToM2KJNd/D5er9445Sc5kmRxTWU7/swmkZiCZPVHHvq45Mpqf
    gdKpm8P5BVc7GW5SQXOPPscp5wh5sX5iYPsD+cD5eOmTIkHdzo1AJCuvu0mD3KiYc7PPm2b7PHOD
    EAQvD7orcI2Iqf0Fp1phzFAJRX9gQXjoJZGiN+76N/Btv3qSyj27mzXfDaU+UkMglLql60yWF/e3
    C/AKT64AloTzC54ofv+f/u/XKNRJIj0/qzClUPf0t8vEX/wJGJXwXQQYsw3Xvx8vjejaRfKOLYEj
    w/kFLjC/VMoZU3g3NeE6Z2jeP91qDxrMyAYvZ7kMOFWe70TgaeAL4Cq8FMs6wc02jShXrc6rEEY/
    Xl73QCAFr7LIxSK3S3ET2xHypisUt/5R1/xYA35Wg3s5rfVuJ+Yeex6b33l45zBru3Qj3PdkYrXV
    WBtz7LFJCBms9zCM9y+YwsbJt/87nKkNDx2T+NGWLEmKDHP9B5mSxuYvppHb87gt8j8AfiQTzi+Y
    KJT1J7zU2RMdz59AyeLZfuMQxJho4sVqitf9fCLUjccbuvN4lU2Fdg0GlZmbdKwmEGTjv+5046zE
    y1T8ToiwUHHzrTr+EqydWjTlnroLZByYT8Zu3Z7Fxp6tq5jbkoLmHnue+zNf3vMvUqzvpKhj5Tkj
    EsY/4VUFRRMC5qNE6mxQXPmNiKL1Ipaq8AoafsRLx4wErk3wxOW6z+8EURYAl4sgGiMS6Ub9f071
    +pV/CKZlPrWjRT1lz0PJ7t4HsESrKwIySqcDA4QOvhVUC8nbdwP7PZanbaRmfji/oAogVlvNprce
    /CXKFncFMxJrE6UtbQvn9cPa3RIgQSBWU1WYe9Cgiu1JZ5lAEBuLni052R9IL/32iyrfIVnA8CTG
    oyI1b5epP2MOugInYm3Qx4dcqb9PbcSh9MfacAPwUL6pEK+OvN7nNhYlnF/g16nfxeG1AaxzQAD3
    FE25h7xBIzHBlIYowRjCQ0bXKWlSBTWZrQmmZSJo8ABwtbD2TMWfz8mDFkkwL9L3mb5rGsWNu8hS
    PSbXf5Su2UbCfIriygp55Ns1gX64mw6cI2/tqpG+FlTppNdPCrSt+tExPOTi5UVTJuwQSW89/HJi
    NVWAPVXwZY2My0KN44lkjL9i8hQhka5ALJCSVhjOL1iHjbGj+tdI6wLctx3n/b2Rz3+DtT9sT0c2
    /utOJ8Tr9exzsTG/grYHXkpy6lrJ0Pa2E/RKjE//AryPjSUugQziFc8kJckl8/52ITCoDnl4PIlr
    H8jG7eqLzUVkpThep3sS9HJMkx407+gzkQVZKIvSR2zT74S3I/KGKVKkHwWDf5MwCftKee/1Osuf
    BEmL5SUvkWd9U2mTuTpvkjyk8Vm1dXiVRylSkByRVtOAnoK5l+JVLY0GzsMEth/jGkNmr+Gkt+0I
    1hKrqeqlOOFhIYIj8Ar+CzXGHjIgNwITJIj/T8ZtOV6RxXF6WCMAiwk8Hs4vKItFatiUkArZQe1H
    GdNkxMvFTZz3kvqf6FlKtjfWzjtupPszsw5e1oeRm8TIJ/a16GfOwVwxzkF1PFuh0YPARkzgpSTK
    e0cSkigig5zYjtLLtSpglhS1XJ+5kK1twrkHyHAnzk3jEDez13AEie6TQH6qDgwTtL1bAzU+yNBK
    sKVtAgv7lhgzKyvYRveMyTIeLsXLlSUpkeK10ANzktBRSrmn4Jn13aOdDEMfQfBTRBZlAm3D+aPX
    FRXes81PNbPXcNLb7AHWZsr7X6NU0rdSykIpapaIrnNlIK7xzcuFeOmmAXgFHAWKo6MiFx4D7gsE
    U+a2Ov4yNr7+jx2toN8Df20QT1mbuwUFvQvfsj8/TC35ZpZLF2x1C7TdCxNKAThSz3YjUG4CIZfi
    cJ51dEKIBBhqSzf+nDl4Gy93729uwcazcg6lCWHZ5UnHkZI0MrhQrHcfOa8Y0DeUlce6F65zcekG
    Xbc90DHnyD8uV3gzjHjeOFfhX3qTCpretiNSpn9IIfLlCe5XJ8bJIzo4kCMFlYXl0ASipI1we18J
    9/F4xQbzBEv3lAGYrOvenWCiQ4K/WbJIm0RJL9c9v5VyjAMeFXuYJsh9B5ix2/pEc446i1CLlsgr
    Z8lQvSsFq5WhCvk8fT95c39bjbca53Ep4l1CDa4/p6uPP2DMfTYauSicX8BOSBGZojfurufytiJf
    bJrsx5YVtO76vlgrS6kFx6xGN7x2mz92I5FY2kEt4L+uCm42+mRrd4VL29uPImA1xryKtfPlqF6L
    lBadmHvMuWye/qhjqH+Uo3kxlJV3iDEGiyn1JN0iI88WPajaQLz61+ckaF8Jmg5IOC6KtxQsR1Ci
    i6Cue0hHyoumS2nSBAUdlf2I+lAuJd8gwW0rKGt07EZB6rWCZ5/7LE9AynmihL+z0j9PC2oQHnoJ
    W5t2URWKo8vn4uXvhkvxr/JZ3d3lqa1ikJP0qlWf0gV5J2mcGfKil8oQ9ZHH+FoW/e/ADeGhY8pL
    vnqfyPJ/a9nigeH8gswkgv45mOKtgLi7A0eF8wsCGndPeZocydPoJERTBtA/nF9gkyj7O0kUpg/W
    lico+HIZbH/bTemOkA/e+09aluR+PcP5BXlJPv+wqHB8dTJkEYvUYALBMxQWHo8x3YLpWYvjp9pe
    cki9gc9sLHqDDD2ShUslM3ZLChpUfPUuMEOpkDS85WNTRdD4W63ceiSB2ImJpd0gPP+QrjVP13Os
    8MeCjnvq2j+JgDrD11l3j1qS5lktYLrIWHTXQ1otiPd3bGzcVjgawkPqijLOkXcbJCU9Gq9UsUL9
    /EhjuFzC+IgP2iawqFwgWPcFcIOMz4mK6Sf4SAKvhNLa63P2O7Kk6OcpaGKxxxY9XcL7+xs5vj/Y
    95u4tvvsTL2SxYN9sLbal0owPqV+eyv66f6enOS4O8Ts+487Xa9k7f8SUi1G8j+nkeM7yBNC/UIc
    iqfeSzi/YJGQ52XKNKTnDR4VrVq/nIx2ndfbWPRwGfeDfWgi0RsP8huvegracsD5Tpsv12Q9pr83
    SdDWJiiokafZTd6zHfWrU/LkHWKCOOmyoucLb8/10irmKbBZsm7t5QWtb8JGAK3lzTJ07VfjDHUM
    EwjeL7ZuCl6ZVw7eUrjVQIrJbF1rKzY0nixr390p53l49b9rZA3vFat8qSDraP3/sIxYB5Ff82K1
    1S5O2VdCfo/gvBUiGKoY+f80b468iIrhO16e/+TcY/4c2Tz9ke1V0OUSsgWNMDoRvGqsmM+o1iru
    TG/knHTJgWtfCpb5k5TzJfA2AWXNFkJYH62uSMxPrxUiaqxIJZREyVs2AmX9DPNijSeZR/5R8rzR
    BOvi4CqfR0uaZfPBUKs+r5UDqoPF4fyCsXrOxUC+CQRfr/j0VSqAvMEXf2oCgbDi/3OEGhFCfAq4
    09oYxT5mP+Tz064cqYsEJ18PsLMG/lwStisk5SmTt5mqWCymiZgtQS/DW9UySjHZPOA04A2PMbTd
    9AA/V6CdowmM+vJVxfrsMbyi+kQ6sdLnaf8qSPo3safpeUefWVv0xt2NJelc8r2PBPtJebc3FFef
    oPTP8xrDegnEq/J8dwL3BVLSLtIVByuu3k3zcYMEZpLm6Eax2ivlQe9S7F6lez0YTG/xl+0ptohW
    lhLMyH5IiIX0dp2SMdRlWNvbyWvV2mWkt+tUirX9t3T9WE0lgZR0MOasOgMZjYAxmEDwj02du3nu
    20RXLUz8+DW9Gm02FovnCk1gSNPHRr2+mMAWUVN6u06se+8ZWvzmQILpLVaIJ9ki4DCBYMTGoq5k
    j5KvZ8YVKiuPSFnx3v73rhVPnUCetyJsAsZMwFoRQrYKDNGqchKNskkgD5DHuEYQ4jkJ1imyuqPl
    xSZIIT6SBf0E2Ecw5WBBzOXyLmvwqihulPfbC2/h9RIJ8gG6T6byrgNlIUfIo06S9d5DFsdtkxLV
    /f+KMf2wtoeg6GPycM/J6kUFNRut0W1xyMmkte6Qohh2meLMKzXGFBmaQ6SkaD720DwcLlJtlvrc
    XUrcTx62WkbnAhmpD3SNk9SvWXiL2d+Vgo6SYTkZeKTsh/nULHqP5va/2RJXs2RLuVqKEFmq2Otb
    CSKKT9vIiw2Q8owGhsjdt5IX6+bzgCmCqJ3E6B4r0iCNeL50oMgo16fW8mZXC152F8E0SrBwhixe
    D1kiV/nSQkp8v8iAzYLNSS1iRs9hpLXugAzIa/L2BxBfjfOilNEppyO57tL4vtI8jdX4b1A+72IR
    YV1FlA3QmP7ig+hvyXNOEey6WzHMsYKTGdldDmqW0mYFrWtHSOiuF3w9Q8r4krztgfIIHX3KGVbA
    myM4OFDx636+61YL6h2BV0X0/6QIC6XoB4po6epT6jP0/3tSnhlS8Oul3IPlpWfi1eVGRAodL2a0
    UkoQEfWdDBCR0X5PNAaX97td4xgsmHiD4kIEP5cJ/p8rg1CkGDRCfHuXE0SS3C6o/bDm6g28BLkL
    MpaILOokj+lSTy8KZTxgo9Ft3Ox7G1bcBZXXC6bW/7/BJYP+ZOB20lbxMZiM8I7KIDX5bd9pdot9
    aXTMWzNvv0ALJRBEXaQIs/GK0m8R9CyRF2wl1nUDXmJ1tZjNbHmRXeRRDsWr2TU+Cv14webBOvZR
    eerT8YqLTxVx4aRhps/zdVU/lijO/FCeOCYv/Lwg6b+kPM/hFVYMlWfaK9ngU/c+yhmpvyk2vgGv
    QOMUxa734C1Ad4o7WX35M24XCWuXYoyLPQfpnlf5WOjVYvVaEK/jvFyM4ziFEL/XfWO6tlugcB/Y
    QHjwqFjR1PvARrfikVoXqpiiwvFJJdRjqwMmmJFl1798E+FBIwFrAGujtRS/WZ/EbX3iWGI1VQYg
    mJ5l179yE1q0bKyNWWMCASDWps9wFl99ZNJetRx4gVsF4xXEe8UwTZUl2ZJFM8nZ57DGNNG6mNPb
    5bFemAaQ9s2EMaFwfkHEkTuOoAoPHgXGmPLlC21O90OIVpaZrZpYR+glFLQnCROD2Fh0R5Rx1pmS
    QFoLx2RlKTY6R1AxHS9PmaIUQwsJz3pB0OnymDPkARbgVR+5wClT1+2n67wnT7JKMeSVYnrfE4Ma
    0UQsVt5wkjzKLL1iYgvXCpJm+OLFWinBbMFM551ykg0+6zcHOKb5QSnXKcTrak8R0+oU6nF52ct9
    TGimdsi/SAjgNvXxUPX/NpFfz8nz5wuWHyNI/LDk7BoRW64AP0cG7ltgorffUnSrHqgE5Aria3WT
    eZBdgVXRyjK/rK8FRtcr4K4jhqocupoXrSpzRNM+wBxjAiHxDyvXz5qU2mDljJMvTzmfB+5WGmGA
    nMBcyY3/9T3QKWffw5HMrVeayn3/pebmUhMImvCQ0WT1Oc2N/Vg5ic8lU1/KIZwZTMt0hgXg8xYd
    9+2uOTjfd+25cgRLEu43UlzBaoxp0fa06xqb++uAtzEBs0M9qKrpjZRmqaz6nvKO1YKkG4kvxnbL
    q4K6TlDvA773h2mA6YoF0yQIK0QgIcGcqGNL5Em7+K6V4rteUArq7h8SJG4lJXbGI1cKt0bwO6sJ
    eNRHxM2uoronabyb8UrAWgsyf604/BQp4xxgvjYru08w+WopY67Y2me93CGTFbteoTk4Un1+Bcwe
    mo+pIpE2y9gN1HzPA1rnDb54Q/HUrbbIWY0ZJZ9hTqwLDeMVb78bzi/4Kgmhlkr95YQhzY2RUa4A
    DsfaBoXm2lIkrLnr4UvbFCveT9rKln5OVpeDgpqzAxO+dgsmMjDmptRWu4G1LpfaXyGCa3vJcK7B
    mLd9HEdIz2+iZNC1a9S3ZAsNJgOvR8o3H9NywPlsevshQh0PJHf/Y7DRSF8Z4gwTCNodq6A+UyqP
    1kLCcZiUoq28wvVSCq13ZA8pXVDEUMyXC50v4qNAwr5EVu1tebj+8tT3SQCWK26sVNzp+tNWryBe
    JYbLb1nFuqcr1TFe90lXLHi8+vYd8cL+hKEyRDD2HcWK5yqG/EjfPyHYeojGd4i8aD6QJYIqW/cb
    K4HNAgYWFY4vkWUdKU9v8ArmKxS3Po21JSWLZpKz7+HvCEYXCjl8LiTwNdDRBAIbdnK4Uyqm+Uvi
    W6JsS7tKZFeD/YWUZxygeyxK/L7JsjofSVZvfenQMcVY210cwE3yyg8qdHg/4ZpLpZB+T2eSXjf+
    nUnWt3B+wQUi8AYFUjPeBMg9YAA2UuOU94yiwvFVO4skCshSXav/R2pSA7Io5XirvmOCPK0Eby6V
    0K5S3nS0LPg+gjWZUupuirumywA8IlLkIg26CC+Jf4U8yWDBxT/Jm58tFrSXvGxPecwDFcemSkB+
    lNW9TUYiU54/0YOGdezZ6tM4KeYQpXLyZfXBK/c7TobgUFnlGl23h0/rz8Mr3m8Zzi8YHc4vuEKe
    Z5yM0V0+iFySu28/omvc+mGmC8qtl8Knqn97/gJ8hEtpTQQmJdbINtUi3p5G0zX+bl69a4N2Fw2X
    fW0/PeTB1PL6RhcjAnNnt15CPG3D+QVGyjlV6bTndgpJJFdfIY85WUL/lhQpRQJ/vKBuD73ektdq
    KZi2h4ii4+UN1ogB7aPzS6WkN4ro6SljkCsvFJTnSFHsdpLiy5cFC1eIHQ0JjvbV38PkKaco3RLD
    KzZwVTo5iWREeo+BKL3ziAxKoYzQUllhJ7R/wivPuhxvedwfxAr3liE5V8TOZLylb8fq3EfwKquM
    7nOU+u+W5MUAfrjjFP/8f6n5OVdethXx5W2/FGn4FxmnG8L5BdfGaqu3eFLJjCcJ5xcUaw7uBDM0
    TkiNRgatJTBLC7frwZhkxsAEQ5QuTl7Y0/LY813xQuu65+rN3yjFr9Xh/IKbhfi27KW3oalaaKlC
    s4/lKPIV/gzf0dve1HnQkgUfIEt/nwT/dXnA1cR/hWyRPMwiMZx/kbC+KIUaJijzrB5WQErmKpIO
    lKCWSlA3StDb6fP2uH2GvLjwaDGzPeQJX1CfThfUfkKetkrHp+pcN0u/V0qmMjFdkLnHPvji6zZ4
    RQZFupert5wk2HSp3k+QhVym+18nRDFFMdEazdPJum4P9fdapZSukRE7OJGZrPxpiYOZfWUInlTa
    Zl+2vB3kDmuCo8PU1/0bWWLVGAk0QsLawe3cJ0JmNPAMNlbrtgbRM9pDhvbZhNfLNhrJ8C1rM0Ba
    OL8gL5xfkBdIywhrXpeIgUdVYsuENtqL7CmSo+gdzi8IhvPH7Jg58gzMHUJ994v07AlUbc/Sxq1S
    0MiKL5Fg7i6G7DjFjjGf4D+sCcxQCuYoKfJCTcZEwZjjZTV/0CRaWbZ58ohTRQZ9ICWZI+XbqJhr
    nmIgI+XLVmxaIwi5j86/SgI9QgajEq98DpE+h6jP30r6fLNs0ZhK8AoKLpfXOgkoKSoc77e6T8vw
    RHTNbwXpxstgddc8FQgp3KKxtBC0dbtjvQC2k8ZTL06pnFvoINvBmtffy5s/LCj+i2wpqnWbS9WH
    OSSve23QNvzrbpdf/BI4UyuCHIs/Arg4kN4iMcZYp3m/LuF1JVCt491+vVWSsSI5jU+98MPc6p5T
    UeF4G8zI/h5rB2iOe8tQzvZCB5vu28pnu1vx1HspevsxJHtnApebYGhp8YzndvjziMegsVoEOwco
    B/mhPFVXwd6oPMpkKUiZ2LuWeqVrIt+QEvn3tHH/l/uE90p51FtkfRI3PD4br7jhYF23hVjOM0To
    HCJP/rH66vbKfRmviidX8eMFUvjG2M6+UsAH5fF6gq1b4VDpxYezfbnUWsU5F+k1VgZqs8a1XEZr
    Dl6Bh1Us/wxQrJl4T157r3B+gQfxvAIAK3b7Pc31k4p5I9vzcFsed+F2CUWsugIxyXcSL03cwkm1
    VP60GIUEF3pEziUIYbwJVBfPalByWy3eYmnCa4kJhmIbvV3/rGTNyDG0lDHsBEyrS/uorX/5RkoW
    zcQEQ1EZ0mtkNINAgbby+fnN2wa0RgTkcyYYwpat3Skxh7+59WqzBMuWywM8Ke+QKc+0XgrxiTp5
    kSbNymP2kMDahIeRrVivJfGywktkCC5IOH6GIGypFPNuefI/S/B/ErPbRt4zJAX5XkJ+qKDmahou
    katLZiveelNKXO0hAM9ZBXfdl9T4b546Y3avXj8KSq1Qf66T8rq88T6Cdn7lugdjztSY3hUh1hq4
    SAbSeYw8zYuD3ddutXLVVBJIzVgHHJNse8zwkItdbnFzY9fY9M7Dbl3sOM3h7TJ8TbbKuYVk7Frw
    hcZ8KDY2G6/K6wSsJbr660bjuq2M/6q08dr/AT+YUEp2MN1TuNxjz/P20VIRwcZln/uZ1xr14wkR
    hzsUdLA1G8D/fJKo7h4LJbS34tW9PqO49GtrY5uNCXTHW8/4ozxLK8Voh+uc+YIjmxM63lkE0NeC
    fI8rfbFAsVaxLKVrXxDfjMmtw5wsBq2jzjkM2M3Goi+ZQBCxvuNkNfOVYxvnETtJEWKNPH0/xbuz
    8W1OnNvzOLCxkLz4Mn3sxupgNEWF478L5xf8TZ/d4rt+BhAxgQA2Frtf4+kvDztX3vdBje073wNf
    K/j3qJRzq7f+2/T2Q4TzCyYJfmeG8wsqTCiFmuI1pOS0dtD+9x7qaBw1F795n8tf9pDBydkGuZqg
    ZzxccdqHjVXeeAo0JglJFKT4gxcaI2leBS60kdoXwZzWcuBfCKSkApyBMS+F8wtqMQHKlnxKdrfe
    jlDqIrn+VbWAL/LFq9hisQTrfrGpj0lZepjkNaEV8gI9pbTl8kQ2Ib+6TiTKbYKiy6TcneW93pSy
    uJYimNhfHnu9YrHrBY/TkxibFI3pXBmIx6XUZZWrljSW+/sKr5Lkfbza28W4bSm94o2LNaY3BJPW
    CH6ehFeaSKhjT1zMmhC7jgLus7GYm7gWgtX5MibXygvbrMNc6TFr5SEu0z0uSzKfW2orNEfrgb1s
    pIaU7FZuf6W/an5O2vRJPcjZ4PraI6lSBvl8GuwX1LCVfTcX8RSdiW+a1lgLes/Mtkx45dloJNBm
    UPJ9rAMeTB0qMivfI7JMEG8VUzHwW2wslLXXwdhYLKgw5nm8Us5Gx5vwnf3PUlBg87zpKOZ4WhPw
    RyldQNbfNOLevxFM4ZrjAAARxUlEQVQj56pF9hSMjRDfaWGG2FqX0jhFBMF+InZWScmq5dmsPPFz
    8iAzRcaco/Oq4hmigPOec3TuMTq2TN6uRiQM9dJK9XeMe0v/TwGy0nvUbc7cUv13hfxLNe6XFZf2
    y9nvCFwsWRdTesKzi5RltJT5PSn444LAP8j4tU1t2dZ53Hmaf5dHnkDy3eSaohmxNnar4vUXwJTL
    GP0o9nw3IBrbWG8HzW8aCKUJEkjNsDJy4xS+uFaluah3jn5UeY3ynkFgak3R6qTZGc3nLOK/UeJe
    s4E99JsxNYmeb8Ort7j7HyqlC2ECUYwZiLfz5N169u73Z67XsV8pZ4nG0lgOqZit303wJ6DGmMDO
    V9DIyvkuVvxYnvFICVOqCIN+SSzLJrGpGwRf3UqMiDzBOcTL4CoEAVdKAPsS/72PyXVxmgeVXYom
    T945V33LbUAoecq2hyarVA/pPhFeYzCmwYoQpTU263qHybNcpVxm68wOdWtu56jfjnE5V9C5nZS3
    rZhWB7vPV0x+mmDmPvIojyv/e6S8aJUE9GhgmaBnlgzZSXiFIIdJUMq2KVc3ZYKDspM1dy0U84e9
    WN+sqhf3ecceHKkoqSz7bp4/58KGSbe6728HjvFttL0EOMYEgg0JLO/4y0QCFpfNeiFZKud94LcK
    HxJfXYFlNZvWutRJr8Q9jLTT33zgoEBKWqTojbuIlG1Cxv9Y4rt3ZOOVTc6OVVdSPO0Bf2XT0ng9
    cty4SXae9eVrG0lJ1SI0tKpo9us7h1VvGA8UoAmyxKt1NgqujJCHfVLW+C15gX0kWFcKDoYE1fRL
    UtwmGPyKyJSNsnK9JPz9pCBBQescefBcndeF+C+hDRNR5LZTOVzQ9E3B6Bx53DIZgys3ffwqseIV
    DYYezh/dUjA0pH4dJKs7kvjGUnsK5r0mAs3IOw+Qtd9PLPIIHTdREDmqOXHbmqwSudRBaaZuuu+L
    Ukg0lm8k2HvKS29QSuml5h+F+t9rDfxy9cZViAk9RZboa3mRrhLIc32KHZFQ9cJLOHcTZLO6dpR4
    3W4e8d9z+buU6DNBpyJZvgG+c4w85hzBvk8EGcfLKjqTmu3LS+bJKy9UPnYS1iZRzroQZJMMQDsZ
    nOOkhJuBNvopuGVCAY/hVUQ57+AWcg8TU9zRx9j+TnOyHq/a6CPNUy3GfI9XfIAQxggRXE5BDxbs
    m6i4qQMNd6prbv+rClr+yYtO8d4VDDlenqlUCvQx8XWNIbGwbX2KlScBTyy2Pgsvib1BHm+zlPO3
    eFUZOSTfZTAmuJIpT3quYiu3/1BYivAQXm70ennCzsBnjbKH8c2cpwlW7463mPwMxX1HR8o3ozHN
    lKK6TbjmSDm/0LG7CrJWCcp3xlVQWRsTwvha8M9tJXqr4tMaQal95IF7Et+7aawM5Yq6fXma2/9U
    CyUT3NrSjaRkt5otaHqW4GiplORe4j/Ma6VYz+IljodJSGsS4HNEnq6NSJiVYjCzpUw/yYtVJGEK
    HcwtkSf6XpByTxmY7+Wxpim4X4+XQ3y2Ln2U7AeKrNs00H4iyJqj1MA4xX8dAGLVlQTSMtym1Vaw
    /ikRUcN1n9/KWBwtJX9d/TEEgmBjB+DV5qI5vFnjL6nZtJbUlu3AS/CPFelxrbzsGhmljVu71Mz/
    ozyNQWJtXG1dPFj8ZnxVlW+9JJqTGkdSNXE9fCw1CnMo/2E+1Yvew0uDNFEyaIwr1fP/krt2a9w6
    8iXJipS655xopOu+9933V+NBsZbSD54mWlWOSI0r5DXL5R39q+BjIkrOkFB/LWgZUcrD/R6GW2Sd
    JnJhrTzfJfp8oRTOv+Vbqs4dpvt2kZHoRP3kv1uTer3Yz1Ol0AuKCsc3+eth5T/Mc4zd6XgrWD4T
    W+3ivzsCaRmoYHy8xnGPUiO/wasDXS1WNEcQfK3mLQL8ARtLxyvccGP7J3BZ1foVJTXFa0htuYtD
    Fy+o/xOESNwqFm+T561kCaWcV9cZqMbTCB8Cg1xJXu6xf/YE15gcMbDl6sN6oYejw/kFtPKqgxKF
    fSDx385ZI47h7Bad9w+Gh45xytlJIcXmhFch1nYI5xfQovfJTjlnAO2knA+KT3DHFydcp1YsOeCt
    PVW4sRk4L9nPCEqWqrE29J/uQRt96tr+r1yw9B3B0Ld8njFNVv5TQcASebtyWdw/eIxhHXN7gAiQ
    9FhNZaUUIiJy5wulGN70sbdj9XehoGWe4uCO1F9iFMKrEDlBMeJTwIXaQrTJVu3tlmcVF1dLyd8T
    SjgKr0jjrEBKmvuJ93fkSe8SqvhBCrW3DM5KedDDdIuReAsKHpbXTFHoEKn49FVSw+0B65bChU0w
    5UyhgScEk8/GmEjRtIe29WcKM0myLjOhZRP/STyC3o4arsIoJoPTWq9BgtxfWBszeYO9Qnhtk3O3
    4vGDdWwrIYoCYA3WpvpkLVfPsb1eu4gwWwEMS2uzR1138LZGQcrX1nfOB5JF9z5Xc+5niEfg5bPv
    bCRsCrATq392LsRNoMNNMFSogY4Tk9nJ5z076kF0klKWysJlKibcTx54f5E3QWBEIDXjKynbPOL7
    ELnVDScp7rpdQhYRNF4uT/WuvFgJ8dzdcl+ceyGYyIZJt24ViV3x02Iyd+tWInJogax1NfF6483A
    /sGM7Pk1RatJa9PB2mjkXKVS3Ar6V0jyY0NqB+PVJt8kL3wJQKthl2BjsVSx1hcCB9po7akaU7kI
    twJjAhCt2qlCoIqh3fHymocDMwUNo/KS3+hZvwe8YAKh03ROUGmlnsCKomkPQ6SCcH7BKqGGY2m4
    IXWsqHB8BWifopT0++SlHwPbPjHEKSocX+M4D81ZVHJb0QSrPU7OYwkwMJxfMOXXyoA3aUWKpz3o
    UjGXEt8s2nnQ5VKsjwVvahS37a3rur2FnpaALhGsminvdC1eFY9jNL+RAr5EvMghJsV3semDup9j
    bf1eYoiMw4bKVUu3cviWqnlTqd6wEhmSM/AS2ver35fhVQ0dBwxNDbd30H8tXqH+UHnGbCnZnyWw
    5wk9nKf49mkhCuepsbFYd0HhUfLYZ0mo39UrE6gqXfrZThcCeZ1xeKm0mUWF4+vFbV6oYMGr6hoO
    9NR7t054rYLGuuNVEPCOjUYarcjZ9NZEFztOkRFO3+oHlywW9taenqz+rNbzuPi/iySq/+QoKhxv
    ZS0vl1I9p/Py5PFW6MGul/crkmeZJ6UaJM97qxT0cL1/WRDxI8GcYyTot0jhA1LeA3X8cLxEfgcp
    u9vTKCoPnQZMilaWUjn3jW2ahPJPJ5HmFZFP0dieFFnzjCDs3Rrr3wMpaVf5WO531e+zZEhexqs3
    Xi/4dZm88HSN3W9MegglvIRXBZWltEuqYN2lsdoqar75YLv1bgu7IiQK+dmegUxen1s05R5vdYqN
    zQWG2lj0CxuLVZlAYJZCk9PCgy5Y4wxQpKyYkhkPNCRtEppgcgcfubhdLePAoY7cGgc8oPrnlxTy
    5ATCHUtiRcv/yxTUPZw3HyJvwLmAvQEvR1chZQ3Ww4ueMFt5DLeR1gDFEK/pAXyMV5RwoizmQMHE
    TcR/wq+TiII9pQyWeC0sPhKqrTxWOjCptrSI0g+e2g5RjlG5aikZu+71mdJK7aRgd+r/jbrX1YrD
    R8v7I+WbTnyDs04SijESloiPV2uPt671BAllia55guKxh4ApJpgyxEaq2fTWxO1WTrHjDzUWfvtC
    FccnpAAVTa06NcEgNhJ7USiD4qkTCOcX9FdMuAyvGOMz4MpQVt6iPm9aZg0yTbG+rr0KvG4Cwdot
    Ve80qqC77QXW7oGXf+69ad50cnv0j+pZPJp7yAmnbcOma78OiFv3tGtKKSqso6MPE7t3k5Tue0HP
    N/T3RAndNKUWbpCXXSnP9Kmgx+26Tj8v/uBxHfeUhHWUlGU2dXvEskDQ+jZ55Gel4LNtLErph09v
    90RUzpviYNzrYo7vV/y5ifhqjlvk/RwBdAXxLU5iEvzF8ojfULdfkR2rMKGb5sOxjjdrPOfKC90O
    nGqjtfzMPVXdYuiPGnlNp375YCgeUpgtXJaKuGE3pGSHa8VDZAlJrBZjvXzxvWP2SlBGq3lZrJer
    5Z0O/C6YmbP9I/bg9UTNIdGV85WCZiIw3AQC5A3+9aHdbaKZtdQnhldEPwpvd74XiO/5k650S4bY
    vAd1D7eDn9uYqlTQdJoe7FzB28cFdwN4yf8D9H+WPnMpm2x5qFHAxlht1c/xNvF82dQJ5PY/h2BG
    1qNSml2kOEcqhsxSGqeVhOszsa03q28lEnSjPrbAK/b4p77foBDhCSlIjqDuGTJqV2BMtGTBhzvi
    2f5IfEf8ZO16P8rXM0sH23jdrxerHkndrhWWtc/XLVUtz+1/ztvBzOy3xVzfoFCktRCI0/CTqZ9H
    /xqojtVWs+6F69iWzcrqPPLQMWBtjgzjnxK89OdCO6NNIHDPf7WCekp6D6EOPcj57VH3Ysy+Yib/
    poewWQ/7JP09BvgCY2Zg7XWCXG+L9ZtD/LdaVsnDfIiNVWACR+LVxtYAL5YvX1DWouN+rr+HyAOf
    5SnVvduagmhSSTe/+6iDYY8qbXOmlG1PxdpPE9/C81O9HtJnlRK4rlLmqOLKDiKwXhAx1E3xtYOg
    twEjsXaH/sL0FgoLEl3lFGAI1r6Y7Jy840Y6pN4fOAsToP2Zt1NdvJpIaRElM55k83uPAZB79DkE
    M7KvFcI4UOkp1xbucEbVQz5u+dHCJGNLk3w9mHvMuTW/JgXdjlyQJbJyPsUfvugm4xKlEp6QZfwC
    r6KmQDHj3lgblJC3Erw7BS939mfiGyCfALTDBFx97vEikZBynixDsAQ4q2rdco9ptDunBK6meA2K
    t1+RF78UbwF1QP3PktJlyGMYwe19Fa8eLba5l8YihtLMF4KYKPKpG8aMxMZoalHzL9BulafPEBta
    17L6/A4TSsHGYuOEYKbUlmygumgVWNvC/xN7AJvffSwx3v0l2vMKlzonee3qMgzB9Kz/bg9ap6al
    qykqHE/L4y6MBEKp0xRHDNHrHLFy/5BQP4ZXMZSniWpDvKA+QxP4rTzqSYLGNyhmdaUrb0hZKPt+
    HjVfv79TJ6Zs1osEWu5O3uGnVthY9Gof6/qCjMQfBE0/FoT6RsTXQrxig25CE3tJcF7z4K2twUsz
    tVIaptZGo/w7CQwbqcGEUj8SalmOMSeF8ws+SjjsDrHSuwOkZIfR30uAv4bzCxJ3Yb9KyjxnZ/Zd
    ZYt9ZfTmBNNbVK5/5eZkiOFyefQRPngdaQRSN/q7Nr8aBXVt0zSPSs/qOyKSmrfL6yJZTsfLP23E
    KyrohVd9Y6Scw/AqhGqJ/1T9OXrgjs3MFFv4jAgPD7I1Vlu7w6U2Rqx4BW6byHB+wVdin4fLyPyA
    l/O9RmM+mfgCgVy8CqyQvGQbvG1Gfi+GeBxAzaZ1lM18dmf0Pgikpu17DNULpzd2jNt9guJpD7ga
    3Fv0LF4UQvgELyfdC2/701SgtqhwPHnHjcSEUn7ESzM9gpfL/VDP9DApbnviW8hsy66EaU0cn5GQ
    PbCa45eABsrpa48rlHDFvikkz6cuIp6b//UraJ3H+fh5MAFCe+xP9t59n/P9CM+umtAT8CqL2ukh
    dtAkZYtYWSzS5Vt5pIQCbeNn637xVjRlgsXGCA+5eJLqYg1ePW6eoHdHCeOD8hx5ir/maHwrga8C
    oVQ2vP6PnTYOG63FBFPGAWR16dmYgpo6IdRCaAevw0NGP48xz0sZD5JnWpAY0xZPu1/HXzwLE3A/
    NXmwlP5zIFayYAaR5XNdsbx+7W5LjskCphdAbcnGet9UrVtOWusOgz2vX2+Ttd7+sSQlN4eOWaeV
    ROAVMmyJrv6P8KA7da/VXc78uxfLWeqNt0WHfShfucj4noivRwZjDBU/LtnmgoNfLHDP60BevxHY
    WKS+UFjr9l9J2NPGgIHqjT9RPuulX+zZm5xdsSWrGnx+qrW80rYLsfXfJT0v8+ATSd/lNzIihty9
    +/DDP05r8l6tThhbpzQmEGDjv+76RcZ41GeWD4Y0PpbEdqO1XNfE2AOtOrP3NZNZWNCD5vbf3kyg
    eQ6a289q/x8lAZxfn+xeLgAAAABJRU5ErkJggg==
'''.replace('\n', '').replace(' ', '')

_default_css = u'''
                @media screen and (min-width: 600px) {
                    html {
                        font-size: 16px;
                    }
                }
                @media screen and (min-width: 900px) {
                    html {
                        font-size: 17px;
                    }
                }
                
                @media screen and (min-width: 1200px) {
                    html {
                        font-size: 18px;
                    }
                }
                %s
                .code {
                    font-family: monospace;
                }
                .box {
                    border-style: solid;
                    border-width: 1px 1px 1px 5px;
                    margin: 0.75em 2em 3em 2em;
                    padding: 12px 1.75em 12px 2.75em;
                    color: #444444;
                }
                .quotebox {
                    border-color: #6FA6A9;
                    background-color: #E4ECEC;
                }
                .codebox {
                    border-color: #A9C98B;
                    background-color: #EFF4E9;
                    font-family: monospace;
                }
                .yellowbox {
                    border-color: #FDD73F;
                    background-color: #FFF8DD;
                }
                .small {
                    font-size: small;
                }
                body {
                    margin: 6ex 23.61%% 18ex 14.59%%;
                    width: 61.8%%;
                    /* color: #646567; old gray, changed to darker one */
                    color: #454447;
                    font-family: "HelveticaNeueLT Pro", "Helvetica Neue", Helvetica, Arial, sans-serif;
                }
                a:link, a:active, a:visited {
                    /* color: #646567; old gray, changed to darker one */
                    color: #454447;
                    text-decoration: underline;
                    text-decoration-style: dotted;
                }
                h1 {
                    color: #6FA6A9;
                }
                h2.omnipath, a.omnipath:link, a.omnipath:active, a.omnipath:visited {
                    color: #6EA945;
                    text-decoration: none;
                }
                h2.base, a.base:link, a.base:active, a.base:visited {
                    /* color: #646567; old gray, changed to darker one */
                    color: #454447;
                    text-decoration: none;
                }
                .align-middle {
                    display: table-cell;
                    vertical-align: middle;
                }
                .nav {
                    padding: 12px 7px 7px 7px;
                    color: #FFFFFF;
                    background-color: #6EA945;
                    cursor: pointer;
                    margin: 5px 5px 5px 5px;
                    display: inline-block;
                    font-weight: 600;
                    text-align: center;
                    font-size: larger;
                }
                .nav:hover {
                    background-color: #FCCC06;
                    /*margin-bottom: 5px;*/
                }
                .nav a {
                    text-decoration: none!important;
                    font-weight: 600;
                }
                a.topmenu {
                    text-decoration: none!important;
                }
                #logos {
                    display: table;
                    width: 100%%;
                }
                .logo {
                    margin: 10px 1.8%% 10px 0;
                }
                #top-left {
                    width: calc(100%% - 245px);
                    min-width: 400px;
                    float: left;
                }
                #top-right {
                    width: 240px;
                    padding-left: 5px;
                    float: left;
                }
                .timeline-Header-title {
                    color: #6EA945!important;
                }
                .timeline-Header-byline {
                    color: #6EA945!important;
                }
                .paper-link {
                    font-size: larger;
                    color: #FFFFFF;
                    background-color: #007B7F;
                    padding: 12px 25px 7px 25px;
                    display: inline-block;
                }''' % _fonts

_header = u'''<!DOCTYPE html>
    <html lang="en">
        <head>
            <meta http-equiv="X-UA-Compatible" content="IE=edge">
            <meta http-equiv="Content-Type" content="text/html; charset=utf-8" />
            <meta name="dc.language" content="en">
            <meta name="viewport" content="width=device-width, 
            initial-scale=1.0">
            <link rel="icon" type="image/png" href="%s" />
            <style type="text/css">%s
            </style>
            <title>%s</title>
        </head>
        <body>
        '''

_footer = u'''
        <br>
        <p> <a href="http://denes.omnipathdb.org/">Dénes Türei,</a>
            <a href="http://saezlab.org/person/nicolas-palacio/">
                Nicolàs Palacio,</a>
            <a href="http://saezlab.org/person/olga-ivanova/">
                Olga Ivanova,</a>
            <a href="http://saezlab.org/">Saez Lab</a> 2016-2020. 
            Feedback: omnipathdb@gmail.com </p>
        <p>
            <a href="https://validator.w3.org/check/referer">
                <img 
                    src="%s"
                     alt="Valid HTML5" />
            </a>
            <a href="https://jigsaw.w3.org/css-validator/check/referer">
                <img 
                    src="%s"
                     alt="Valid CSS3" />
            </a>
        </p>
        \t</body>\n
    </html>\n''' % (_valid_html5, _valid_css3)


def get_header(title=''):
    return _header % (_favicon, _default_css, 'OmniPath :: %s' % title)


def default_template(content,
                     page_title,
                     title='',
                     logos_bottom=False,
                     above_title='',
                     format='b'):

    logos = u'\t<div id="logos">\n\t\t'\
            '<img class="align-middle logo" src="%s" alt="EMBL-EBI, Hinxton"/>\n\t\t'\
            '<img class="align-middle logo" src="%s" '\
                'alt="JRC-COMBINE, RWTH Aachen"/>\n\t\t'\
            '<img class="align-middle logo" src="%s" '\
                'alt="Earlham Institure, Norwich"/>\n\t\t'\
            '<img class="align-middle logo" src="%s" '\
                'alt="University Hospital Heidelberg"/>\n\t'\
            '</div>\n' % (
                _ebi_logo,
                _rwth_aachen_logo,
                _ei_logo,
                _uniklinik_hd_logo,
            )

    page = u'%s\n%s\t<h1>%s</h1>\n\t%s<br>\n%s\n%s\n' % \
        (
            get_header(title),
            above_title if logos_bottom else logos,
            page_title,
            content,
            logos if logos_bottom else '',
            _footer
        )

    ready_page = bs4.BeautifulSoup(page, 'lxml').prettify()

    if format == 'b':
        ready_page = ready_page.encode('utf-8')

    return ready_page

    ##TODO @denes to check, the generating of html didn't work with encode function.
    #return bs4.BeautifulSoup(page, 'lxml', from_encoding='utf-8')\
     #   .prettify().encode('utf-8')


def main_page():
    notebooks = []
    notebook_dir = ['pypath-docs', 'notebooks']

    notebook_names = [
        ('pypath_guide.html', 'Comprehensive guide'),
        (
            'omnipathr_drug_targets_tutorial.html',
            'Drug target lookup with OmnipathR',
        ),
        #('pypath_build_networks.html', 'Quick start'),
        #('intro.html', 'Introduction (directions, enzyme-substrate, subnetworks)'),
        #('pypath_mapping.html', 'ID conversion'),
        #('pathway_extraction.html', 'Pathway annotations'),
        #('node_neighbourhood.html', 'Extracting signaling networks'),
        #('pypath_dorothea_demo.html', 'DoRothEA in pypath'),
        #('TF_location.html', 'Finding transcription factors')
    ]

    if os.path.exists(os.path.join(*notebook_dir)):
        notebooks = os.listdir(os.path.join(*notebook_dir))

    above_title = (
        '<p>'
        '<img'
        ' src="%s"'
        ' title="Welcome to OmniPath!">'
        '</p>' % _omnipath_logo
    )

    with open(os.path.join(common.ROOT, 'data', 'main.html'), 'r') as f:
        doc = f.read()
    
    tut = ''
    
    tut += '<h3>Check out our tutorials:</h3>'
    for nb, name in notebook_names:
        if nb in notebooks:
            tut += '<p><a href="%s" title="%s">%s</a></p>' % (
                'http://pypath.omnipathdb.org/notebooks/%s' % nb, name, name)
    tut += '<p>Special thanks for Luis Tobalina for providing some of the tutorials.</p>'
    
    doc = doc.replace('<!--TUTORIALS-->', tut)

    return default_template(
        doc,
        'OmniPath: literature curated<br>mammalian signaling pathways',
        'literature curated mammalian signaling pathways',
        logos_bottom=True,
        above_title=above_title)


def http_500():

    with open(os.path.join(common.ROOT, 'data', 'http500.html'), 'r') as f:

        html = f.read()

    return html
